#pragma once

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/auth/common/auth.h>
#include <drive/backend/auth/external/database.h>
#include <drive/backend/background/manager/manager.h>
#include <drive/backend/localization/config.h>
#include <drive/backend/notifications/manager.h>
#include <drive/backend/processor/processor.h>
#include <drive/backend/rt_background/manager/manager.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/settings/config.h>

#include <drive/library/cpp/datasync/config.h>
#include <drive/library/cpp/openssl/abstract.h>
#include <drive/library/cpp/searchserver/http_status_config.h>
#include <drive/telematics/server/pusher/interface.h>
#include <drive/telematics/server/pusher/meta.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>
#include <kernel/daemon/module/module.h>

#include <library/cpp/tvmauth/checked_user_ticket.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/api/search_client/client.h>
#include <rtline/library/executor/executor.h>
#include <rtline/library/storage/structured.h>

#include <util/stream/output.h>

class TRequestsHandlerConfig {
private:
    ui32 ThreadsCount = 8;

public:
    ui32 GetThreadsCount() const {
        return ThreadsCount;
    }

    void Init(const TYandexConfig::Section* section) {
        ThreadsCount = section->GetDirectives().Value("ThreadsCount", ThreadsCount);
    }

    void ToString(IOutputStream& os) const {
        os << "ThreadsCount: " << ThreadsCount << Endl;
    }
};

class TTvmConfig {
public:
    const TString& GetCache() const {
        return Cache;
    }
    const TString& GetSecret() const {
        return Secret;
    }
    const TSet<ui32>& GetDestinationClientIds() const {
        return DestinationClientIds;
    }
    const TSet<NTvmAuth::EBlackboxEnv>& GetUserTicketEnvironments() const {
        return UserTicketEnvironments;
    }
    ui32 GetSelfClientId() const {
        return SelfClientId;
    }

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

private:
    TString Cache;
    TString Secret;
    TString SecretFile;
    TSet<ui32> DestinationClientIds;
    TSet<NTvmAuth::EBlackboxEnv> UserTicketEnvironments = {
        NTvmAuth::EBlackboxEnv::Prod,
    };
    ui32 SelfClientId = 0;
};

class TMapsLinkerConfig {
public:
    const TString& GetEndpoint() const {
        return Endpoint;
    }
    const TString& GetRelinkerName() const {
        return RelinkerName;
    }

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

private:
    TString Endpoint;
    TString RelinkerName;
};

class TMapsRouterConfig {
public:
    ui32 GetSelfClientId() const;
    ui32 GetDestinationTvmId() const;
    const TString& GetHostUrl() const;
    const TDuration& GetRequestTimeout() const;
    const TString& GetVehicleType() const;

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

private:
    ui32 SelfClientId = 0;
    ui32 DestinationTvmId = 0;
    TString HostUrl = "http://core-driving-router.maps.yandex.net";
    TDuration RequestTimeout = TDuration::Seconds(1);
    TString VehicleType;
};

class TServerBaseConfig: public IServerConfig {
public:
    class TSensorApiConfig {
    public:
        enum class EPusherType {
            IndexingClient = 1,
            CustomPusher = 2,
        };

        void Init(const TYandexConfig::Section* section) {
            CHECK_WITH_LOG(section);
            const auto& d = section->GetDirectives();
            SensorApiName = d.Value("Name", SensorApiName);
            HistoryApiName = d.Value("HistoryApiName", HistoryApiName);
            HistoryApiTimeout = d.Value("HistoryApiTimeout", HistoryApiTimeout);
            PusherType = d.Value("PusherType", PusherType);

            TYandexConfig::TSectionsMap sectionsMap = section->GetAllChildren();
            auto range = sectionsMap.equal_range("Pusher");
            auto sectionsCount = std::distance(range.first, range.second);
            if (sectionsCount == 1) {
                PusherOptions = NDrive::IPusherOptions::ConstructPusherOptions(*range.first->second);
            } else if (sectionsCount > 1) {
                auto metaPusherOptions = MakeHolder<NDrive::TMetaPusherOptions>();
                for (auto it = range.first; it != range.second; ++it) {
                    metaPusherOptions->MutablePushersOptions().push_back(NDrive::IPusherOptions::ConstructPusherOptions(*it->second));
                }
                PusherOptions = std::move(metaPusherOptions);
            }
        }

        void ToString(IOutputStream& os) const {
            os << "Name: " << SensorApiName  << Endl;
            os << "HistoryApiName: " << HistoryApiName << Endl;
            os << "HistoryApiTimeout: " << HistoryApiTimeout << Endl;
            os << "PusherType: " << PusherType << Endl;
            if (PusherOptions) {
                PusherOptions->Print(os);
            }
        }

    private:
        R_READONLY(TString, SensorApiName);
        R_READONLY(TString, HistoryApiName);
        R_READONLY(TDuration, HistoryApiTimeout, TDuration::Seconds(2));
        R_READONLY(THolder<NDrive::IPusherOptions>, PusherOptions);
        R_READONLY(EPusherType, PusherType, EPusherType::IndexingClient);
    };

public:
    R_FIELD(TDuration, ProcessorDefaultTimeout, TDuration::Seconds(30));

public:
    const TDaemonConfig DaemonConfig;

protected:
    TAnyYandexConfig RawConfig;

    TMap<TString, TRequestsHandlerConfig> RequestHandlers;
    TDaemonConfig::THttpOptions HttpServerOptions;
    THttpStatusManagerConfig HttpStatusManagerConfig;
    THolder<TTaskExecutorConfig> TaskExecutorConfig;
    TMap<TString, NDrive::INotifierConfig::TPtr> Notifications;
    THolder<TNotifiersManagerConfig> NotifiersManagerConfig;
    TMap<TString, TRTLineAPIConfig> RTLineAPIConfigs;
    TMap<TVersionedKey, IRequestProcessorConfig::TPtr> ProcessorsInfo;
    TMap<TString, IAuthModuleConfig::TPtr> AuthModules;
    TMap<TString, NRTLine::TStorageOptions> DatabaseConfigs;
    TMap<TString, NSimpleMeta::TConfig> RequestPolicy;
    TMap<ui32, TTvmConfig> TvmConfigs;
    THolder<TBackgroundProcessesManagerConfig> BackgroundProcessesConfig;
    THolder<TRTBackgroundManagerConfig> RTBackgroundManagerConfig;
    THolder<TDatasyncClientConfig> DatasyncConfig;
    THolder<TLocalizationConfig> Localization;
    THolder<TSettingsConfig> Settings;
    TMap<TString, NStorage::TDatabaseConfig> Databases;
    TPluginConfigs<IDaemonModuleConfig> ModulesConfig;
    TSet<TString> UsingModules;
    TString EventLog;
    TString UnifiedAgentEventLog;
    size_t UnifiedAgentGrpcMaxMessageSize;
    TDuration UnifiedAgentGrpcReconnectDelay;
    TDuration UnifiedAgentGrpcSendDelay;
    size_t UnifiedAgentMaxInflightBytes;
    TString ItsWatchPath;
    THolder<TAsyncDeliveryResources::TConfig> ADGlobalConfig;
    TMap<TString, NOpenssl::TCipherConfig> Ciphers;
    THolder<TExternalAccessTokensManagerConfig> ExternalAccessTokensManagerConfig;
    THolder<TMapsLinkerConfig> MapsLinkerConfig;
    THolder<TMapsRouterConfig> MapsRouterConfig;
    THolder<TMapsRouterConfig> PedestrianRouterConfig;

protected:
    const IAbstractModuleConfig* GetModuleConfigImpl(const TString& name) const override {
        return ModulesConfig.Get<IAbstractModuleConfig>(name);
    }
    void Init(const TYandexConfig::Section* section);
    virtual void DoToString(IOutputStream& /*os*/) const {
    }

public:
    TServerBaseConfig(const TServerConfigConstructorParams& params);

    virtual void CheckConfig() const;

    const TAsyncDeliveryResources::TConfig* GetADGlobalConfig() const {
        if (!ADGlobalConfig || !ADGlobalConfig->GetEnabled()) {
            return nullptr;
        }
        return ADGlobalConfig.Get();
    }

    TVector<TString> GetServerProcessors() const {
        TVector<TString> result;
        for (auto&& i : ProcessorsInfo) {
            result.emplace_back(i.first.GetName());
        }
        return result;
    }

    const TMap<TVersionedKey, IRequestProcessorConfig::TPtr>& GetServerProcessorsObjects() const {
        return ProcessorsInfo;
    }

    const TMap<TString, NDrive::INotifierConfig::TPtr>& GetNotifications() const {
        return Notifications;
    }

    const TTaskExecutorConfig* GetTaskExecutorConfig() const {
        return TaskExecutorConfig.Get();
    }

    const TDatasyncClientConfig* GetDatasyncConfig() const {
        return DatasyncConfig.Get();
    }

    const TLocalizationConfig* GetLocalizationConfig() const {
        return Localization.Get();
    }

    const TSettingsConfig* GetSettingsConfig() const {
        return Settings.Get();
    }

    const TMap<TString, NRTLine::TStorageOptions>& GetDatabases() const {
        return DatabaseConfigs;
    }

    const TMap<ui32, TTvmConfig>& GetTvmConfigs() const {
        return TvmConfigs;
    }

    const TMap<TString, NOpenssl::TCipherConfig>& GetCiphers() const {
        return Ciphers;
    }

    TRTBackgroundManagerConfig* GetRTBackgroundManagerConfig() const {
        return RTBackgroundManagerConfig.Get();
    }

    TNotifiersManagerConfig* GetNotifiersManagerConfig() const {
        return NotifiersManagerConfig.Get();
    }

    TBackgroundProcessesManagerConfig* GetBackgroundProcessesConfig() const {
        return BackgroundProcessesConfig.Get();
    }

    const TMap<TString, NStorage::TDatabaseConfig>& GetExternalDatabases() const;

    const IAuthModuleConfig* GetAuthModuleInfo(const TString& authModuleName) const {
        auto it = AuthModules.find(authModuleName);
        if (it == AuthModules.end()) {
            return nullptr;
        }
        return it->second.Get();
    }

    const TString& GetEventLog() const {
        return EventLog;
    }

    const TString& GetUnifiedAgentEventLog() const {
        return UnifiedAgentEventLog;
    }

    size_t GetUnifiedAgentGrpcMaxMessageSize() const {
        return UnifiedAgentGrpcMaxMessageSize;
    }

    TDuration GetUnifiedAgentGrpcReconnectDelay() const {
        return UnifiedAgentGrpcReconnectDelay;
    }

    TDuration GetUnifiedAgentGrpcSendDelay() const {
        return UnifiedAgentGrpcSendDelay;
    }

    size_t GetUnifiedAgentMaxInflightBytes() const {
        return UnifiedAgentMaxInflightBytes;
    }

    TString GetItsWatchPath() const {
        return ItsWatchPath;
    }

    const TMapsLinkerConfig* GetMapsLinkerConfig() const;
    const TMapsRouterConfig* GetMapsRouterConfig() const;
    const TMapsRouterConfig* GetPedestrianRouterConfig() const;

    IRequestProcessorConfig::TPtr GetProcessorInfo(const TVersionedKey& processorKey) const;

    const TMap<TString, TRTLineAPIConfig>& GetRTLineAPIConfigs() const {
        return RTLineAPIConfigs;
    }

    const THttpStatusManagerConfig& GetHttpStatusManagerConfig() const {
        return HttpStatusManagerConfig;
    }

    const THttpServerOptions& GetHttpServerOptions() const {
        return HttpServerOptions;
    }

    TSet<TString> GetModulesSet() const override {
        return UsingModules;
    }

    const TMap<TString, TRequestsHandlerConfig>& GetHandlers() const {
        return RequestHandlers;
    }

    const TDaemonConfig& GetDaemonConfig() const override {
        return DaemonConfig;
    }

    const TExternalAccessTokensManagerConfig* GetExternalAccessTokensManagerConfig() const {
        return ExternalAccessTokensManagerConfig.Get();
    }

    TString ToString() const;
    void ToString(IOutputStream& os) const;
};
