#include "replier.h"

#include <drive/backend/logging/events.h>
#include <drive/backend/logging/logging.h>

#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/reqid/reqid.h>

#include <rtline/library/json/exception.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/types/exception.h>

namespace {
    TMutex HandlersCounterMutex;
    TMap<TString, i32> HandlersCounter;

    const TStringBuf ReqIdClass = "YD";
}

TReplier::THandlersCounterGuard::THandlersCounterGuard(const TString& key, const TString& metric)
    : Key(key + "-" + metric)
{
    TGuard<TMutex> g(HandlersCounterMutex);
    HandlersCounter[Key] = HandlersCounter[Key] + 1;
    TUnistatSignalsCache::SignalLastX("handlers-" + Key, "count", HandlersCounter[Key]);
}

TReplier::THandlersCounterGuard::~THandlersCounterGuard() {
    TGuard<TMutex> g(HandlersCounterMutex);
    HandlersCounter[Key] = HandlersCounter[Key] - 1;
    TUnistatSignalsCache::SignalLastX("handlers-" + Key, "count", HandlersCounter[Key]);
}

TReplier::TReplier(IReplyContext::TPtr context, const TServerBaseConfig* config, const IServerBase* server)
    : IHttpReplier(context, &config->GetHttpStatusManagerConfig())
    , Config(config)
    , Server(server)
{
    NDrive::FormReqId(Context->MutableCgiParameters(), &Context->GetRequestData(), ReqIdClass);
    Key = Server->GetVersionedKey(*Context);
    NDrive::TEventLog::TContextGuard contextGuard(Context.Get());
    NDrive::TEventLog::Log(NDrive::TEventLog::Access, *Context);
    if (!!Server->GetProcessorInfo(Key)) {
        SignalGuard.Reset(new THandlersCounterGuard(Key.GetName(), "total-requests"));
    }
}

TReplier::~TReplier() {
}

TDuration TReplier::GetDefaultTimeout() const {
    IRequestProcessorConfig::TPtr processorGenerator = Server->GetProcessorInfo(Key);
    if (processorGenerator && processorGenerator->GetRequestTimeout() != IRequestProcessorConfig::NotInitializedTimeout) {
        return processorGenerator->GetRequestTimeout();
    }
    return Config->GetProcessorDefaultTimeout();
}

void TReplier::OnRequestExpired() {
    IRequestProcessorConfig::TPtr processorGenerator = Server->GetProcessorInfo(Key);
    TString processorName = "UNKNOWN";
    if (processorGenerator) {
        processorName = processorGenerator->GetName();
    }
    const auto deadline = Context->GetRequestDeadline();
    const auto start = Context->GetRequestStartTime();
    const auto finish = Now();
    NDrive::TUnistatSignals::OnAccess(processorName);
    NDrive::TUnistatSignals::OnReply(processorName, HTTP_GATEWAY_TIME_OUT, start, finish, deadline);
    NDrive::TEventLog::TContextGuard contextGuard(Context.Get());
    NDrive::TEventLog::Log(NDrive::TEventLog::Timeout, *Context, HTTP_GATEWAY_TIME_OUT, NJson::JSON_NULL);
}

void TReplier::DoSearchAndReply() {
    const auto& httpCodes = Config->GetHttpStatusManagerConfig();
    const auto& settings = Server->GetSettings();
    NDrive::TEventLog::TContextGuard contextGuard(Context.Get());
    NDrive::TEventLog::TUserIdGuard pendingUserIdGuard;
    try {
        R_ENSURE(
            settings.GetHandlerValueDef(Key.GetName(), "base_enabled", true),
            HTTP_SERVICE_UNAVAILABLE,
            "handler " << Key.GetName() << " is disabled",
            NDrive::MakeError("handler_disabled")
        );
        IRequestProcessorConfig::TPtr processorGenerator = Server->GetProcessorInfo(Key);
        R_ENSURE(
            processorGenerator,
            HTTP_BAD_REQUEST,
            "cannot construct processor generator for " << Key.ToString(),
            NDrive::MakeError("handler_not_found")
        );

        const TString whitelistStr = settings.GetHandlerValueDef<TString>(Key.GetName(), "whitelist", "");
        if (whitelistStr) {
            auto whitelist = MakeSet(SplitString(whitelistStr, ","));
            auto clientIp = NUtil::GetClientIp(Context->GetRequestData());
            R_ENSURE(
                whitelist.contains(clientIp),
                HTTP_FORBIDDEN,
                "client ip " << clientIp << " is not found in whitelist",
                NDrive::MakeError("blocked_by_ip_whitelist")
            );
        }
        auto blacklistString = settings.GetHandlerValueDef<TString>(Key.GetName(), "blacklist", "");
        if (blacklistString) {
            auto blacklist = MakeSet(SplitString(blacklistString, ","));
            auto clientIp = NUtil::GetClientIp(Context->GetRequestData());
            R_ENSURE(
                !blacklist.contains(clientIp),
                HTTP_FORBIDDEN,
                "client ip " << clientIp << " is blacklisted",
                NDrive::MakeError("blocked_by_ip_blacklist")
            );
        }

        THandlersCounterGuard signalGuard(Key.GetName(), "in-progress");
        if (!!processorGenerator->GetOverrideCgi() || !!processorGenerator->GetAdditionalCgi()) {
            const TString cgiStr = !!processorGenerator->GetOverrideCgi() ? processorGenerator->GetOverrideCgi() : processorGenerator->GetAdditionalCgi();
            TCgiParameters originalCgi = Context->MutableCgiParameters();
            if (!!processorGenerator->GetOverrideCgi()) {
                Context->MutableCgiParameters().clear();
                auto reqid = originalCgi.find("reqid");
                if (reqid != originalCgi.end()) {
                    Context->MutableCgiParameters().insert(*reqid);
                }
            }
            TCgiParameters cgi;
            cgi.Scan(cgiStr);
            for (auto&& i : cgi) {
                if (i.second.StartsWith("$") && i.second.EndsWith("$") && i.second.size() > 1) {
                    Context->MutableCgiParameters().InsertUnescaped(i.first, originalCgi.Get(i.second.substr(1, i.second.size() - 2)));
                } else {
                    Context->MutableCgiParameters().InsertUnescaped(i.first, i.second);
                }
            }
        }
        if (!!processorGenerator->GetOverrideCgiPart()) {
            const TString cgiStr = processorGenerator->GetOverrideCgiPart();
            TCgiParameters originalCgi = Context->MutableCgiParameters();
            TCgiParameters cgi;
            cgi.Scan(cgiStr);
            for (auto&& i : cgi) {
                if (i.second.StartsWith("$") && i.second.EndsWith("$") && i.second.size() > 1) {
                    Context->MutableCgiParameters().ReplaceUnescaped(i.first, originalCgi.Get(i.second.substr(1, i.second.size() - 2)));
                } else {
                    Context->MutableCgiParameters().ReplaceUnescaped(i.first, i.second);
                }
            }
        }
        if (!!processorGenerator->GetOverridePost()) {
            Context->SetBuf(processorGenerator->GetOverridePost());
        }
        Processor = processorGenerator->ConstructProcessor(Context, Server);
        R_ENSURE(Processor, HTTP_BAD_REQUEST, "cannot construct processor", NDrive::MakeError("handler_not_constructed"));

        Processor->SetVersion(processorGenerator->GetVersion());
        Processor->SetTypeName(Key.GetName());
        Processor->Process();
    } catch (const TCodedException& cException) {
        TJsonReport report(Context, Key.ToString());
        report.Finish(cException);
    } catch (const yexception& e) {
        TJsonReport report(Context, Key.ToString());
        TCodedException ce(httpCodes.UnknownErrorStatus);
        ce << e.AsStrBuf();
        if (auto backtrace = e.BackTrace()) {
            ce.AddInfo("backtrace", NJson::ToJson(*backtrace));
        } else {
            auto bt = TBackTrace::FromCurrentException();
            ce.AddInfo("backtrace", NJson::ToJson(bt));
        }
        report.Finish(ce);
    } catch (...) {
        TJsonReport report(Context, Key.ToString());
        TCodedException ce(httpCodes.UnknownErrorStatus);
        auto bt = TBackTrace::FromCurrentException();
        ce.AddInfo("backtrace", NJson::ToJson(bt));
        ce.AddInfo("exception", CurrentExceptionInfo());
        report.Finish(ce);
    }
}

IThreadPool* TReplier::DoSelectHandler() {
    IRequestProcessorConfig::TPtr processorGenerator = Server->GetProcessorInfo(Key);
    if (!processorGenerator) {
        return nullptr;
    }
    if (processorGenerator->GetHandlerName() == "") {
        return Server->GetRequestsHandler("default");
    }
    IThreadPool* handler = Server->GetRequestsHandler(processorGenerator->GetHandlerName());
    R_ENSURE(handler, HTTP_INTERNAL_SERVER_ERROR, "incorrect handler for request " << Yensured(Context)->GetCgiParameters().Print());
    return handler;
}
