#include "server.h"

#include <drive/backend/localization/localization.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/logging.h>

#include <drive/library/cpp/datasync/client.h>
#include <drive/library/cpp/logger/formatter.h>
#include <drive/library/cpp/tvm/logger.h>

#include <library/cpp/logger/global/common.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <logbroker/unified_agent/client/cpp/logger/backend.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/library/unistat/log/backend.h>

#include <kernel/daemon/base_controller.h>

namespace {
    class TLoggerFormatter: public ILoggerFormatter {
    public:
        void Format(const TLogRecordContext& context, TLogElement& elem) const override {
            elem << NLoggingImpl::GetLocalTimeS() << '\t';
            elem << context.CustomMessage << '\t';
            elem << NDrive::StripFilename(context.SourceLocation.File, 2) << ":" << context.SourceLocation.Line << '\t';
            if (auto reqid = NDrive::TEventLog::GetReqId()) {
                elem << reqid << '\t';
            }
            if (context.Priority > TLOG_RESOURCES && !ExitStarted()) {
                elem << NLoggingImpl::GetSystemResources() << '\t';
            }
        }
    };

    TUnistatSignal<> HttpBusyThreadsCount{
        {
            "http-busy-threads",
        },
        EAggregationType::LastValue,
        "annn"
    };
    TUnistatSignal<> HttpFreeThreadsCount{
        {
            "http-free-threads",
        },
        EAggregationType::LastValue,
        "annn"
    };
}

void NDrive::TController::CheckConfig(IServerConfig& config, bool onStart) const {
    Y_UNUSED(onStart);
    const auto& impl = dynamic_cast<const TServerBaseConfig&>(config);
    impl.CheckConfig();
}

class TServerBase::TRequestHandlersMtpQueue : public TCountedMtpQueue {
public:
    TRequestHandlersMtpQueue(const TString& name)
        : TCountedMtpQueue(IThreadPool::TParams()
            .SetThreadName(name)
        )
        , Name(name)
        , BusyThreadsCount(name + "-busy-threads", EAggregationType::LastValue, "annn")
        , FreeThreadsCount(name + "-free-threads", EAggregationType::LastValue, "annn")
    {
    }

    void Refresh() {
        BusyThreadsCount.Signal(GetBusyThreadsCount());
        FreeThreadsCount.Signal(GetFreeThreadsCount());
    }

private:
    const TString Name;
    const TNamedSignalCustom BusyThreadsCount;
    const TNamedSignalCustom FreeThreadsCount;
};

TServerBase::TServerBase(const TConfig& config)
    : Config(config)
{
    if (const auto& globalLog = Config.GetDaemonConfig().GetLoggerType(); globalLog || !GlobalLogInitialized()) {
        DoInitGlobalLog(MakeHolder<TUnistatLogBackend>(
            Config.GetDaemonConfig().GetLoggerType(),
            static_cast<ELogPriority>(Config.GetDaemonConfig().GetLogLevel())
        ), MakeHolder<TLoggerFormatter>());
    }
    if (const TString& eventLog = Config.GetEventLog()) {
        Config.GetDaemonConfig().StartLoggingUnfiltered<NDrive::TEventLog>(eventLog);
    }

    if (const TString& unifiedAgentEventLog = Config.GetUnifiedAgentEventLog()) {
        auto parameters = NUnifiedAgent::TClientParameters(unifiedAgentEventLog);
        if (TLoggerOperator<NDrive::TEventLog>::Usage()) {
            parameters.SetLog(TLoggerOperator<NDrive::TEventLog>::Log());
        }
        if (size_t grpcMaxMessageSize = Config.GetUnifiedAgentGrpcMaxMessageSize()) {
            parameters.SetGrpcMaxMessageSize(grpcMaxMessageSize);
        }
        if (TDuration grpcReconnectDelay = Config.GetUnifiedAgentGrpcReconnectDelay()) {
            parameters.SetGrpcReconnectDelay(grpcReconnectDelay);
        }
        if (TDuration grpcSendDelay = Config.GetUnifiedAgentGrpcSendDelay()) {
            parameters.SetGrpcSendDelay(grpcSendDelay);
        }
        if (size_t maxInflightBytes = Config.GetUnifiedAgentMaxInflightBytes()) {
            parameters.SetMaxInflightBytes(maxInflightBytes);
        }
        auto backend = NUnifiedAgent::MakeLogBackend(parameters);
        TLoggerOperator<NDrive::TEventLog>::Set(new NDrive::TEventLog(std::move(backend)));
    }
    NDrive::TUnistatSignals::RegisterFallback();
    RegisterGlobalMessageProcessor(this);
}

TServerBase::~TServerBase() {
    UnregisterGlobalMessageProcessor(this);
    TLoggerFormatterOperator::Set(MakeHolder<NDrive::TLoggerFormatter>().Release());
}

bool TServerBase::Process(IMessage* message_) {
    if (auto message = message_->As<TCollectDaemonInfoMessage>()) {
        message->SetCType(Config.DaemonConfig.GetCType());
        message->SetService(Config.DaemonConfig.GetService());
        return true;
    }
    if (auto message = message_->As<TMessageUpdateUnistatSignals>()) {
        if (HttpServer) {
            HttpBusyThreadsCount.Signal(HttpServer->GetBusyThreadsCount());
            HttpFreeThreadsCount.Signal(HttpServer->GetFreeThreadsCount());
        }
        for (auto&&[name, handler] : RequestHandlers) {
            handler->Refresh();
        }
        return true;
    }
    if (auto message = message_->As<TSystemStatusMessage>()) {
        Cerr << "SystemStatus: " << message->GetMessage() << Endl;
        return true;
    }
    return false;
}

TAtomicSharedPtr<NDrive::INotifier> TServerBase::GetNotifier(const TString& name) const {
    auto it = Notifications.find(name);
    if (it != Notifications.end()) {
        return it->second;
    }
    if (NotifiersManager) {
        TMaybe<TNotifierContainer> container = NotifiersManager->GetObject(name);
        return container ? container->GetNotifierPtr() : nullptr;
    }
    return nullptr;
}

const TRTLineAPI* TServerBase::GetRTLineAPI(const TString& serviceName) const {
    auto p = RTLineAPIs.find(serviceName);
    if (p != RTLineAPIs.end()) {
        return &p->second;
    }
    return nullptr;
}

TVector<TString> TServerBase::ListRTLineAPIs() const {
    return MakeVector(NContainer::Keys(RTLineAPIs));
}

IRequestProcessorConfig::TPtr TServerBase::GetProcessorInfo(const TVersionedKey& procKey) const {
    IRequestProcessorConfig::TPtr result;
    const TVector<TString> pathes = {GetCType() + ".handlers", "handlers"};
    const TString handlerConfig = GetSettings().GetValueDef<TString>(pathes, procKey.GetName(), "");
    if (!!handlerConfig) {
        TAnyYandexConfig config;
        if (!config.ParseMemory(handlerConfig)) {
            ALERT_LOG << "Fail to parse handler conf: " << handlerConfig << Endl;
            return Config.GetProcessorInfo(procKey);
        }
        const TYandexConfig::Section* section = config.GetRootSection();
        if (!section) {
            ALERT_LOG << "Fail to parse handler conf sections: " << handlerConfig << Endl;
            return Config.GetProcessorInfo(procKey);
        }
        const TString typeName = section->GetDirectives().Value<TString>("ProcessorType", procKey.GetName());
        IRequestProcessorConfig::TPtr result = IRequestProcessorConfig::TFactory::Construct(typeName, procKey.GetName());
        if (!!result) {
            result->Init(section);
            return result;
        }
    }
    return Config.GetProcessorInfo(procKey);
}

IThreadPool* TServerBase::GetRequestsHandler(const TString& handlerName) const {
    auto it = RequestHandlers.find(!!handlerName ? handlerName : "default");
    if (it == RequestHandlers.end()) {
        return nullptr;
    }
    return it->second.Get();
}

TAtomicSharedPtr<NTvmAuth::TTvmClient> TServerBase::GetTvmClient(ui32 selfClientId, NTvmAuth::EBlackboxEnv env) const {
    auto p = TvmClients.find(std::make_pair(selfClientId, env));
    if (p != TvmClients.end()) {
        return p->second;
    } else {
        return nullptr;
    }
}

void TServerBase::Run() {
    try {
        INFO_LOG << "Actual config: " << Endl << Config.ToString() << Endl;
        NSimpleMeta::TConfig adc = NSimpleMeta::TConfig::ForRequester();

        for (auto&& i : Config.GetCiphers()) {
            NOpenssl::IAbstractCipher::TPtr cipher = THolder(NOpenssl::IAbstractCipher::TFactory::Construct(i.second.GetType()));
            AssertCorrectConfig(!!cipher, "could not construct cipher with type: " + ::ToString(i.second.GetType()));
            cipher->Init(i.second);
            Ciphers.emplace(i.first, std::move(cipher));
        }

        for (auto&& i : Config.GetDatabases()) {
            Storages[i.first] = i.second.ConstructStorage();
        }

        for (auto&& i : Config.GetExternalDatabases()) {
            Databases[i.first] = i.second.ConstructDatabase();
        }

        for (auto&&[selfClientId, config] : Config.GetTvmConfigs()) {
            NTvmAuth::NTvmApi::TClientSettings settings;
            settings.SetSelfTvmId(config.GetSelfClientId());
            settings.EnableServiceTicketChecking();
            settings.EnableUserTicketChecking(NTvmAuth::EBlackboxEnv::Prod);
            if (const auto& cache = config.GetCache()) {
                settings.SetDiskCacheDir(cache);
            }
            if (const auto& secret = config.GetSecret()) {
                NTvmAuth::NTvmApi::TClientSettings::TDstVector destinations = {
                    config.GetDestinationClientIds().begin(),
                    config.GetDestinationClientIds().end()
                };
                settings.EnableServiceTicketsFetchOptions(secret, std::move(destinations));
            }
            for (auto&& env : config.GetUserTicketEnvironments()) {
                settings.EnableUserTicketChecking(env);
                auto client = MakeAtomicShared<NTvmAuth::TTvmClient>(settings, CreateGlobalTvmLogger());
                TvmClients[std::make_pair(settings.SelfTvmId, env)] = client;
            }
        }

        for (auto&&[serviceName, config] : Config.GetRTLineAPIConfigs()) {
            TMaybe<NDrive::TTvmAuth> tvmAuth = Nothing();
            if (ui32 selfClientId = config.GetSelfClientId()) {
                auto tvm = GetTvmClient(selfClientId);
                AssertCorrectConfig(!!tvm, "Tvm Client does not exists");
                tvmAuth = MakeMaybe<NDrive::TTvmAuth>(tvm, config.GetDestinationTvmId());
            }
            auto api = TRTLineAPI(config, std::move(tvmAuth));
            RTLineAPIs.emplace(serviceName, std::move(api));
        }

        if (auto datasyncConfig = Config.GetDatasyncConfig()) {
            ui32 selfClientId = datasyncConfig->GetSelfClientId();
            auto tvm = selfClientId ? GetTvmClient(selfClientId) : nullptr;
            DatasyncClient = MakeAtomicShared<TDatasyncClient>(*datasyncConfig, tvm);
        }

        AssertCorrectConfig(Config.GetSettingsConfig(), "Incorrect settings configuration");
        if (Config.GetSettingsConfig()->GetDBName() != "fake") {
            auto it = Databases.find(Config.GetSettingsConfig()->GetDBName());
            AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for settings");
            SettingsHistoryContext = MakeHolder<THistoryContext>(it->second);
            Settings = MakeHolder<TSettingsDB>(*SettingsHistoryContext, *Config.GetSettingsConfig());
        } else {
            Settings = MakeHolder<TFakeSettings>();
        }

        if (Config.GetLocalizationConfig()) {
            auto it = Databases.find(Config.GetLocalizationConfig()->GetDBName());
            AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for localization");
            LocalizationHistoryContext = MakeHolder<THistoryContext>(it->second);
            if (!Config.GetLocalizationConfig()->GetDefaultLocalePreffered()) {
                Localization = MakeHolder<NLocalization::TLocalizationDB>(*LocalizationHistoryContext, *Config.GetLocalizationConfig());
            } else {
                Localization = MakeHolder<NLocalization::TLocalizationDBDefaultLocalePreffered>(*LocalizationHistoryContext, *Config.GetLocalizationConfig());
            }
        } else {
            Localization = MakeHolder<TFakeLocalization>();
        }

        for (auto&& i : Config.GetHandlers()) {
            const TString& name = i.first;
            auto queue = MakeHolder<TRequestHandlersMtpQueue>(name);
            queue->Start(i.second.GetThreadsCount());
            RequestHandlers.emplace(name, std::move(queue));
        }

        if (Config.GetNotifiersManagerConfig()) {
            auto it = Databases.find(Config.GetNotifiersManagerConfig()->GetDBName());
            AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for notifiers manager");
            NotifiersHistoryContext.Reset(new THistoryContext(it->second));
            NotifiersManager.Reset(new TNotifiersManager(*this, *NotifiersHistoryContext, *Config.GetNotifiersManagerConfig()));
            Y_ENSURE_BT(NotifiersManager->Start());
        }

        if (Config.GetExternalAccessTokensManagerConfig()) {
            auto it = Databases.find(Config.GetExternalAccessTokensManagerConfig()->GetDBName());
            AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for external access tokens manager");
            ExternalAccessTokensHistoryContext.Reset(new THistoryContext(it->second));
            ExternalAccessTokensManager.Reset(new TExternalAccessTokensManager(*ExternalAccessTokensHistoryContext));
        }

        for (auto&& i : Config.GetNotifications()) {
            auto notification = i.second->Construct();
            if (const auto& clientId = notification->GetSelfTvmId()) {
                notification->SetTvmClient(GetTvmClient(*clientId));
            }
            notification->Start(this);
            Notifications.emplace(i.first, notification);
        }

        if (Config.GetTaskExecutorConfig()) {
            GlobalTaskExecutor.Reset(new TTaskExecutor(*Config.GetTaskExecutorConfig(), this));
            GlobalTaskExecutor->Start();
        }

        DoRun();

        if (!!Config.GetBackgroundProcessesConfig()) {
            BackgroundProcessesManager.Reset(new TBackgroundProcessesManager(*Config.GetBackgroundProcessesConfig(), this->GetAsPtrSafe<IServerBase>()));
            BackgroundProcessesManager->Start();
        }

        if (!!Config.GetRTBackgroundManagerConfig()) {
            RTBackgroundManager.Reset(new TRTBackgroundManager(*this->GetAsPtrSafe<IServerBase>(), *Config.GetRTBackgroundManagerConfig()));
            Y_ENSURE_BT(RTBackgroundManager->Start());
        }

        for (auto&& i : Config.GetServerProcessorsObjects()) {
            i.second->CheckServerForProcessor(this);
        }

        SendGlobalMessage<TServerStartedMessage>();

        HttpServer = MakeHolder<TBaseHttpServer>(Config, this);
        HttpServer->Start();

        if (!ItsWatcher && Config.GetItsWatchPath()) {
            try {
                ItsWatcher = NDrive::MakeItsWatcher(Config.GetItsWatchPath());
                ItsWatcher->Run();
            } catch (const std::exception& e) {
                ALERT_LOG << "cannot start ItsWatcher: " << FormatExc(e) << Endl;
            }
        }
    } catch (...) {
        auto message = CurrentExceptionInfo(/*forceBacktrace=*/true).GetStringRobust();
        AbortFromCorruptedIndex("cannot start: %s", message.c_str());
    }
}

void TServerBase::Stop(ui32 rigidStopLevel, const TCgiParameters* cgiParams /*= nullptr*/) {
    if (ItsWatcher && Config.GetItsWatchPath()) {
        ItsWatcher.Reset();
    }

    if (!!BackgroundProcessesManager) {
        BackgroundProcessesManager->Stop();
    }

    if (RTBackgroundManager && !RTBackgroundManager->Stop()) {
        ERROR_LOG << "cannot stop RTBackgroundManager" << Endl;
    }

    DoStop(rigidStopLevel, cgiParams);

    if (NotifiersManager && !NotifiersManager->Stop()) {
        ERROR_LOG << "cannot stop NotifiersManager" << Endl;
    }

    HttpServer->Shutdown();
    HttpServer->Stop();

    for (auto&& i : RequestHandlers) {
        i.second->Stop();
    }
    if (Config.GetTaskExecutorConfig()) {
        GlobalTaskExecutor->Stop();
        GlobalTaskExecutor.Destroy();
    }

    for (auto&& i : Notifications) {
        i.second->Stop();
    }

    DatasyncClient.Drop();
    RequestHandlers.clear();
    Storages.clear();
    Notifications.clear();
    TvmClients.clear();
}
