#pragma once

#include "config.h"
#include "http_server.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/settings/settings.h>
#include <drive/backend/background/manager/manager.h>
#include <drive/backend/notifications/manager.h>

#include <drive/library/cpp/its_watcher/its_watcher.h>

#include <kernel/daemon/base_controller.h>
#include <kernel/daemon/messages.h>
#include <kernel/daemon/server.h>

namespace NDrive {
    class TController: public NController::TController {
    private:
        using TBase = NController::TController;

    public:
        using TBase::TBase;

        void CheckConfig(IServerConfig& config, bool onStart) const override;
    };
}

class TServerBase
    : public NController::IServer
    , public virtual IServerBase
    , public IMessageProcessor
{
private:
    class TRequestHandlersMtpQueue;

private:
    const TServerBaseConfig& Config;

    THolder<TBaseHttpServer> HttpServer;
    TMap<TString, TAtomicSharedPtr<TRequestHandlersMtpQueue>> RequestHandlers;
    TMap<TString, NRTLine::IVersionedStorage::TPtr> Storages;

    THolder<IHistoryContext> NotifiersHistoryContext;
    THolder<TNotifiersManager> NotifiersManager;

    TMap<TString, NOpenssl::IAbstractCipher::TPtr> Ciphers;
    THolder<IHistoryContext> SecretsHistoryContext;

    TMap<TString, NDrive::INotifier::TPtr> Notifications;
    TMap<std::pair<ui32, NTvmAuth::EBlackboxEnv>, TAtomicSharedPtr<NTvmAuth::TTvmClient>> TvmClients;
    TMap<TString, TRTLineAPI> RTLineAPIs;
    TAtomicSharedPtr<TDatasyncClient> DatasyncClient;
    THolder<THistoryContext> LocalizationHistoryContext;
    THolder<ILocalization> Localization;

    THolder<THistoryContext> SettingsHistoryContext;
    THolder<ISettings> Settings;

    THolder<TTaskExecutor> GlobalTaskExecutor;
    THolder<TBackgroundProcessesManager> BackgroundProcessesManager;
    THolder<TRTBackgroundManager> RTBackgroundManager;
    THolder<NDrive::IItsWatcher> ItsWatcher;
    THolder<IHistoryContext> ExternalAccessTokensHistoryContext;
    THolder<IExternalAccessTokensManager> ExternalAccessTokensManager;

protected:
    TMap<TString, TDatabasePtr> Databases;

public:
    using TController = NDrive::TController;
    using TConfig = TServerBaseConfig;
    using TInfoCollector = TCollectServerInfo;

protected:
    virtual void DoRun() {
    };

    virtual void DoStop(ui32 /*rigidStopLevel*/, const TCgiParameters* /*cgiParams*/ = nullptr) {
    }

public:
    TServerBase(const TConfig& config);
    ~TServerBase();

    virtual const TString& GetServiceName() const override {
        return Config.GetService();
    }

    virtual const TString& GetCType() const override {
        return Config.GetCType();
    }

    virtual TString Name() const override {
        return "BaseServer";
    }
    virtual bool Process(IMessage* message) override;

    virtual const ILocalization* GetLocalization() const override {
        return Localization.Get();
    }

    virtual const ISettings& GetSettings() const override {
        return *Settings;
    }

    virtual const TSettingsDB* GetSettingsManager() const override {
        return &dynamic_cast<TSettingsDB&>(*Settings);
    }

    virtual bool HasSettings() const override {
        return !!Settings;
    }

    virtual const NDrive::INotifiersManager* GetNotifiersManager() const override {
        return NotifiersManager.Get();
    }

    virtual const TS3Client* GetMDSClient() const override {
        return nullptr;
    }

    virtual const TStartrekClient* GetStartrekClient() const override {
        return nullptr;
    }

    virtual const NDrive::TSupportAIClient* GetSupportAIClient() const override {
        return nullptr;
    }

    virtual const TClckClient* GetClckClient() const override {
        return nullptr;
    }

    virtual TSet<TString> GetNotifierNames() const override {
        TSet<TString> result = MakeSet(NContainer::Keys(Notifications));
        if (NotifiersManager) {
            const TSet<TString> dbNotifiers = NotifiersManager->GetObjectNames();
            result.insert(dbNotifiers.begin(), dbNotifiers.end());
        }
        return result;
    }

    virtual TAtomicSharedPtr<NDrive::INotifier> GetNotifier(const TString& name) const override;

    const TRTLineAPI* GetRTLineAPI(const TString& serviceName) const override;
    virtual TVector<TString> ListRTLineAPIs() const override;

    virtual const TRTBackgroundManager* GetRTBackgroundManager() const override {
        return RTBackgroundManager.Get();
    }

    virtual TMaybe<TString> GetOrCreateUserId(const TString& /*username*/, const TString& /*operatorUserId*/) const override {
        return {};
    }

    virtual TMaybe<TString> GetRtBackgroundRobotUserId(const TString& /*name*/) const override {
        return {};
    }

    virtual const TBackgroundProcessesManager* GetBackgroundProcessesManager() const override {
        return BackgroundProcessesManager.Get();
    }

    virtual const NOpenssl::IAbstractCipher* GetCipher(const TString& cipherName) const override {
        auto cipherIt = Ciphers.find(cipherName);
        return cipherIt != Ciphers.end() ? cipherIt->second.Get() : nullptr;
    }

    virtual NStorage::IDatabase::TPtr GetDatabase(const TString& name) const override {
        auto it = Databases.find(name);
        if (it == Databases.end()) {
            return nullptr;
        } else {
            return it->second;
        }
    }

    virtual TVector<TString> GetDatabaseNames() const override {
        TVector<TString> result;
        result.reserve(Databases.size());
        for (auto&& it : Databases) {
            result.emplace_back(it.first);
        }
        return result;
    }

    virtual TAtomicSharedPtr<TDatasyncClient> GetDatasyncClient() const override {
        return DatasyncClient;
    }

    const THttpStatusManagerConfig& GetHttpStatusManagerConfig() const override {
        return Config.GetHttpStatusManagerConfig();
    }

    const TConfig& GetConfig() const {
        return Config;
    }

    virtual const TTaskExecutor& GetGlobalTaskExecutor() const override {
        Y_ENSURE_BT(GlobalTaskExecutor);
        return *GlobalTaskExecutor.Get();
    }

    virtual IRequestProcessorConfig::TPtr GetProcessorInfo(const TVersionedKey& procKey) const override;

    virtual const IAuthModuleConfig* GetAuthModuleInfo(const TString& authModuleName) const override {
        return GetConfig().GetAuthModuleInfo(authModuleName);
    }

    virtual NRTLine::IVersionedStorage::TPtr GetVersionedStorage(const TString& storageName) const override {
        auto it = Storages.find(storageName);
        if (it != Storages.end()) {
            return it->second;
        } else {
            return nullptr;
        }
    }

    TVersionedKey GetVersionedKey(IReplyContext& context) const override {
        TStringBuf type = context.GetUri();
        type.SkipPrefix("/");
        return TVersionedKey(TString{type});
    }

    virtual TVector<TString> GetServerProcessors() const override {
        return Config.GetServerProcessors();
    }

    virtual const IExternalAccessTokensManager* GetExternalAccessTokensManager() const override {
        return ExternalAccessTokensManager.Get();
    }

    virtual IThreadPool* GetRequestsHandler(const TString& handlerName) const override;
    virtual TAtomicSharedPtr<NTvmAuth::TTvmClient> GetTvmClient(ui32 selfClientId, NTvmAuth::EBlackboxEnv env = NTvmAuth::EBlackboxEnv::Prod) const override;

    virtual void Run() override;
    virtual void Stop(ui32 rigidStopLevel, const TCgiParameters* cgiParams = nullptr) override;
};
