#include "bonus.h"

namespace NDrive::NBilling {

    bool TBonusAccountRecordImpl::ParseMeta(const NJson::TJsonValue& meta) {
        if (!meta.Has("balance") || !meta["balance"].IsUInteger()) {
            return false;
        }
        Balance = meta["balance"].GetUInteger();
        return NJson::ParseField(meta, "limited_balance", LimitedBalances, false);
    }

    NDrive::TScheme TBonusAccountRecordImpl::DoGetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme result;
        result.Add<TFSNumeric>("balance", "Баланс").SetVisual(TFSNumeric::EVisualType::Money).SetReadOnly(true);
        result.Add<TFSArray>("limited_balance", "Ограниченный баланс").SetElement(TLimitedBalance::GetScheme(server)).SetReadOnly(true);
        return result;
    }

    void TBonusAccountRecordImpl::SaveMeta(NJson::TJsonValue& meta) const {
        meta["balance"] = Balance;
        if (LimitedBalances.size()) {
            meta["limited_balance"] = NJson::ToJson(LimitedBalances);
        }
    }

    template <class TRecord, EWalletDataType WalletDataType>
    i64 TBonusAccountImpl<TRecord, WalletDataType>::GetBalance() const {
        auto record = GetRecordAs<TRecord>();
        CHECK_WITH_LOG(record);
        ui64 balance = record->GetBalance();
        if (IsLimitedPolicy()) {
            for (const auto& limit : record->GetLimitedBalances()) {
                balance += limit.GetBalance();
            }
        }
        return balance;
    }

    template <class TRecord, EWalletDataType WalletDataType>
    TLimitedBalances TBonusAccountImpl<TRecord, WalletDataType>::GetLimitedBalanceBySum(ui32 sum) const {
        if (!IsLimitedPolicy()) {
            return {};
        }
        auto record = GetRecordAs<TRecord>();
        CHECK_WITH_LOG(record);
        NDrive::NBilling::TLimitedBalances limitsBySum;
        ui64 limitedSum = sum;
        const auto& limits = record->GetLimitedBalances();
        for (const auto& limit : limits) {
            if (limit.GetBalance() <= limitedSum) {
                limitsBySum.insert(limit);
                limitedSum -= limit.GetBalance();
            } else {
                TLimitedBalance newLimit = limit;
                newLimit.SetBalance(limitedSum);
                limitsBySum.insert(newLimit);
                break;
            }
        }
        return limitsBySum;
    }

    template <class TRecord, EWalletDataType WalletDataType>
    EDriveOpResult TBonusAccountImpl<TRecord, WalletDataType>::DoAdd(ui32 sum, NDrive::TEntitySession& session, TInstant deadline, const TString& source) const {
        TAccountRecord::TPtr record = ReadAccountData(WalletDataType, session);
        if (!record) {
            return EDriveOpResult::TransactionError;
        }
        auto bonus = VerifyDynamicCast<TRecord*>(record.Get());

        if (IsLimitedPolicy() && (deadline != TInstant::Max() || !!source)) {
            TLimitedBalance limit(sum, deadline, source);
            auto& limits = bonus->MutableLimitedBalances();
            auto it = limits.find(limit);
            if (it != limits.end()) {
                limit.MutableBalance() += it->GetBalance();
                limits.erase(it);
            }
            limits.insert(limit);
        } else {
            bonus->SetBalance(bonus->GetBalance() + sum);
        }
        return UpsertAccountData(*record, session, GetUserId()) ? EDriveOpResult::Ok : EDriveOpResult::TransactionError;
    }

    template <class TRecord, EWalletDataType WalletDataType>
    EDriveOpResult TBonusAccountImpl<TRecord, WalletDataType>::DoRemove(ui32 sum, NDrive::TEntitySession& session) const {
        TAccountRecord::TPtr record = ReadAccountData(WalletDataType, session);
        if (!record) {
            return EDriveOpResult::TransactionError;
        }
        auto bonus = VerifyDynamicCast<TRecord*>(record.Get());

        ui64 sumForRemove = sum;
        if (IsLimitedPolicy()) {
            auto& limits = bonus->MutableLimitedBalances();
            for (auto it = limits.begin(); it != limits.end();) {
                if (it->GetBalance() <= sumForRemove) {
                    sumForRemove -= it->GetBalance();
                    it = limits.erase(it);
                } else {
                    TLimitedBalance newLimit(*it);
                    newLimit.SetBalance(it->GetBalance() - sumForRemove);
                    limits.erase(it);
                    limits.insert(newLimit);
                    sumForRemove = 0;
                    break;
                }
            }
        }

        if (bonus->GetBalance() < sumForRemove) {
            return EDriveOpResult::LogicError;
        }

        bonus->SetBalance(bonus->GetBalance() - sumForRemove);
        bonus->SetSpent(bonus->GetSpent() + sum);
        return UpsertAccountData(*record, session, GetUserId()) ? EDriveOpResult::Ok : EDriveOpResult::TransactionError;
    }

    template <class TRecord, EWalletDataType WalletDataType>
    bool TBonusAccountImpl<TRecord, WalletDataType>::DoRefresh(TAccountRecord::TPtr record) const {
        CHECK_WITH_LOG(IsLimitedPolicy());

        if (!record) {
            return false;
        }
        auto bonus = VerifyDynamicCast<TRecord*>(record.Get());

        const TInstant refreshTime = ModelingNow();
        auto& limits = bonus->MutableLimitedBalances();
        for (auto it = limits.begin(); it != limits.end();) {
            if (it->GetDeadline() <= refreshTime) {
                it = limits.erase(it);
            } else {
                break;
            }
        }
        return true;
    }

    template <class TRecord, EWalletDataType WalletDataType>
    TInstant TBonusAccountImpl<TRecord, WalletDataType>::GetNextRefresh() const {
        if (IsLimitedPolicy()) {
            auto limits = GetLimitedBalance();
            if (limits.size()) {
                return limits.begin()->GetDeadline();
            }
        }
        return TInstant::Max();
    }


    TAccountRecord::TFactory::TRegistrator<TBonusAccountRecord> TBonusAccountRecord::Registrator(EWalletDataType::Bonus);
    TAccountRecord::TFactory::TRegistrator<TCoinsAccountRecord> TCoinsAccountRecord::Registrator(EWalletDataType::Coins);

    IBillingAccount::TFactory::TRegistrator<TBonusAccount> TBonusAccount::Registrator(EAccount::Bonus);
    IBillingAccount::TFactory::TRegistrator<TCoinsAccount> TCoinsAccount::Registrator(EAccount::Coins);

    NJson::TJsonValue GetBonusPaymethodUserReport(ELocalization locale, const IServerBase& server, const IBillingAccount::TReportContext& context) {
        NJson::TJsonValue result;
        auto local = server.GetLocalization();
        result.InsertValue("group_type", ::ToString(EPaymentGroupType::Radio));
        result.InsertValue("name", local ? local->GetLocalString(locale, "bonus_account_name", "Бонусные рубли") : "Бонусные рубли");

        result.InsertValue("type", ::ToString(EAccount::Bonus));
        result.InsertValue("icon", server.GetSettings().GetValueDef<TString>("bonus_account_icon", ""));

        i64 balance = (context.ExternalBalance ? *context.ExternalBalance : 0);
        auto amount = std::abs(balance / 100);
        auto currency = local ? local->FormatNumeral(amount, locale, false, " ", "units.full.bonus_rubles_for_methods", "", "", "") : "";
        auto balanceValue = (balance < 0 ? "-" : "") + ::ToString(amount) + (currency ? " " : "") + currency;
        result["value"] = balanceValue;
        result.InsertValue("selectable", false);
        result.InsertValue("selected", true);
        return result;
    }
}
