#include "limited.h"

#include <util/generic/cast.h>

namespace {
    TString BuildActiveKey(const TString& name) {
        return TStringBuilder() << name << "_notification";
    }

    TString BuildLastNotifyKey(const TString& name) {
        return TStringBuilder() << "last_" << BuildActiveKey(name);
    }

    TString BuildLimitKey(const TString& name) {
        return TStringBuilder() << name << "_limit";
    }
}

namespace NDrive::NBilling {

    NJson::TJsonValue TBalanceClientInfo::ToJson() const {
        NJson::TJsonValue json;
        json.InsertValue("client_id", ClientId);
        json.InsertValue("person_id", PersonId);
        json.InsertValue("contract_id", ContractId);
        json.InsertValue("external_contract_id", ExternalContractId);
        json.InsertValue("overdraft", Overdraft);
        json.InsertValue("receipt_sum", ReceiptSum);
        json.InsertValue("act_sum", ActSum);
        json.InsertValue("terminated", Terminated);
        return json;
    }

    bool TBalanceClientInfo::FromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "client_id", ClientId, true)
            && NJson::ParseField(json, "person_id", PersonId, false)
            && NJson::ParseField(json, "contract_id", ContractId, true)
            && NJson::ParseField(json, "external_contract_id", ExternalContractId, true)
            && NJson::ParseField(json, "terminated", Terminated, false)
            && NJson::ParseField(json, "receipt_sum", ReceiptSum, false)
            && NJson::ParseField(json, "act_sum", ActSum, false)
            && NJson::ParseField(json, "overdraft", Overdraft, false);
    }

    TString INotifySetting::GetActiveKey() const {
        return BuildActiveKey(GetName());
    }

    TString INotifySetting::GetLastNotifyKey() const {
        return BuildLastNotifyKey(GetName());
    }

    void INotifySetting::ToJson(NJson::TJsonValue& json) const {
        if (Active) {
            json.InsertValue(GetActiveKey(), Active);
            json.InsertValue(GetLastNotifyKey(), LastNotify.Seconds());
        }
    }

    bool INotifySetting::FromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json[GetActiveKey()], Active)
            && NJson::ParseField(json[GetLastNotifyKey()], LastNotify);
    }

    TString ILimitedNotifySetting::GetLimitKey() const {
        return BuildLimitKey(GetName());
    }

    void ILimitedNotifySetting::ToJson(NJson::TJsonValue& json) const {
        TBase::ToJson(json);
        if (TBase::GetActive()) {
            json.InsertValue(GetLimitKey(), Limit);
        }
    }

    bool ILimitedNotifySetting::FromJson(const NJson::TJsonValue& json) {
        return TBase::FromJson(json)
            && NJson::ParseField(json[GetLimitKey()], Limit);
    }

    void TNotifySettings::ToJson(NJson::TJsonValue& json) const {
        NJson::InsertNonNull(json, "emails", Emails);
        DailyBalanceNotify.ToJson(json);
        LowBalanceNotify.ToJson(json);
        BalanceChangeNotify.ToJson(json);
    }

    bool TNotifySettings::FromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json["emails"], Emails)
            && DailyBalanceNotify.FromJson(json)
            && LowBalanceNotify.FromJson(json)
            && BalanceChangeNotify.FromJson(json);
    }

    TString TDailyBalance::Name("daily_balance");
    TString TLowBalance::Name("low_balance");
    TString TBalanceChange::Name("balance_change");

    TAccountRecord::TFactory::TRegistrator<TLimitedAccountRecord> TLimitedAccountRecord::Registrator(EWalletDataType::Simple);
    IBillingAccount::TFactory::TRegistrator<TLimitedAccount> TLimitedAccount::Registrator(EAccount::Wallet);

    bool TLimitedAccountRecord::ParseMeta(const NJson::TJsonValue& meta) {
        TAccountRecord::ParseMeta(meta);
        if (!meta.Has("expenditure") || !meta["expenditure"].IsUInteger()) {
            return false;
        }
        Expenditure = meta["expenditure"].GetUInteger();
        if (!meta.Has("next_refresh")) {
            return false;
        }
        return PatchMeta(meta);
    }

    bool TLimitedAccountRecord::PatchMeta(const NJson::TJsonValue& meta) {
        TAccountRecord::PatchMeta(meta);
        if (meta.Has("next_refresh")) {
            if (!meta["next_refresh"].IsUInteger()) {
                return false;
            }
            NextRefresh = TInstant::Seconds(meta["next_refresh"].GetUInteger());
        }
        if (meta.Has("hard_limit")) {
            LocalHardLimit = meta["hard_limit"].GetUInteger();
        }
        if (meta.Has("soft_limit")) {
            LocalSoftLimit = meta["soft_limit"].GetUInteger();
        }
        if (meta.Has("offers_filter")) {
            LocalOffersFilter = meta["offers_filter"].GetString();
        }
        if (meta.Has("offers_filter_name")) {
            LocalOffersFilterName = meta["offers_filter_name"].GetString();
        }
        if (meta.Has("time_restrictions")) {
            LocalTimeRestrictions.ConstructInPlace();
            if (!LocalTimeRestrictions->DeserializeFromJson(meta["time_restrictions"])) {
                return false;
            }
        }
        if (meta.Has("balance_info")) {
            BalanceInfo.ConstructInPlace();
            if (!BalanceInfo->FromJson(meta["balance_info"])) {
                return false;
            }
        }

        if (!HasNotifySettings()) {
            TNotifySettings notifySettings;
            if (!notifySettings.FromJson(meta)) {
                return false;
            }
            if (notifySettings) {
                NotifySettings = notifySettings;
            }
        } else if (!NotifySettings->FromJson(meta)) {
            return false;
        }
        JREAD_UINT_OPT(meta, "refresh_interval", RefreshInterval);
        if (meta.Has("refresh_policy")) {
            TRefreshSchedule::EPolicy refreshPolicy;
            if (!TryFromString(meta["refresh_policy"].GetString(), refreshPolicy)) {
                return false;
            }
            RefreshPolicy = refreshPolicy;
            if (refreshPolicy == TRefreshSchedule::EPolicy::None && meta.Has("set_global_spent") && meta["set_global_spent"].GetBoolean()) {
                Expenditure = GetSpent();
            }
        }
        return NJson::ParseField(meta, "enable_toll_roads", LocalPayForTollRoads)
            && NJson::ParseField(meta, "custom_toll_roads", LocalTollRoads);
    }

    void TLimitedAccountRecord::SaveMeta(NJson::TJsonValue& meta) const {
        TAccountRecord::SaveMeta(meta);
        meta["expenditure"] = Expenditure;
        meta["next_refresh"] = NextRefresh.Seconds();
        meta["soft_limit"] = LocalSoftLimit;
        meta["hard_limit"] = LocalHardLimit;
        if (LocalOffersFilter) {
            meta["offers_filter"] = *LocalOffersFilter;
        }
        if (LocalOffersFilterName) {
            meta["offers_filter_name"] = *LocalOffersFilterName;
        }
        if (LocalTimeRestrictions && !LocalTimeRestrictions.Empty()) {
            meta["time_restrictions"] = LocalTimeRestrictions->SerializeToJson();
        }
        if (LocalPayForTollRoads) {
            meta["enable_toll_roads"] = *LocalPayForTollRoads;
        }
        if (LocalTollRoads && !LocalTollRoads->empty()) {
            NJson::InsertField(meta, "custom_toll_roads", *LocalTollRoads);
        }
        if (BalanceInfo) {
            meta["balance_info"] = BalanceInfo->ToJson();
        }
        if (RefreshPolicy) {
            meta["refresh_policy"] = ::ToString(RefreshPolicy);
            meta["refresh_interval"] = RefreshInterval;
        }
        if (NotifySettings) {
            NotifySettings->ToJson(meta);
        }
    }

    NDrive::TScheme TLimitedAccountRecord::DoGetScheme(const NDrive::IServer* /*server*/) const {
        NDrive::TScheme result;
        result.Add<TFSNumeric>("expenditure", "Затраты").SetVisual(TFSNumeric::EVisualType::Money).SetReadOnly(true);
        result.Add<TFSNumeric>("next_refresh", "Время следующего обновления").SetVisual(TFSNumeric::EVisualType::DateTime);
        result.Add<TFSNumeric>("hard_limit", "Верхний лимит").SetRequired(false);
        result.Add<TFSNumeric>("soft_limit", "Нижний лимит").SetRequired(false);
        result.Add<TFSText>("offers_filter", "Фильтр для тарифов (полный фильтр)");
        result.Add<TFSString>("offers_filter_name", "Фильтр для тарифов (имя фильтра)");
        result.Add<TFSStructure>("time_restrictions", "Расписание активации").SetStructure(TTimeRestrictionsPool<class TTimeRestriction>::GetScheme());
        result.Add<TFSBoolean>("enable_toll_roads", "Включить оплату платных дорог").SetDefault(false);
        result.Add<TFSArray>("custom_toll_roads", "Оплата отдельных дорог").SetElement<TFSString>();
        result.Add<TFSVariants>("refresh_policy", "Политика обновления").InitVariants<TRefreshSchedule::EPolicy>().SetDefault(::ToString(TRefreshSchedule::EPolicy::Month));
        result.Add<TFSNumeric>("refresh_interval", "Частота обновления").SetDefault(1);

        result.Add<TFSArray>("emails", "Почтовые адреса для уведомлений").SetElement<TFSString>();
        result.Add<TFSBoolean>(BuildActiveKey(TDailyBalance::Name), "Уведомлять об остатке на кошельке");
        result.Add<TFSNumeric>(BuildLastNotifyKey(TDailyBalance::Name), "Время последнего оповещения о балансе").SetVisual(TFSNumeric::EVisualType::DateTime);
        result.Add<TFSBoolean>(BuildActiveKey(TLowBalance::Name), "Уведомлять о низком балансе");
        result.Add<TFSNumeric>(BuildLastNotifyKey(TLowBalance::Name), "Время последнего оповещения о низком балансе").SetVisual(TFSNumeric::EVisualType::DateTime);
        result.Add<TFSNumeric>(BuildLimitKey(TLowBalance::Name), "Лимит для оповещения");
        result.Add<TFSBoolean>(BuildActiveKey(TBalanceChange::Name), "Уведомлять о резком изменении баланса");
        result.Add<TFSNumeric>(BuildLastNotifyKey(TBalanceChange::Name), "Время последнего оповещения об изменении баланса").SetVisual(TFSNumeric::EVisualType::DateTime);
        result.Add<TFSNumeric>(BuildLimitKey(TBalanceChange::Name), "Лимит изменений в процентах");
        return result;
    }

    i64 TLimitedAccount::GetBalance() const {
        const TLimitedAccountRecord& record = *GetRecordAs<TLimitedAccountRecord>();
        return GetSoftLimit() - record.GetExpenditure();
    }

    ui64 TLimitedAccount::GetCurrentSpent() const {
        auto record = GetRecordAs<TLimitedAccountRecord>();
        CHECK_WITH_LOG(record);
        return record->GetExpenditure();
    }

    TInstant TLimitedAccount::GetNextRefresh() const {
        auto record = GetRecordAs<TLimitedAccountRecord>();
        CHECK_WITH_LOG(record);
        if (record->HasRefreshPolicy() && record->GetRefreshPolicyRef() == TRefreshSchedule::EPolicy::None) {
            return TInstant::Max();
        }
        return record->GetNextRefresh();
    }

    ui64 TLimitedAccountRecord::GetHardLimit() const {
        return LocalHardLimit;
    }

    ui64 TLimitedAccountRecord::GetSoftLimit() const {
        return LocalSoftLimit;
    }

    TMaybe<TString> TLimitedAccountRecord::GetOffersFilter() const {
        return LocalOffersFilter;
    }
    TMaybe<TString> TLimitedAccountRecord::GetOffersFilterName() const {
        return LocalOffersFilterName;
    }
    TMaybe<TTimeRestrictionsPool<class TTimeRestriction>> TLimitedAccountRecord::GetTimeRestrictions() const {
        return LocalTimeRestrictions;
    }
    TMaybe<bool> TLimitedAccountRecord::EnableTollRoadsPay() const {
        return LocalPayForTollRoads;
    }
    TMaybe<TVector<TString>> TLimitedAccountRecord::GetTollRoadsToPayFor() const {
        return LocalTollRoads;
    }

    bool TLimitedAccount::DoRefresh(TAccountRecord::TPtr record) const {
        auto limited = VerifyDynamicCast<TLimitedAccountRecord*>(record.Get());
        CHECK_WITH_LOG(limited);
        limited->SetExpenditure(0);
        if (limited->HasRefreshPolicy()) {
            limited->SetNextRefresh(TRefreshSchedule::GetNextInstant(limited->GetRefreshPolicyRef(), limited->GetRefreshInterval(), Now()));
        }
        limited->SetNextRefresh(Description.GetNextInstant(Now()));
        return true;
    }

    EDriveOpResult TLimitedAccount::DoAdd(ui32 sum, NDrive::TEntitySession& session, TInstant /*deadline*/, const TString& /*source*/) const {
        TAccountRecord::TPtr record = ReadAccountData(Description.GetDataType(), session);
        if (!record) {
            return EDriveOpResult::TransactionError;
        }
        auto limited = VerifyDynamicCast<TLimitedAccountRecord*>(record.Get());
        ui64 expenditure = sum > limited->GetExpenditure() ? 0 : limited->GetExpenditure() - sum;
        limited->SetExpenditure(expenditure);
        return UpsertAccountData(*record, session, GetUserId()) ? EDriveOpResult::Ok : EDriveOpResult::TransactionError;
    }

    EDriveOpResult TLimitedAccount::DoRemove(ui32 sum, NDrive::TEntitySession& session) const {
        TAccountRecord::TPtr record = ReadAccountData(Description.GetDataType(), session);
        if (!record) {
            return EDriveOpResult::TransactionError;
        }
        auto limited = VerifyDynamicCast<TLimitedAccountRecord*>(record.Get());
        if (limited->GetExpenditure() + sum > GetHardLimit()) {
            return EDriveOpResult::LogicError;
        }

        ui64 expenditure = limited->GetExpenditure() + sum;
        limited->SetExpenditure(expenditure);
        ui64 spent = record->GetSpent() + sum;
        record->SetSpent(spent);
        return UpsertAccountData(*record, session, GetUserId()) ? EDriveOpResult::Ok : EDriveOpResult::TransactionError;
    }

    NJson::TJsonValue TLimitedAccount::GetNewUserReport(ELocalization locale, const IServerBase& server, const TVector<IBillingAccount::TPtr>& userAccounts, const IBillingAccount::TReportContext& context) const {
        auto result = IBillingAccount::GetNewUserReport(locale, server, userAccounts, context);
        result["b2b"] = NJson::TMapBuilder("company", GetCompanyName());
        return result;
    }
}
