#pragma once

#include <drive/backend/billing/interfaces/account.h>

namespace NDrive::NBilling {

    class TBalanceClientInfo {
        R_FIELD(ui64, ClientId, 0);
        R_FIELD(ui64, ContractId, 0);
        R_FIELD(ui64, PersonId, 0);
        R_FIELD(TString, ExternalContractId);
        R_FIELD(ui64, Overdraft, 0);
        R_FIELD(ui64, ReceiptSum, 0);
        R_FIELD(ui64, ActSum, 0);
        R_FIELD(bool, Terminated, false);

    public:
        NJson::TJsonValue ToJson() const;
        bool FromJson(const NJson::TJsonValue& json);
    };

    class INotifySetting {
    private:
        R_FIELD(bool, Active, false);
        R_FIELD(TInstant, LastNotify, TInstant::Zero());

    protected:
        virtual const TString& GetName() const = 0;

    public:
        TString GetActiveKey() const;
        TString GetLastNotifyKey() const;

        void ToJson(NJson::TJsonValue& json) const;
        bool FromJson(const NJson::TJsonValue& json);

        explicit operator bool() const {
            return Active || LastNotify;
        }
    };

    class ILimitedNotifySetting : public INotifySetting {
    private:
        using TBase = INotifySetting;
        R_FIELD(i64, Limit, 0);

        TString GetLimitKey() const;

    public:
        void ToJson(NJson::TJsonValue& json) const;
        bool FromJson(const NJson::TJsonValue& json);
    };

    class TDailyBalance : public INotifySetting {
    public:
        static TString Name;
    private:
        virtual const TString& GetName() const override {
            return Name;
        }
    };

    class TLowBalance : public ILimitedNotifySetting {
    public:
        static TString Name;
    private:
        virtual const TString& GetName() const override {
            return Name;
        }
    };

    class TBalanceChange : public ILimitedNotifySetting {
    public:
        static TString Name;
    private:
        virtual const TString& GetName() const override {
            return Name;
        }
    };

    class TNotifySettings {
    public:
        void ToJson(NJson::TJsonValue& json) const;
        bool FromJson(const NJson::TJsonValue& json);

        explicit operator bool() const {
            return Emails || DailyBalanceNotify || LowBalanceNotify || BalanceChangeNotify;
        }

    private:
        R_FIELD(TSet<TString>, Emails);
        R_FIELD(TDailyBalance, DailyBalanceNotify);
        R_FIELD(TLowBalance, LowBalanceNotify);
        R_FIELD(TBalanceChange, BalanceChangeNotify);
    };

    class TLimitedAccountRecord : public TAccountRecord {
    private:
        R_FIELD(ui64, LocalHardLimit, 0);
        R_FIELD(ui64, LocalSoftLimit, 0);
        R_FIELD(ui64, Expenditure, 0);
        R_FIELD(TInstant, NextRefresh, TInstant::Zero());
        R_OPTIONAL(TString, LocalOffersFilter);
        R_OPTIONAL(TString, LocalOffersFilterName);
        R_OPTIONAL(bool, LocalPayForTollRoads);
        R_OPTIONAL(TVector<TString>, LocalTollRoads);
        R_OPTIONAL(TTimeRestrictionsPool<class TTimeRestriction>, LocalTimeRestrictions);
        R_OPTIONAL(TBalanceClientInfo, BalanceInfo);
        R_OPTIONAL(TNotifySettings, NotifySettings);
        R_OPTIONAL(TRefreshSchedule::EPolicy, RefreshPolicy);
        R_READONLY(ui32, RefreshInterval, 1);

    public:
        TLimitedAccountRecord() = default;
        virtual bool ParseMeta(const NJson::TJsonValue& meta) override;
        virtual bool PatchMeta(const NJson::TJsonValue& meta) override;
        virtual void SaveMeta(NJson::TJsonValue& meta) const override;
        virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* /*server*/) const override;
        virtual ui64 GetHardLimit() const override;
        virtual ui64 GetSoftLimit() const override;
        virtual TMaybe<TString> GetOffersFilter() const override;
        virtual TMaybe<TString> GetOffersFilterName() const override;
        virtual TMaybe<bool> EnableTollRoadsPay() const override;
        virtual TMaybe<TVector<TString>> GetTollRoadsToPayFor() const override;
        virtual TMaybe<TTimeRestrictionsPool<class TTimeRestriction>> GetTimeRestrictions() const override;

    private:
        static TFactory::TRegistrator<TLimitedAccountRecord> Registrator;
    };

    class TLimitedAccount : public IBillingAccount {
    public:
        using IBillingAccount::IBillingAccount;

        i64 GetBalance() const override;
        virtual TInstant GetNextRefresh() const override;
        virtual bool DoRefresh(TAccountRecord::TPtr record) const override;
        ui64 GetCurrentSpent() const;
        virtual NJson::TJsonValue GetNewUserReport(ELocalization locale, const IServerBase& server, const TVector<IBillingAccount::TPtr>& userAccounts, const IBillingAccount::TReportContext& context) const override;

        TSet<EAccount> GetExcludeSwitchTypes() const override {
            return { EAccount::YAccount };
        }

    protected:
        virtual EDriveOpResult DoAdd(ui32 sum, NDrive::TEntitySession& session, TInstant deadline = TInstant::Max(), const TString& source = {}) const override;
        virtual EDriveOpResult DoRemove(ui32 sum, NDrive::TEntitySession& session) const override;

    private:
        static TFactory::TRegistrator<TLimitedAccount> Registrator;
    };

}
