#include "trust.h"

namespace NDrive::NBilling {

    bool TTrustAccountRecord::ParseMeta(const NJson::TJsonValue& meta) {
        if (!meta.Has("default_card")) {
            return false;
        }

        const NJson::TJsonValue& defaultCard = meta["default_card"];
        if (defaultCard.GetMap().empty()) {
            return true;
        }

        if (!defaultCard.Has("code")) {
            return false;
        }
        DefaultCard = defaultCard["code"].GetStringRobust();
        if (!defaultCard.Has("prefix") || !defaultCard["prefix"].IsInteger()) {
            return false;
        }
        PanPrefix = defaultCard["prefix"].GetUInteger();
        if (!defaultCard.Has("suffix") || !defaultCard["suffix"].IsInteger()) {
            return false;
        }
        PanSuffix = defaultCard["suffix"].GetUInteger();
        return PatchMeta(meta);
    }

    void TTrustAccountRecord::SaveMeta(NJson::TJsonValue& meta) const {
        if (!!DefaultCard) {
            meta["default_card"]["code"] = DefaultCard;
            meta["default_card"]["prefix"] = PanPrefix;
            meta["default_card"]["suffix"] = PanSuffix;
        } else {
            meta["default_card"] = NJson::TJsonValue(NJson::JSON_MAP);
        }
    }

    NDrive::TScheme TTrustAccountRecord::DoGetScheme(const NDrive::IServer* /*server*/) const {
        NDrive::TScheme cardInfo;
        cardInfo.Add<TFSString>("code", "code").SetReadOnly(true);
        cardInfo.Add<TFSNumeric>("prefix", "prefix").SetReadOnly(true);
        cardInfo.Add<TFSNumeric>("suffix", "suffix").SetReadOnly(true);

        NDrive::TScheme result;
        result.Add<TFSStructure>("default_card", "Данные карты по-умолчанию").SetStructure(cardInfo);
        return result;
    }

    TString TTrustAccountRecord::GetSuffixString() const {
        ui32 zeroCount = 0;
        if (GetPanSuffix() < 10) {
            zeroCount = 3;
        } else if (GetPanSuffix() < 100) {
            zeroCount = 2;
        } else if (GetPanSuffix() < 1000) {
            zeroCount = 1;
        }
        return TString(zeroCount, '0') + ::ToString(GetPanSuffix());
    }

    NJson::TJsonValue TTrustAccountRecord::GetReport() const {
        NJson::TJsonValue json;
        json["paymethod_id"] = GetDefaultCard();
        json["card_mask"] = ::ToString(GetPanPrefix()) + "****" + GetSuffixString();
        return json;
    }

    TAccountRecord::TFactory::TRegistrator<TTrustAccountRecord> TTrustAccountRecord::Registrator(EWalletDataType::Trust);

    i64 TTrustAccount::GetBalance() const {
        return 0;
    }

    TString TTrustAccount::GetDefaultCard() const {
        auto record = GetRecordAs<TTrustAccountRecord>();
        CHECK_WITH_LOG(record);
        return record->GetDefaultCard();
    }

    bool TTrustAccount::SetDefaultCard(const NDrive::NTrustClient::TPaymentMethod& defaultCard, const TString& historyUser, NDrive::TEntitySession& session) {
        TAccountRecord::TPtr record = ReadAccountData(EWalletDataType::Trust, session);
        if (!record) {
            return false;
        }
        auto trust = VerifyDynamicCast<TTrustAccountRecord*>(record.Get());

        trust->SetDefaultCard(defaultCard.GetId());
        ui32 prefix = 0;
        ui32 suffix = 0;
        if (defaultCard.GetAccount()) {
            TVector<TString> parts = SplitString(defaultCard.GetAccount(), "*");
            if (parts.size() != 2 || !TryFromString<ui32>(parts[0], prefix) || !TryFromString<ui32>(parts[1], suffix)) {
                ERROR_LOG << "Incorrect account format from trust: " << defaultCard.GetAccount() << Endl;
                return false;
            }
        }

        trust->SetPanPrefix(prefix);
        trust->SetPanSuffix(suffix);
        return UpsertAccountData(*record, session, historyUser);
    }

    NJson::TJsonValue TTrustAccount::GetUserReport() const {
        auto record = GetRecordAs<TTrustAccountRecord>();
        CHECK_WITH_LOG(record);
        NJson::TJsonValue json;
        json["paymethod_id"] = record->GetDefaultCard();
        json["pan"]["suffix"] = record->GetSuffixString();
        json["pan"]["prefix"] = ::ToString(record->GetPanPrefix());
        return json;
    }

    IBillingAccount::TFactory::TRegistrator<TTrustAccount> TTrustAccount::Registrator(EAccount::Trust);
}
