#include "yandex_account.h"

#include <rtline/library/json/parse.h>
#include <util/string/subst.h>

namespace NDrive::NBilling {

    bool TYandexAccountRecord::ParseMeta(const NJson::TJsonValue& meta) {
        return NJson::ParseField(meta, "paymethod_id", PaymethodId, true) && TAccountRecord::ParseMeta(meta);
    }

    bool TYandexAccountRecord::PatchMeta(const NJson::TJsonValue& meta) {
        if (meta.Has("paymethod_id")) {
            PaymethodId = meta["paymethod_id"].GetString();
        }
        return TAccountRecord::PatchMeta(meta);
    }

    void TYandexAccountRecord::SaveMeta(NJson::TJsonValue& meta) const {
        meta["paymethod_id"] = PaymethodId;
    }

    NDrive::TScheme TYandexAccountRecord::DoGetScheme(const NDrive::IServer* /*server*/) const {
        NDrive::TScheme result;
        result.Add<TFSString>("paymethod_id").SetReadOnly(true);
        return result;
    }

    NJson::TJsonValue TYandexAccountRecord::GetReport() const {
        NJson::TJsonValue json;
        json["paymethod_id"] = PaymethodId;
        return json;
    }

    TAccountRecord::TFactory::TRegistrator<TYandexAccountRecord> TYandexAccountRecord::Registrator(EWalletDataType::YAccount);

    i64 TYandexAccount::GetBalance() const {
        return 0;
    }

    TString GetBalanceString(i64 balance, ELocalization locale, const IServerBase& server) {
        auto local = server.GetLocalization();
        return (balance < 0 ? "-" : "") + (local ? local->FormatNumeral(std::abs(balance / 100), locale, true, "\u00A0", "units.full.yandex_account", "баллов", "балл", "балла") : ::ToString(std::abs(balance / 100)));
    }

    TString GetBalanceReport(const TString& key, const TString& balance, ELocalization locale, const IServerBase& server) {
        auto local = server.GetLocalization();
        auto valueTemplate = local ? local->GetLocalString(locale, key, "Баланс: _balance_") : "_balance_";
        SubstGlobal(valueTemplate, "_balance_", balance);
        return valueTemplate;
    }

    NJson::TJsonValue TYandexAccount::GetNewUserReport(ELocalization locale, const IServerBase& server, const TVector<IBillingAccount::TPtr>& userAccounts, const IBillingAccount::TReportContext& context) const {
        auto result = IBillingAccount::GetNewUserReport(locale, server, userAccounts, context);
        i64 balance = (context.ExternalBalance ? *context.ExternalBalance : GetBalance());
        auto balanceString = GetBalanceString(balance, locale, server);
        auto name = GetName();
        SubstGlobal(name, "_balance_", balanceString);
        result["name"] = name;

        result["value"] = GetBalanceReport("yandex_account_balance", balanceString, locale, server);
        result["description"] = result["value"];
        result["request_parameters"].InsertValue("yandex_account_balance", ::ToString(balance));
        return result;
    }

    IBillingAccount::TFactory::TRegistrator<TYandexAccount> TYandexAccount::Registrator(EAccount::YAccount);

    NJson::TJsonValue TFakeYandexAccount::GetNewUserReport(ELocalization locale, const IServerBase& server, const TVector<IBillingAccount::TPtr>& userAccounts, const IBillingAccount::TReportContext& context) const {
        auto result = IBillingAccount::GetNewUserReport(locale, server, userAccounts, context);
        i64 balance = (context.ExternalBalance ? *context.ExternalBalance : GetBalance());
        auto balanceString = GetBalanceString(balance, locale, server);
        auto name = GetName();
        SubstGlobal(name, "_balance_", balanceString);
        result["name"] = name;

        result["value"] = GetBalanceReport("yandex_account_balance_fake", balanceString, locale, server);
        result["description"] = result["value"];
        result.InsertValue("link", server.GetSettings().GetValueDef<TString>("plus_wallet_landing", "https://plus.yandex.ru/"));
        return result;
    }
}
