#include "accounts_manager.h"

#include <drive/backend/billing/trust/account_old.h>
#include <drive/backend/logging/evlog.h>
#include <rtline/library/storage/structured.h>

namespace {
    TNamedSignalSimple BillingBonusAccountCreate("billing-bonus-account-create");
    TNamedSignalSimple BillingTrustAccountCreate("billing-trust-account-create");
    TNamedSignalSimple BillingAccountCreateError("billing-account-create-error");
}

namespace NDrive::NBilling {

    bool TAccountsManager::DoStart() {
        return DescriptionsDB.Start()
            && AccountLinksDB.Start()
            && AccountsDB.Start();
    }

    bool TAccountsManager::DoStop() {
        return AccountsDB.Stop()
            && AccountLinksDB.Stop()
            && DescriptionsDB.Stop();
    }

    TAccountsManager::TAccountsManager(TDatabasePtr database, const ISettings& settings, const THistoryConfig& historyConfig)
        : Database(database)
        , DescriptionsDB(*this, historyConfig)
        , AccountsDB(*this, historyConfig)
        , AccountLinksDB(*this)
        , Settings(settings)
    {
    }

    TAccountsManager::~TAccountsManager() {
    }

    bool TAccountsManager::GetUserAccountDescriptions(const TString& userId, const TInstant lastInstant, TMap<ui32, TAccountDescriptionRecord>& result, TVector<TAccountData>* accData) const {
        result.clear();
        auto userLinks = AccountLinksDB.GetCachedUserAccounts(userId, lastInstant);
        if (!userLinks) {
            ERROR_LOG << userLinks.GetError() << Endl;
            return false;
        }

        TVector<TAccountData> accountsData;
        if (!userLinks->empty()) {
            TSet<ui32> accountsIds;
            Transform(userLinks->begin(), userLinks->end(), std::inserter(accountsIds, accountsIds.begin()), [](const auto& acc){ return acc.GetAccountId(); });
            if (!AccountsDB.GetCustomObjectsFromCache(accountsData, accountsIds, lastInstant)) {
                ERROR_LOG << "Error reading accounts info" << Endl;
                return false;
            }
        }

        ui32 cardTypeId;
        if (!Settings.GetValue("billing.card_account_type", cardTypeId)) {
            ERROR_LOG << "Error reading settings" << Endl;
            return false;
        }

        TSet<ui32> accountTypeIds;
        accountTypeIds.emplace(cardTypeId);
        for (auto accData : accountsData) {
            accountTypeIds.emplace(accData.GetRecord()->GetTypeId());
        }

        for (auto&& description : GetRegisteredAccounts(accountTypeIds, lastInstant)) {
            result.emplace(description.GetId(), description);
        }

        if (accData) {
            *accData = std::move(accountsData);
        }
        return true;
    }

    TVector<NDrive::NBilling::IBillingAccount::TPtr> TAccountsManager::GetUserAccounts(const TString& userId, const TInstant lastInstant) const {
        auto eg1 = NDrive::BuildEventGuard("GetUserAccounts");
        auto evlog = NDrive::GetThreadEventLogger();
        TVector<IBillingAccount::TPtr> results;
        TVector<TAccountData> accountsData;
        TMap<ui32, TAccountDescriptionRecord> descriptions;
        if (!GetUserAccountDescriptions(userId, lastInstant, descriptions, &accountsData)) {
            results.push_back(TTrustAccountOld::ConstructAccount(userId, Database, AccountsDB.GetHistoryManager()));
            return results;
        }
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
            ("event", "UserAccountDescriptionsResult")
            ("count", descriptions.size())
            );
        }

        for (auto accData : accountsData) {
            auto it = descriptions.find(accData.GetRecord()->GetTypeId());
            if (it == descriptions.end()) {
                continue;
            }
            IBillingAccount::TPtr parent;
            if (it->second.GetParentId()) {
                auto eg = NDrive::BuildEventGuard("GetAccountParent");
                parent = GetAccountById(it->second.GetParentId(), lastInstant);
            }

            auto eg = NDrive::BuildEventGuard("ConstructBillingAccount");
            auto accountImpl = IBillingAccount::Construct(userId, it->second, accData.GetRecord(), parent, AccountsDB.GetHistoryManager());
            if (accountImpl) {
                results.emplace_back(accountImpl);
            }
        }

        auto eg2 = NDrive::BuildEventGuard("AddTrustIfNotExists");
        bool hasTrust = false;
        for (auto&& account : results) {
            if (account->GetType() == EAccount::Trust) {
                hasTrust = true;
            }
        }
        if (!hasTrust) {
            auto eg = NDrive::BuildEventGuard("ConstructOldTrustAccount");
            results.push_back(TTrustAccountOld::ConstructAccount(userId, Database, AccountsDB.GetHistoryManager()));
        }
        return results;
    }

    TVector<NDrive::NBilling::IBillingAccount::TPtr> SortUserAccounts(TVector<NDrive::NBilling::IBillingAccount::TPtr>& userAccounts) {
        const auto compare = [](const NDrive::NBilling::IBillingAccount::TPtr& left, const NDrive::NBilling::IBillingAccount::TPtr& right) -> bool {
            return left->GetPriority() > right->GetPriority() || (left->GetPriority() == right->GetPriority() && left->GetId() < right->GetId());
        };
        Sort(userAccounts.begin(), userAccounts.end(), compare);
        return userAccounts;
    }

    TVector<NDrive::NBilling::IBillingAccount::TPtr> TAccountsManager::GetSortedUserAccounts(const TString& userId, const TInstant lastInstant) const {
        auto userAccounts = GetUserAccounts(userId, lastInstant);
        SortUserAccounts(userAccounts);
        return userAccounts;
    }

    TMaybe<TVector<NDrive::NBilling::IBillingAccount::TPtr>> TAccountsManager::GetUserAccounts(const TString& userId, NDrive::TEntitySession& session) const {
        auto eg1 = NDrive::BuildEventGuard("GetUserAccountsBySession");
        auto evlog = NDrive::GetThreadEventLogger();

        auto accounts = GetUsersAccounts({userId}, session);
        if (!accounts) {
            return {};
        }
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
            ("event", "GetUsersAccounts")
            ("count", accounts->size())
            );
        }

        auto eg2 = NDrive::BuildEventGuard("AddTrustIfNotExists");
        bool hasTrust = false;
        for (auto&& account : *accounts) {
            if (account->GetType() == EAccount::Trust) {
                hasTrust = true;
            }
        }
        if (!hasTrust) {
            auto eg = NDrive::BuildEventGuard("ConstructOldTrustAccount");
            accounts->push_back(TTrustAccountOld::ConstructAccount(userId, Database, AccountsDB.GetHistoryManager()));
        }
        return *accounts;
    }

    TMaybe<TVector<NDrive::NBilling::IBillingAccount::TPtr>> TAccountsManager::GetUsersAccounts(const TSet<TString>& userIds, NDrive::TEntitySession& session) const {
        auto eg1 = NDrive::BuildEventGuard("GetUserAccountsBySession");
        auto evlog = NDrive::GetThreadEventLogger();

        auto accountLinks = AccountLinksDB.GetUsersAccounts(userIds, session);
        if (!accountLinks) {
            return {};
        }
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
            ("event", "GetLinkedUserAccounts")
            ("count", accountLinks->size())
            );
        }

        TSet<ui64> accountIds;
        Transform(accountLinks->begin(), accountLinks->end(), std::inserter(accountIds, accountIds.begin()), [](const auto& link){ return link.GetAccountId(); });

        NSQL::TQueryOptions options;
        options.SetGenericCondition("account_id", accountIds);

        auto accounts = GetAccounts(options, session);
        if (!accounts) {
            return {};
        }

        TMap<ui64, TSet<TString>> userAccounts;
        ForEach(accountLinks->begin(), accountLinks->end(), [&userAccounts](const auto& link) {
            userAccounts[link.GetAccountId()].emplace(link.GetUserId());
        });

        return GetBillingAccounts(*accounts, session, userAccounts);
    }

    TMaybe<TVector<NDrive::NBilling::IBillingAccount::TPtr>> TAccountsManager::GetSortedUserAccounts(const TString& userId, NDrive::TEntitySession& session) const {
        auto userAccounts = GetUserAccounts(userId, session);
        if (userAccounts) {
            SortUserAccounts(*userAccounts);
        }
        return userAccounts;
    }

    TMaybe<TVector<NDrive::NBilling::IBillingAccount::TPtr>> TAccountsManager::GetAccountsByName(const TString& accName, const TInstant lastInstant) const {
        TVector<IBillingAccount::TPtr> results;
        TMaybe<TAccountDescriptionRecord> description = GetDescriptionByName(accName, lastInstant);
        if (!description.Defined()) {
            return {};
        }

        IBillingAccount::TPtr parent;
        if (description->GetParentId()) {
            parent = GetAccountById(description->GetParentId(), lastInstant);
        }

        const auto action = [&results, &description, &parent, this](const TAccountData& accData) {
            auto accountImpl = IBillingAccount::Construct("fake", description.GetRef(), accData.GetRecord(), parent, AccountsDB.GetHistoryManager());
            if (accountImpl) {
                results.emplace_back(accountImpl);
            }
        };
        if (!AccountsDB.ListAccountsByTypeId(description->GetId(), action, lastInstant)) {
            ERROR_LOG << "Error reading accounts info" << Endl;
            return {};
        }
        return results;
    }

    TMaybe<TVector<IBillingAccount::TPtr>> TAccountsManager::GetAccountsByIds(const TSet<ui32>& accountIds, const TInstant lastInstant) const {
        TVector<TAccountData> accountsData;
        if (!AccountsDB.GetCustomObjectsFromCache(accountsData, accountIds, lastInstant)) {
            ERROR_LOG << "Error reading accounts info" << Endl;
            return {};
        }

        auto billingAccounts = GetBillingAccounts(accountsData, lastInstant);
        if (!billingAccounts) {
            ERROR_LOG << billingAccounts.GetError() << Endl;
            return {};
        }
        return *billingAccounts;
    }

    TExpected<TVector<IBillingAccount::TPtr>, TString> TAccountsManager::GetBillingAccounts(const TVector<TAccountData>& accountsData, const TInstant lastInstant) const {
        TSet<ui32> typeIds;
        for (const auto& accData : accountsData) {
            typeIds.emplace(Yensured(accData.GetRecord())->GetTypeId());
        }

        TVector<TAccountDescriptionRecord> descriptions;
        if (!DescriptionsDB.GetCustomObjectsByTypeIds(descriptions, typeIds, lastInstant)) {
            return MakeUnexpected<TString>("cannot fetch descriptions");
        }

        TMap<ui32, IBillingAccount::TPtr> parents;
        TMap<ui32, TAccountDescriptionRecord> descriptionByType;
        for (auto&& desc : descriptions) {
            ui32 id = desc.GetId();
            if (desc.GetParentId()) {
                parents.emplace(id, GetAccountById(desc.GetParentId(), lastInstant));
            }
            descriptionByType.emplace(id, std::move(desc));
        }

        TVector<IBillingAccount::TPtr> results;
        for (const auto& accData : accountsData) {
            auto typeIt = descriptionByType.find(accData.GetRecord()->GetTypeId());
            if (typeIt == descriptionByType.end()) {
                return MakeUnexpected<TString>("cannot fetch description " + ToString(accData.GetRecord()->GetTypeId()));
            }

            IBillingAccount::TPtr parent;
            auto parentIt = parents.find(accData.GetRecord()->GetTypeId());
            if (parentIt != parents.end()) {
                parent = parentIt->second;
            }

            auto accountImpl = IBillingAccount::Construct("fake", typeIt->second, accData.GetRecord(), parent, AccountsDB.GetHistoryManager());
            if (accountImpl) {
                results.emplace_back(accountImpl);
            }
        }
        return results;
    }

    IBillingAccount::TPtr TAccountsManager::GetAccountById(const ui32 accountId, const TInstant lastInstant /*= TInstant::Zero()*/) const {
        auto result = GetAccountsByIds({ accountId }, lastInstant);
        if (result && result->size()) {
            return result->front();
        }
        return nullptr;
    }

    bool TAccountsManager::UpsertAccountDescription(const TAccountDescriptionRecord& description, const TString& userId, NDrive::TEntitySession& session) const {
        return DescriptionsDB.Upsert(description, userId, session);
    }


    bool TAccountsManager::RemoveAccountDescription(const ui32 typeId, const TString& originatiorId, NDrive::TEntitySession& session) const {
        return DescriptionsDB.Remove(typeId, originatiorId, session);
    }

    bool TAccountsManager::RegisterAccount(const TAccountRecord::TPtr& accountRecord, const TString& historyUser, NDrive::TEntitySession& session, ui32* newAccountId /*= nullptr*/) const {
        if (!accountRecord) {
            return false;
        }

        auto table = Database->GetTable("billing_account");
        NStorage::TObjectRecordsSet<TAccountData, IHistoryContext> affected(this);
        auto result = table->AddRow(accountRecord->SerializeToTableRecord(), session.GetTransaction(), "", &affected);
        if (!result->IsSucceed() || affected.size() != 1) {
            return false;
        }
        auto updateResult = affected.front();

        if (!!newAccountId) {
            *newAccountId = updateResult.GetRecord()->GetAccountId();
        }
        return AccountsDB.GetHistoryManager().AddHistory(updateResult, historyUser, EObjectHistoryAction::Add, session);
    }

    IBillingAccount::TPtr TAccountsManager::LinkAccount(const TString& userId, const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const {
        auto table = Database->GetTable("billing_account");
        NStorage:: TObjectRecordsSet<TAccountData, IHistoryContext> records(this);
        auto qResult = table->GetRows(NStorage::TRecordBuilder("account_id", accountId), records, session.GetTransaction());
        if (!qResult->IsSucceed() || records.size() != 1) {
            return nullptr;
        }

        auto description = GetDescriptionById(records.front().GetRecord()->GetTypeId(), TInstant::Zero());
        if (!description.Defined() || !description->GetIsPersonal()) {
            return nullptr;
        }

        if (!AccountLinksDB.AddLink(userId, accountId, records.front().GetRecord()->GetTypeId(), historyUser, description->GetMaxLinks(), session)) {
            return nullptr;
        }
        return IBillingAccount::Construct( userId, description.GetRef(), records.front().GetRecord(), nullptr, AccountsDB.GetHistoryManager());
    }

    bool TAccountsManager::UnLinkAccount(const TString& userId, const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const {
        return AccountLinksDB.RemoveLink(userId, accountId, historyUser, session);
    }

    bool TAccountsManager::ActivateAccounts(TConstArrayRef<ui32> accountIds, bool activeFlag, const TString& historyUser, NDrive::TEntitySession& session) const {
        auto transaction = session.GetTransaction();
        if (!transaction) {
            return false;
        }
        if (accountIds.empty()) {
            return true;
        }
        TString condition = "account_id IN (" +  JoinSeq(", ", accountIds) + ")";
        NStorage::TRecordBuilder update("active", activeFlag);
        TString updateStr = NStorage::TTableRecord(update).BuildSet(*transaction);
        updateStr += ", version = version + 1";

        NStorage::TObjectRecordsSet<TAccountData, IHistoryContext> affected(this);
        auto qResult = Database->GetTable("billing_account")->UpdateRow(condition, updateStr, transaction, &affected);
        if (!qResult->IsSucceed() || affected.size() != accountIds.size()) {
            return false;
        }
        for(auto&& updateResult : affected) {
            if (!AccountsDB.GetHistoryManager().AddHistory(updateResult, historyUser, EObjectHistoryAction::UpdateData, session)) {
                return false;
            }
        }
        return true;
    }

    bool TAccountsManager::RemoveAccount(const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const {
        NStorage::TObjectRecordsSet<TAccountData, IHistoryContext> affected(this);
        NStorage::TRecordBuilder condition("account_id", accountId);
        auto qResult = Database->GetTable("billing_account")->RemoveRow(condition, session.GetTransaction(), &affected);
        if (!qResult->IsSucceed() || affected.size() != 1) {
            session.SetErrorInfo("accounts_manager", "remove_error", EDriveSessionResult::ResourceLocked);
            return false;
        }
        auto removeResult = affected.front();
        return AccountsDB.GetHistoryManager().AddHistory(removeResult, historyUser, EObjectHistoryAction::Remove, session);
    }

    TMaybe<TVector<TAccountData>> TAccountsManager::GetAccounts(const NSQL::TQueryOptions& options, NDrive::TEntitySession& session) const {
        NStorage::TObjectRecordsSet<TAccountData, IHistoryContext> records(this);
        auto transaction = session.GetTransaction();
        auto query = options.PrintQuery(*Yensured(transaction), "billing_account");
        auto queryResult = transaction->Exec(query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return {};
        }
        return records.DetachObjects();
    }

    TMaybe<TMap<ui64, IBillingAccount::TPtr>> TAccountsManager::GetBillingAccounts(const TSet<ui64>& accountIds, NDrive::TEntitySession& session, const TString& userId) const {
        NSQL::TQueryOptions options;
        options.SetGenericCondition("account_id", accountIds);

        auto accounts = GetBillingAccounts(options, session, userId);
        if (!accounts) {
            return {};
        }

        TMap<ui64, IBillingAccount::TPtr> result;
        for (const auto& id : accountIds) {
            result[id] = nullptr;
        }
        for (const auto& account : *accounts) {
            result[account->GetId()] = std::move(account);
        }
        return result;
    }

    TMaybe<TVector<IBillingAccount::TPtr>> TAccountsManager::GetBillingAccounts(const NSQL::TQueryOptions& options, NDrive::TEntitySession& session, const TString& userId) const {
        auto accounts = GetAccounts(options, session);
        if (!accounts) {
            return {};
        }

        TMap<ui64, TSet<TString>> userAccounts;
        ForEach(accounts->begin(), accounts->end(), [&userAccounts, &userId](const auto& acc) {
            userAccounts[acc.GetAccountId()].emplace(userId);
        });

        return GetBillingAccounts(*accounts, session, userAccounts);
    }

    TMaybe<TVector<IBillingAccount::TPtr>> TAccountsManager::GetBillingAccounts(const TVector<TAccountData>& accounts, NDrive::TEntitySession& session, const TMap<ui64, TSet<TString>>& userIds) const {
        TSet<ui32> typeIds;
        for (const auto& accData : accounts) {
            typeIds.emplace(Yensured(accData.GetRecord())->GetTypeId());
        }

        auto descriptions = DescriptionsDB.GetDescriptionsById(typeIds, session);
        if (!descriptions) {
            return {};
        }

        TSet<ui64> parentIds;
        for (auto&& [id, desc] : descriptions.GetResult()) {
            if (desc.GetParentId()) {
                parentIds.emplace(desc.GetParentId());
            }
        }

        TMap<ui64, IBillingAccount::TPtr> parents;
        if (parentIds) {
            auto parentAccounts = GetBillingAccounts(parentIds, session);
            if (!parentAccounts) {
                return {};
            }
            parents = std::move(*parentAccounts);
        }

        TVector<IBillingAccount::TPtr> results;
        for (const auto& accData : accounts) {
            auto typeIt = descriptions.GetResultPtr(accData.GetRecord()->GetTypeId());
            if (!typeIt) {
                session.SetErrorInfo("GetBillingAccounts", "unknown description for " + ToString(accData.GetAccountId()));
                return {};
            }

            IBillingAccount::TPtr parent;
            auto parentIt = parents.find(typeIt->GetParentId());
            if (parentIt != parents.end()) {
                parent = parentIt->second;
            }

            TSet<TString> defaultUser({ "fake" });
            TSet<TString> users = userIds.Value(accData.GetAccountId(), defaultUser);
            if (!users) {
                users = defaultUser;
            }
            for (const auto& user : users) {
                auto accountImpl = IBillingAccount::Construct(user, *typeIt, accData.GetRecord(), parent, AccountsDB.GetHistoryManager());
                if (accountImpl) {
                    results.emplace_back(accountImpl);
                } else {
                    session.SetErrorInfo("GetBillingAccounts", "cannot construct " + ToString(accData.GetAccountId()));
                    return {};
                }
            }
        }
        return results;
    }

    TVector<NDrive::NBilling::TAccountDescriptionRecord> TAccountsManager::GetRegisteredAccounts(const TInstant lastInstant) const {
        TVector<TAccountDescriptionRecord> accounts;
        if (!DescriptionsDB.GetAllObjectsFromCache(accounts, lastInstant)) {
            ERROR_LOG << "Error reading accounts description" << Endl;
        }
        return accounts;
    }

    TVector<NDrive::NBilling::TAccountDescriptionRecord> TAccountsManager::GetRegisteredAccounts(const TSet<ui32>& typeIds, const TInstant reqInstant) const {
        TVector<TAccountDescriptionRecord> accounts;
        if (!DescriptionsDB.GetCustomObjectsByTypeIds(accounts, typeIds, reqInstant)) {
            ERROR_LOG << "Error reading accounts description" << Endl;
        }
        return accounts;
    }

    IBillingAccount::TPtr TAccountsManager::GetOrCreateAccount(const TString& userId, const TAccountDescriptionRecord& description, const TString& historyUser, NDrive::TEntitySession& session) const {
        auto userAccounts = GetUserAccounts(userId, session);
        if (!userAccounts) {
            return nullptr;
        }
        for (auto&& account : *userAccounts) {
            if (account->GetUniqueName() == description.GetName()) {
                return account;
            }
        }

        TAccountRecord::TPtr accountRecord = TAccountRecord::TFactory::Construct(description.GetDataType());
        accountRecord->SetTypeId(description.GetId());
        accountRecord->SetActive(true);

        ui32 newAccountId = 0;

        if (!RegisterAccount(accountRecord, historyUser, session, &newAccountId)) {
            BillingAccountCreateError.Signal(1);
            return nullptr;
        }

        IBillingAccount::TPtr linkedAccount = LinkAccount(userId, newAccountId, historyUser, session);
        if (!linkedAccount) {
            BillingAccountCreateError.Signal(1);
            return nullptr;
        }

        BillingBonusAccountCreate.Signal(1);
        return linkedAccount;
    }

    IBillingAccount::TPtr TAccountsManager::GetOrCreateTrustAccount(const TString& userId, const TString& historyUser, NDrive::TEntitySession& session) const {
        IBillingAccount::TPtr trustAccount = GetTrustAccount(userId, session);
        if (!trustAccount) {
            return nullptr;
        }

        if (!!trustAccount->GetAs<TTrustAccountOld>()) {
            ui32 typeId;
            if (!Settings.GetValue("billing.card_account_type", typeId)) {
                session.SetErrorInfo("GetOrCreateTrustAccount", "cannot find card type id");
                return nullptr;
            }
            TAccountRecord::TPtr accountRecord = TAccountRecord::TFactory::Construct(EWalletDataType::Trust);
            accountRecord->SetTypeId(typeId);
            accountRecord->SetActive(true);

            ui32 trustAccountId = 0;

            if (!RegisterAccount(accountRecord, historyUser, session, &trustAccountId)) {
                BillingAccountCreateError.Signal(1);
                return nullptr;
            }

            IBillingAccount::TPtr linkedAccount = LinkAccount(userId, trustAccountId, historyUser, session);
            if (!linkedAccount) {
                BillingAccountCreateError.Signal(1);
                return nullptr;
            }

            BillingTrustAccountCreate.Signal(1);
            return linkedAccount;
        }
        return trustAccount;
    }

    TMaybe<NDrive::NBilling::TAccountDescriptionRecord> TAccountsManager::GetDescriptionByName(const TString& name, TInstant lastInstant) const {
        TVector<TAccountDescriptionRecord> description;
        if (!DescriptionsDB.GetCustomObjectsFromCache(description, NContainer::Scalar(name), lastInstant) || description.empty()) {
            ERROR_LOG << "Error reading accounts info" << Endl;
            return TMaybe<TAccountDescriptionRecord>();
        }
        return description.front();
    }


    NJson::TJsonValue TAccountsManager::GetScheme(const NDrive::IServer* server) const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        result["accounts"] = TAccountDescriptionRecord::GetScheme(server).SerializeToJson();

        NJson::TJsonValue& dataSchemas = result.InsertValue("accounts_data", NJson::JSON_MAP);
        TSet<EWalletDataType> keys;
        TAccountRecord::TFactory::GetRegisteredKeys(keys);
        for (auto&& dType : keys) {
            auto instance = Hold(TAccountRecord::TFactory::Construct(dType));
            dataSchemas.InsertValue(::ToString(dType), instance->GetScheme(nullptr).SerializeToJson());
        }
        return result;
    }

    TMap<TString, ui64> TAccountsManager::GetAccountsChildren(const TSet<ui64>& parentIds, TInstant lastInstant) const {
        TMap<TString, ui64> links;
        for (auto&& description : GetRegisteredAccounts(lastInstant)) {
            if (description.GetParentId() != 0 && parentIds.contains(description.GetParentId())) {
                links.emplace(description.GetName(), description.GetParentId());
            }
        }
        return links;
    }

    TMaybe<TMap<TString, ui64>> TAccountsManager::GetAccountsChildren(const TAccountParentFilter& parentFilter, TInstant lastInstant) const {
        TSet<ui64> parentIds = parentFilter.ParentIds;
        TMap<TString, ui64> result;
        if (!parentFilter.Accounts.empty()) {
            for (auto&& description : GetRegisteredAccounts(lastInstant)) {
                if (parentFilter.WithBonusAccounts && description.GetType() == NDrive::NBilling::EAccount::Bonus) {
                    result.emplace(description.GetName(), 0);
                    continue;
                }
                if (parentFilter.Accounts.contains(description.GetName())) {
                    if (parentFilter.WithParent) {
                        result.emplace(description.GetName(), 0);
                    }
                    if (!description.GetIsPersonal()) {
                        auto accounts = GetAccountsByName(description.GetName(), lastInstant);
                        if (!accounts) {
                            return {};
                        }
                        for (auto acc : *accounts) {
                            parentIds.emplace(acc->GetId());
                        }
                    }
                }
            }
        }
        auto children = GetAccountsChildren(parentIds, lastInstant);
        result.insert(children.begin(), children.end());
        return result;
    }

    const NDrive::NBilling::TAccountsHistoryManager& TAccountsManager::GetAccountsHistory() const {
        return AccountsDB.GetHistoryManager();
    }

    const NDrive::NBilling::TAccountDescriptionHistory& TAccountsManager::GetAccountsDescriptionHistory() const {
        return DescriptionsDB.GetHistoryManager();
    }

    const NDrive::NBilling::TAccountsLinksDB& TAccountsManager::GetLinksManager() const {
        return AccountLinksDB;
    }

    const NDrive::NBilling::TAccountsDescriptionDB& TAccountsManager::GetDescriptionDB() const {
        return DescriptionsDB;
    }

    TMaybe<NDrive::NBilling::TAccountDescriptionRecord> TAccountsManager::GetDescriptionById(ui32 typeId, TInstant lastInstant) const {
        return DescriptionsDB.GetDescriptionById(typeId, lastInstant);
    }

    ui32 TAccountsManager::GetBonuses(const TString& userId, const TInstant lastInstant) const {
        return GetBonuses(GetUserAccounts(userId, lastInstant));
    }

    ui32 TAccountsManager::GetBonuses(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts) {
        ui32 bonuses = 0;
        for (auto&& account : accounts) {
            if (account && account->IsActive() && (account->GetType() == NDrive::NBilling::EAccount::Bonus || account->GetType() == NDrive::NBilling::EAccount::Coins)) {
                bonuses += account->GetBalance();
            }
        }
        return bonuses;
    }

    NDrive::NBilling::TLimitedBalances TAccountsManager::GetLimitedBonuses(const TString& userId, const TInstant lastInstant) const {
        return GetLimitedBonuses(GetUserAccounts(userId, lastInstant));
    }

    NDrive::NBilling::TLimitedBalances TAccountsManager::GetLimitedBonuses(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts) {
        NDrive::NBilling::TLimitedBalances bonuses;
        for (auto&& account : accounts) {
            if (account && account->IsActive() && account->GetType() == NDrive::NBilling::EAccount::Bonus || account->GetType() == NDrive::NBilling::EAccount::Coins) {
                for (const auto& bonus : account->GetLimitedBalance()) {
                    auto it = bonuses.find(bonus);
                    NDrive::NBilling::TLimitedBalance newBonus = bonus;
                    if (it != bonuses.end()) {
                        newBonus.MutableBalance() += it->GetBalance();
                        bonuses.erase(it);
                    }
                    bonuses.emplace(bonus);
                }
            }
        }
        return bonuses;
    }

    NDrive::NBilling::IBillingAccount::TPtr TAccountsManager::GetTrustAccount(const TString& userId, const TInstant actuality /*= TInstant::Zero()*/) const {
        auto eg = NDrive::BuildEventGuard("GetTrustAccount");
        return GetTrustAccount(GetUserAccounts(userId, actuality));
    }

    NDrive::NBilling::IBillingAccount::TPtr TAccountsManager::GetTrustAccount(const TString& userId, NDrive::TEntitySession& session) const {
        auto eg = NDrive::BuildEventGuard("GetTrustAccountBySession");
        auto accounts = GetUserAccounts(userId, session);
        if (!accounts) {
            return nullptr;
        }
        return GetTrustAccount(*accounts);
    }

    NDrive::NBilling::IBillingAccount::TPtr TAccountsManager::GetTrustAccount(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts) {
        auto eg = NDrive::BuildEventGuard("FindTrustAccount");
        for (auto&& account : accounts) {
            if (account && account->GetType() == NDrive::NBilling::EAccount::Trust) {
                return account;
            }
        }
        return nullptr;
    }
}

