#pragma once


#include <drive/backend/billing/interfaces/account.h>
#include <drive/backend/billing/interfaces/account_link.h>


namespace NDrive::NBilling {
    struct TAccountParentFilter {
        TSet<TString> Accounts;
        TSet<ui64> ParentIds;
        bool WithParent = false;
        bool WithBonusAccounts = false;

        bool operator !() const {
            return Accounts.empty() && ParentIds.empty();
        }
    };


    class TAccountsManager : public IAccountsContext, public IStartStopProcess {
    public:
        TAccountsManager(TDatabasePtr database, const ISettings& settings, const THistoryConfig& historyConfig);
        ~TAccountsManager();
        bool GetUserAccountDescriptions(const TString& userId, const TInstant lastInstant, TMap<ui32, TAccountDescriptionRecord>& result, TVector<TAccountData>* accData = nullptr) const;

        TMaybe<TVector<IBillingAccount::TPtr>> GetAccountsByName(const TString& accName, const TInstant lastInstant = TInstant::Zero()) const;
        IBillingAccount::TPtr GetAccountById(const ui32 accountId, const TInstant lastInstant = TInstant::Zero()) const;
        TMaybe<TVector<IBillingAccount::TPtr>> GetAccountsByIds(const TSet<ui32>& accountIds, const TInstant lastInstant = TInstant::Zero()) const;
        bool UpsertAccountDescription(const TAccountDescriptionRecord& description, const TString& userId, NDrive::TEntitySession& session) const;
        bool RemoveAccountDescription(const ui32 typeId, const TString& originatiorId, NDrive::TEntitySession& session) const;
        bool RegisterAccount(const TAccountRecord::TPtr& accountRecord, const TString& historyUser, NDrive::TEntitySession& session, ui32* newAccountId = nullptr) const;
        bool RemoveAccount(const ui64 accountId, const TString& originatiorId, NDrive::TEntitySession& session) const;
        TMaybe<TVector<TAccountData>> GetAccounts(const NSQL::TQueryOptions& options, NDrive::TEntitySession& session) const;

        TExpected<TVector<IBillingAccount::TPtr>, TString> GetBillingAccounts(const TVector<TAccountData>& accountData, const TInstant lastInstant = TInstant::Zero()) const;
        TMaybe<TVector<IBillingAccount::TPtr>> GetBillingAccounts(const NSQL::TQueryOptions& options, NDrive::TEntitySession& session, const TString& userId = "fake") const;
        TMaybe<TVector<IBillingAccount::TPtr>> GetBillingAccounts(const TVector<TAccountData>& accounts, NDrive::TEntitySession& session, const TMap<ui64, TSet<TString>>& userIds = {}) const;
        TMaybe<TMap<ui64, IBillingAccount::TPtr>> GetBillingAccounts(const TSet<ui64>& accountIds, NDrive::TEntitySession& session, const TString& userId = "fake") const;

        IBillingAccount::TPtr LinkAccount(const TString& userId, const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const;
        bool UnLinkAccount(const TString& userId, const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const;
        bool ActivateAccounts(TConstArrayRef<ui32> accountIds, bool activeFlag, const TString& historyUser, NDrive::TEntitySession& session) const;

        TVector<TAccountDescriptionRecord> GetRegisteredAccounts(const TInstant lastInstant = TInstant::Zero()) const;
        TVector<NDrive::NBilling::TAccountDescriptionRecord> GetRegisteredAccounts(const TSet<ui32>& typeIds, const TInstant lastInstant = TInstant::Zero()) const;

        IBillingAccount::TPtr GetOrCreateAccount(const TString& userId, const TAccountDescriptionRecord& description, const TString& historyUser, NDrive::TEntitySession& session) const;
        IBillingAccount::TPtr GetOrCreateTrustAccount(const TString& userId, const TString& historyUser, NDrive::TEntitySession& session) const;

        virtual TMaybe<TAccountDescriptionRecord> GetDescriptionById(ui32 typeId, TInstant lastInstant = TInstant::Zero()) const override;
        TMaybe<TAccountDescriptionRecord> GetDescriptionByName(const TString& name, TInstant lastInstant = TInstant::Zero()) const;

        virtual bool DoStart() override;
        virtual bool DoStop() override;

        NJson::TJsonValue GetScheme(const NDrive::IServer* server) const;

        const ISettings& GetSettings() const {
            return Settings;
        }

        TDatabasePtr GetDatabase() const override {
            return Database;
        }

        TMap<TString, ui64> GetAccountsChildren(const TSet<ui64>& parentIds, TInstant lastInstant = TInstant::Zero()) const;
        TMaybe<TMap<TString, ui64>> GetAccountsChildren(const TAccountParentFilter& parentFilter, TInstant lastInstant = TInstant::Zero()) const;

        const TAccountsHistoryManager& GetAccountsHistory() const;
        const NDrive::NBilling::TAccountDescriptionHistory& GetAccountsDescriptionHistory() const;
        const TAccountsLinksDB& GetLinksManager() const;
        const TAccountsDescriptionDB& GetDescriptionDB() const;

        // user options
        NDrive::NBilling::TLimitedBalances GetLimitedBonuses(const TString& userId, const TInstant lastInstant = TInstant::Zero()) const;
        static NDrive::NBilling::TLimitedBalances GetLimitedBonuses(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts);

        NDrive::NBilling::IBillingAccount::TPtr GetTrustAccount(const TString& userId, const TInstant actuality = TInstant::Zero()) const;
        NDrive::NBilling::IBillingAccount::TPtr GetTrustAccount(const TString& userId, NDrive::TEntitySession& session) const;
        static NDrive::NBilling::IBillingAccount::TPtr GetTrustAccount(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts);

        ui32 GetBonuses(const TString& userId, const TInstant lastInstant = TInstant::Zero()) const;
        static ui32 GetBonuses(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts);

        TVector<IBillingAccount::TPtr> GetUserAccounts(const TString& userId, const TInstant lastInstant = TInstant::Zero()) const;
        TVector<IBillingAccount::TPtr> GetSortedUserAccounts(const TString& userId, const TInstant lastInstant = TInstant::Zero()) const;
        TMaybe<TVector<IBillingAccount::TPtr>> GetUserAccounts(const TString& userId, NDrive::TEntitySession& session) const;
        TMaybe<TVector<IBillingAccount::TPtr>> GetSortedUserAccounts(const TString& userId, NDrive::TEntitySession& session) const;
        TMaybe<TVector<IBillingAccount::TPtr>> GetUsersAccounts(const TSet<TString>& userIds, NDrive::TEntitySession& session) const;

    private:
        TDatabasePtr Database;
        mutable TAccountsDescriptionDB DescriptionsDB;
        mutable TAccountsStorage AccountsDB;
        mutable TAccountsLinksDB AccountLinksDB;
        const ISettings& Settings;
    };
}

