#include "charge_logic.h"

#include <drive/backend/billing/accounts/bonus.h>

IPaymentMeta::TFactory::TRegistrator<TBonusPaymentMeta> TBonusPaymentMeta::Registrator(NDrive::NBilling::EAccount::Bonus);
IPaymentMeta::TFactory::TRegistrator<TBonusPaymentMeta> TBonusPaymentMeta::RegistratorCoins(NDrive::NBilling::EAccount::Coins);

TString TBonusLogic::GetPaymentPrefix() const {
    return (GetType() == NDrive::NBilling::EAccount::Bonus) ? "b_" : "c_";
}

TChargeInfo TBonusLogic::CorrectCharge(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const {
    if (snapshot.IsClosed()) {
        return TChargeInfo(chargeInfo.Type);
    }

    if (chargeInfo.Type != EBillingType::CarUsage && chargeInfo.Type != EBillingType::Deposit) {
        return TChargeInfo(chargeInfo.Type);
    }

    ui32 maxSum = account->GetBalance();

    if (GetType() == NDrive::NBilling::EAccount::Bonus) {
        if (snapshot.GetSnapshot().GetBonusSum() >= snapshot.GetBillingTask().GetAvailableBonus()
                || account->GetBalance() <= 0)
        {
            return TChargeInfo(chargeInfo.Type);
        }
        maxSum = Min<ui32>(snapshot.GetBillingTask().GetAvailableBonus() - snapshot.GetSnapshot().GetBonusSum(), account->GetBalance());
    }

    TChargeInfo bonusCharge = CorrectChargeForBonus(snapshot, chargeInfo);
    bonusCharge.Sum = Min(bonusCharge.Sum, maxSum);
    return bonusCharge;
}

bool TBonusLogic::DoRefund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session, NStorage::TTableRecord& newPaymentRecord) const {
    if (account->GetType() != GetType()) {
        return false;
    }

    if (sum > payment.GetSum()) {
        return false;
    }

    ui32 commonSum = sum;
    if (account->IsLimitedPolicy()) {
        auto meta = dynamic_cast<const TBonusPaymentMeta*>(payment.GetMeta().Get());
        if (!meta) {
            return false;
        }
        ui32 limitedBonuses = 0;
        for (const auto& limit : meta->GetLimitedBonuses()) {
            limitedBonuses += limit.GetBalance();
        }

        if (payment.GetSum() < limitedBonuses) {
            ERROR_LOG << "Incorrect payment_task" << Endl;
            return false;
        }
        commonSum = Min<ui32>(payment.GetSum() - limitedBonuses, sum);

        NDrive::NBilling::TLimitedBalances newLimitedBonuses;
        ui32 currentLimitedSum = (limitedBonuses + commonSum) - sum;
        for (const auto& limit : meta->GetLimitedBonuses()) {
            if (currentLimitedSum > 0) {
                NDrive::NBilling::TLimitedBalance newLimit(limit);
                if (limit.GetBalance() > currentLimitedSum) {
                    newLimit.SetBalance(currentLimitedSum);
                    if (account->Add(limit.GetBalance() - currentLimitedSum, session, limit.GetDeadline(), limit.GetSource()) != EDriveOpResult::Ok) {
                        return false;
                    }
                    currentLimitedSum = 0;
                } else {
                    currentLimitedSum -= limit.GetBalance();
                }
                newLimitedBonuses.emplace(newLimit);
            } else {
                if (limit.GetBalance() && account->Add(limit.GetBalance(), session, limit.GetDeadline(), limit.GetSource()) != EDriveOpResult::Ok) {
                    return false;
                }
            }
        }
        newPaymentRecord.Set("meta", TBonusPaymentMeta(std::move(newLimitedBonuses)).ToJson());
    }
    if (commonSum != 0) {
        return account->Add(commonSum, session) == EDriveOpResult::Ok;
    }
    return true;
}

bool TBonusLogic::DoBuildPaymentTask(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsManager& /*manager*/, const TBillingTask& /*task*/, const TChargeInfo& charge, NStorage::TTableRecord& paymentRecord) const {
    auto limits = account->GetLimitedBalanceBySum(charge.Sum);
    if (limits.size()) {
        paymentRecord.Set("meta", TBonusPaymentMeta(std::move(limits)).ToJson());
    }
    return true;
}


void TBonusLogicConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* /*requestPolicy*/) {
    if (section->Name == ::ToString(NDrive::NBilling::EAccount::Coins)) {
        BonusType = NDrive::NBilling::EAccount::Coins;
    }
}

void TBonusLogicConfig::ToString(IOutputStream& os) const {
    Y_UNUSED(os);
}

IChargeLogic::TPtr TBonusLogicConfig::ConstructLogic(const ISettings& settings, TUsersDB& /*users*/, TAtomicSharedPtr<NDrive::INotifier> /*notifier*/) const {
    return new TBonusLogic(BonusType, settings);
}

ILogicConfig::TFactory::TRegistrator<TBonusLogicConfig> TBonusLogicConfig::Registrator(NDrive::NBilling::EAccount::Bonus);
ILogicConfig::TFactory::TRegistrator<TBonusLogicConfig> TBonusLogicConfig::RegistratorCoins(NDrive::NBilling::EAccount::Coins);
