#pragma once

#include "client.h"

#include <drive/library/cpp/trust/trust_cache.h>
#include <drive/backend/billing/interfaces/account.h>
#include <drive/backend/billing/interfaces/payments.h>

class TPaymentOpContext {
    R_FIELD(TDriveUserData, UserData, {});
    R_FIELD(TString, PassportUid);
    R_READONLY(TChargeInfo, Charge, TChargeInfo(EBillingType::CarUsage));
    R_READONLY(TBillingTask, BillingTask, {});
    TMaybe<NDrive::NTrustClient::TPayment> Payment;

public:
    TPaymentOpContext(const TDriveUserData& userData, const TChargeInfo& charge, const TBillingTask& task, const TString& passportUid)
        : UserData(userData)
        , PassportUid(passportUid)
        , Charge(charge)
        , BillingTask(task)
    {
    }

    void SetPayment(const NDrive::NTrustClient::TPayment& payment) {
        Payment = payment;
    }

    const NDrive::NTrustClient::TPayment& GetPayment() {
        CHECK_WITH_LOG(Payment.Defined());
        return Payment.GetRef();
    }
};

class ITrustLogic;

class TGuardedCallback : public TBillingCallback {
public:
    TGuardedCallback(const ITrustLogic& trustLogic, TBillingSyncGuard& guard);

    virtual ~TGuardedCallback() {
        Guard.UnRegisterObject();
    }

    virtual TString GetOperation() const = 0;

    void Notify(const TString& message) const;

    virtual TBillingSyncGuard* GetGuard() override {
        return &Guard;
    }

protected:
    TBillingSyncGuard& Guard;
    const ITrustLogic& TrustLogic;
};

class TLogicCallback : public TBillingCallback {
public:
    TLogicCallback(ITrustLogic& trustLogic);

    virtual ~TLogicCallback();

    virtual TString GetOperation() const = 0;

    virtual void OnRequestStart();

    void Notify(const TString& message) const;

protected:
    bool Started = false;
    ITrustLogic& TrustLogic;
};

template <class TBase>
class TStartPaymentCallback: public TBase {
public:
    template <class... TArgs>
    TStartPaymentCallback(ITrustLogic& trustLogic, const TPaymentTask& paymentTask, const TPaymentsManager& manager, bool updateBillingTask, TArgs&... args)
        : TBase(trustLogic, args...)
        , PaymentTask(paymentTask)
        , Owner(manager)
        , UpdateBillingTask(updateBillingTask)
    {}

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& /*client*/) override {
        auto syncGuard = this->GetGuard();
        if (isOk) {
            TString orderId = json["orders"][0]["order_id"].GetString();
            Owner.MarkStarted(orderId, PaymentTask.GetPaymentId(), UpdateBillingTask);
        } else {
            if (syncGuard) {
                syncGuard->SetResult(EProduceResult::Error);
            }
            auto session = Owner.BuildSession(false);
            if (UpdateBillingTask && (Owner.UpdateTaskStatus(PaymentTask.GetSessionId(), EPaymentStatus::CreationError, session) != EDriveOpResult::Ok || !session.Commit())) {
                if (syncGuard) {
                    syncGuard->SetResult(EProduceResult::TransactionError);
                }
            }
            this->Notify(PaymentTask.GetSessionId() + "/card:" + PaymentTask.GetPayMethod() + "/" + ToString(PaymentTask.GetSum()) + "/" + json.GetStringRobust());
        }
    }

    virtual TString GetOperation() const override {
        return "start_payment";
    }

private:
    TPaymentTask PaymentTask;
    const TPaymentsManager& Owner;
    bool UpdateBillingTask;
};

class TTrustPaymentMeta : public IPaymentMeta {
    R_READONLY(ui32, CashbackPercent, 0);
    R_OPTIONAL(NJson::TJsonValue, ProcessingInfo);
public:
    TTrustPaymentMeta() = default;
    TTrustPaymentMeta(const ui32 cashbackPercent)
        : IPaymentMeta()
        , CashbackPercent(cashbackPercent)
    {}
    virtual bool Parse(const NJson::TJsonValue& json) override {
        return NJson::ParseField(json, "cashback_percent", CashbackPercent, false)
            && NJson::ParseField(json, "processing_info", ProcessingInfo);
    }
    virtual NJson::TJsonValue ToJson() const override {
        NJson::TJsonValue result;
        NJson::InsertNonNull(result, "cashback_percent", CashbackPercent);
        NJson::InsertNonNull(result, "processing_info", ProcessingInfo);
        return result;
    }
    static TFactory::TRegistrator<TTrustPaymentMeta> Registrator;
    static TFactory::TRegistrator<TTrustPaymentMeta> RegistratorMobilePayment;
};

class TYCashbackPaymentMeta : public IPaymentMeta {
    R_READONLY(NJson::TJsonValue, Payload);
public:
    TYCashbackPaymentMeta() = default;
    TYCashbackPaymentMeta(const NJson::TJsonValue& payload)
        : IPaymentMeta()
        , Payload(payload)
    {}
    virtual bool Parse(const NJson::TJsonValue& json) override {
        return NJson::ParseField(json["payload"], Payload);
    }
    virtual NJson::TJsonValue ToJson() const override {
        return NJson::TMapBuilder("payload", Payload);
    }
    static TFactory::TRegistrator<TYCashbackPaymentMeta> Registrator;
};

template <class TBase>
class TCreatePaymentCallback: public TBase {
public:
    template <class... TArgs>
    TCreatePaymentCallback(ITrustLogic& trustLogic, const TPaymentOpContext& context, NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsManager& manager, TArgs&... args)
        : TBase(trustLogic, args...)
        , Context(context)
        , Account(account)
        , Owner(manager)
    {
    }

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& client) override {
        const NDrive::NTrustClient::TPayment& payment = Context.GetPayment();
        const TBillingTask& bTask = Context.GetBillingTask();
        auto syncGuard = this->GetGuard();
        if (isOk) {
            TString paymentId = json["purchase_token"].GetString();
            if (paymentId.empty()) {
                this->Notify(bTask.GetId() + "/card:" + payment.PaymethodId + "/" + ::ToString(Context.GetCharge().Sum) + "/" + json.GetStringRobust());
                return;
            }
            ui64 ts = ModelingNow().Seconds();

            NStorage::TTableRecord record;
            record.Set("sum", Context.GetPayment().Amount);
            record.Set("cleared", 0);
            record.Set("billing_type", ToString(Context.GetCharge().Type));
            record.Set("payment_type", ::ToString(GetPaymentType()));
            record.Set("last_update_ts", ts);
            record.Set("created_at_ts", ts);
            record.Set("session_id", Context.GetBillingTask().GetId());
            record.Set("status", NDrive::NTrustClient::EPaymentStatus::NotStarted);
            record.Set("pay_method", Context.GetPayment().PaymethodId);
            record.Set("payment_id", paymentId);
            if (Account) {
                record.Set("account_id", Account->GetId());
            }
            if ((GetPaymentType() == NDrive::NBilling::EAccount::Trust || GetPaymentType() == NDrive::NBilling::EAccount::MobilePayment) && bTask.GetCashbackPercent() > 0) {
                record.Set("meta", TTrustPaymentMeta(bTask.GetCashbackPercent()).ToJson());
            }
            if (GetPaymentType() == NDrive::NBilling::EAccount::YCashback) {
                record.Set("meta", TYCashbackPaymentMeta(payment.Payload ? payment.Payload->ToJson() : NJson::TJsonValue()).ToJson());
            }
            record.Set("user_id", Context.GetUserData().GetUserId());

            TPaymentTask paymentTask;
            auto tx = Owner.BuildSession(false);
            if (Owner.RegisterPayment(record, paymentTask, tx, Context.GetCharge().PaymentTaskId) == EDriveOpResult::Ok && tx.Commit()) {
                THolder<TBillingCallback> callback;
                if (syncGuard) {
                    callback = MakeHolder<TStartPaymentCallback<TGuardedCallback>>(const_cast<ITrustLogic&>(this->TrustLogic), paymentTask, Owner, true, *syncGuard);
                } else {
                    callback = MakeHolder<TStartPaymentCallback<TLogicCallback>>(const_cast<ITrustLogic&>(this->TrustLogic), paymentTask, Owner, true);
                }

                TBillingClient::TOperation operation(ETrustOperatinType::PaymentStart, callback.Release());
                operation.SetPaymentId(paymentId).SetBillingType(bTask.GetBillingType());

                if (syncGuard) {
                    client.RunOperation(operation);
                } else {
                    TBase::TrustLogic.AddOperation(std::move(operation));
                }
            }
        } else {
            if (syncGuard) {
                syncGuard->SetResult(EProduceResult::Error);
            }
            auto session = Owner.BuildSession(false);
            TMaybe<EPaymentStatus> newStatus;
            if (!bTask.IsDebt()) {
                newStatus = EPaymentStatus::CreationError;
            }
            if (Owner.UpdateTaskStatus(bTask.GetId(), newStatus, session) != EDriveOpResult::Ok || !session.Commit()) {
                if (syncGuard) {
                    syncGuard->SetResult(EProduceResult::TransactionError);
                }
            }
            this->Notify(bTask.GetId() + "/card:" + payment.PaymethodId + "/" + ::ToString(payment.Amount) + "/" + json.GetStringRobust());
        }
    }

    virtual TString GetOperation() const override {
        return "create_payment";
    }

    virtual NDrive::NBilling::EAccount GetPaymentType() const {
        if (Account) {
            return Account->GetType();
        }
        return NDrive::NBilling::EAccount::Fake;
    }

    virtual ETrustOperatinType GetStartOperation() const {
        return ETrustOperatinType::PaymentStart;
    }

private:
    TPaymentOpContext Context;
    NDrive::NBilling::IBillingAccount::TPtr Account;
    const TPaymentsManager& Owner;
};

struct TBillSyncParams {
    bool UpdateBillingTask = false;
    TMaybe<ui64> LastPaymentId;
    TBillingTask BillingTask;
};

template <class TBase>
class TSyncPaymentCallback: public TBase {
public:
    template <class... TArgs>
    TSyncPaymentCallback(ITrustLogic& trustLogic, const TPaymentTask& paymentTask, const TPaymentsManager& manager, const TString& userId, NDrive::ITrustStorage* cache, const TBillSyncParams& billSyncParams, TArgs&... args)
        : TBase(trustLogic, args...)
        , PaymentTask(paymentTask)
        , BillSyncParams(billSyncParams)
        , Owner(manager)
        , UserId(userId)
        , Cache(cache)
    {}

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& /*client*/) override {
        if (isOk) {
            NDrive::NTrustClient::TPayment result;
            result.FromJson(json);
            TUnistatSignalsCache::SignalAdd("billing-payments-statuses", ::ToString(result.PaymentStatus), 1);
            Y_UNUSED(UpdatePayment(PaymentTask, result));
            if (Cache) {
                Cache->UpdateValue(UserId);
            }
        } else {
            TBase::Notify(json.GetStringRobust());
        }
    }

    virtual TString GetOperation() const override {
        return "sync_payment";
    }

private:
    bool UpdatePayment(const TPaymentTask& lastTask, NDrive::NTrustClient::TPayment payment) const {
        CHECK_WITH_LOG(lastTask.GetPaymentId() == payment.PurchaseToken);

        if (!TBillingGlobals::AuthStatuses.contains(payment.PaymentStatus) && lastTask.GetStatus() == payment.PaymentStatus) {
            if (lastTask.GetLastUpdate() && lastTask.GetLastUpdate() + TBase::TrustLogic.GetOperationTimeout() < Now()) {
                payment.PaymentStatus = NDrive::NTrustClient::EPaymentStatus::NotAuthorized;
                payment.PaymentError = "internal_operation_timeout";
            } else {
                return true;
            }
        }

        auto session = Owner.BuildSession(false);
        auto table = session->GetDatabase().GetTable("drive_payments");

        NStorage::TTableRecord update;
        update.Set("status", payment.PaymentStatus);
        update.Set("rrn", payment.RRN);
        update.Set("card_mask", payment.CardMask);
        update.Set("last_update_ts", ModelingNow().Seconds());
        update.Set("id", "nextval('drive_payments_id_seq')");

        if (payment.ProcessingInfo && (lastTask.GetPaymentType() == NDrive::NBilling::EAccount::Trust || lastTask.GetPaymentType() == NDrive::NBilling::EAccount::MobilePayment)) {
            auto meta = std::dynamic_pointer_cast<TTrustPaymentMeta>(lastTask.GetMeta());
            if (!lastTask.GetMeta()) {
                meta = MakeAtomicShared<TTrustPaymentMeta>((ui32)0);
            }
            if (meta) {
                meta->SetProcessingInfo(*payment.ProcessingInfo);
                update.Set("meta", meta->ToJson());
            }
        }

        if (payment.PaymentStatus == NDrive::NTrustClient::EPaymentStatus::NotAuthorized) {
            update.Set("payment_error", payment.PaymentError);
            update.Set("payment_error_desc", payment.PaymentErrorDescription);
        }

        if (payment.PaymentStatus == NDrive::NTrustClient::EPaymentStatus::Authorized) {
            auto holdsTable = session->GetDatabase().GetTable("clearing_tasks");

            NStorage::TTableRecord holdRecord;
            holdRecord.Set("payment_id", lastTask.GetPaymentId());
            holdRecord.Set("session_id", lastTask.GetSessionId());
            holdRecord.Set("order_id", payment.OrderId);
            holdRecord.Set("created_at_ts", ModelingNow().Seconds());
            holdRecord.Set("sum", lastTask.GetSum());
            holdRecord.Set("refund", 0);
            holdRecord.Set("billing_type", lastTask.GetBillingType());

            NStorage::TTableRecord unique;
            unique.Set("payment_id", lastTask.GetPaymentId());

            auto result = holdsTable->AddIfNotExists(holdRecord, session.GetTransaction(), unique);
            if (!result || !result->IsSucceed()) {
                return false;
            }
            TUnistatSignalsCache::SignalAdd("drive-frontend-billing", "authorized", lastTask.GetSum());
        }

        if (payment.PaymentStatus == NDrive::NTrustClient::EPaymentStatus::Cleared) {
            update.Set("cleared", payment.Cleared);
        }

        NStorage::TTableRecord condition;
        condition.Set("payment_id", lastTask.GetPaymentId());
        TRecordsSet affected;
        auto queryResult = table->UpdateRow(condition, update, session.GetTransaction(), &affected);
        if (!queryResult || !queryResult->IsSucceed()) {
            return false;
        }

        if (affected.GetRecords().size() == 1) {
            TPaymentTask newTask;
            CHECK_WITH_LOG(newTask.DeserializeFromTableRecord(affected.GetRecords().front(), nullptr));
            auto ctx = session.template GetContextAs<TBillingSessionContext>();
            CHECK_WITH_LOG(ctx);
            ctx->SetVersion(newTask.GetSessionId(), newTask.GetId());

            TUnistatSignalsCache::SignalHistogram("billing-trust-processing", "time", (newTask.GetLastUpdate() - newTask.GetCreatedAt()).MilliSeconds(), NRTLineHistogramSignals::IntervalsTrustProcessing);

            if (!BillSyncParams.UpdateBillingTask) {
                return session.Commit();
            }
            TMaybe<EPaymentStatus> taskStatus;
            if (newTask.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Authorized || newTask.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared) {
                taskStatus = EPaymentStatus::Ok;
            }
            if (newTask.GetStatus() == NDrive::NTrustClient::EPaymentStatus::NotAuthorized) {
                if (TBase::TrustLogic.GetNoFundsStatuses().contains(newTask.GetPaymentError())) {
                    taskStatus = EPaymentStatus::NoFunds;
                } else if (!BillSyncParams.BillingTask.IsDebt()) {
                    taskStatus = EPaymentStatus::InternalError;
                }
            }

            return Owner.UpdatePaymentStatus(newTask, taskStatus, session, BillSyncParams.LastPaymentId) == EDriveOpResult::Ok && session.Commit();
        }
        return false;
    }

private:
    TPaymentTask PaymentTask;
    const TBillSyncParams BillSyncParams;
    const TPaymentsManager& Owner;
    const TString UserId;
    NDrive::ITrustStorage* Cache;
};
