#include "charge_logic.h"

void ITrustLogicConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
    ClientConfig.Init(section, requestPolicy);
}

void ITrustLogicConfig::ToString(IOutputStream& os) const {
    ClientConfig.ToString(os);
}

ITrustLogic::ITrustLogic(const ITrustLogicConfig& trustConfig, const ISettings& settings, TUsersDB& usersDB, TAtomicSharedPtr<NDrive::INotifier> notifier)
    : IChargeLogic(trustConfig.GetType(), settings)
    , TrustClient(trustConfig.GetClientConfig(), usersDB.GetDatabase())
    , UsersDB(usersDB)
    , MaxReqInFly(trustConfig.GetMaxInFly())
    , Notifier(notifier)
{}

EProduceResult ITrustLogic::SyncPayments(const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager, const NDrive::NBilling::TAccountsManager& /*accountsManager*/, NDrive::ITrustStorage* cache, const bool sync) const {
    return SyncPaymentsImpl(snapshot, paymentsManager, ETrustOperatinType::PaymentStart, ETrustOperatinType::PaymentInfo, cache, sync);
}

EProduceResult ITrustLogic::SyncPaymentsImpl(const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager, const ETrustOperatinType startOperation, const ETrustOperatinType syncOperation, NDrive::ITrustStorage* cache, const bool sync) const {
    if (!snapshot.GetSnapshot().WaitAnyPayments()) {
        return EProduceResult::Ok;
    }

    const TPaymentTask* notFinished = snapshot.GetSnapshot().GetProcessingPayment();
    CHECK_WITH_LOG(!!notFinished);
    TBillSyncParams params = {/* UpdateBillingTask = */ true, snapshot.GetBillingTask().GetLastPaymentId(), snapshot.GetBillingTask() };
    return SyncPaymentTask(*notFinished, snapshot.GetBillingTask().GetUserId(), paymentsManager, startOperation, syncOperation, cache, sync, params);
}

EProduceResult ITrustLogic::SyncPaymentTask(const TPaymentTask& task, const TString& userId, const TPaymentsManager& paymentsManager, const ETrustOperatinType startOperation, const ETrustOperatinType syncOperation, NDrive::ITrustStorage* cache, const bool sync, const TBillSyncParams& billSyncParams) const {
    if (task.GetPaymentType() != GetType()) {
        return EProduceResult::Ok;
    }

    auto result = EProduceResult::Wait;
    if (task.GetStatus() == NDrive::NTrustClient::EPaymentStatus::NotStarted) {
        if (sync) {
            TBillingSyncGuard g(result);
            TBillingClient::TOperation operation(startOperation, new TStartPaymentCallback<TGuardedCallback>(const_cast<ITrustLogic&>(*this), task, paymentsManager, billSyncParams.UpdateBillingTask, g));
            operation.SetPaymentId(task.GetPaymentId()).SetBillingType(task.GetBillingType());
            TrustClient.RunOperation(operation);
        } else {
            TBillingClient::TOperation operation(startOperation, new TStartPaymentCallback<TLogicCallback>(const_cast<ITrustLogic&>(*this), task, paymentsManager, billSyncParams.UpdateBillingTask));
            operation.SetPaymentId(task.GetPaymentId()).SetBillingType(task.GetBillingType());
            AddOperation(std::move(operation));
        }
        return result;
    }

    if (sync) {
        TBillingSyncGuard g(result);
        TBillingClient::TOperation operation(syncOperation, new TSyncPaymentCallback<TGuardedCallback>(const_cast<ITrustLogic&>(*this), task, paymentsManager, userId, cache, billSyncParams, g));
        operation.SetPaymentId(task.GetPaymentId()).SetBillingType(task.GetBillingType());
        TrustClient.RunOperation(operation);
    } else {
        TBillingClient::TOperation operation(syncOperation, new TSyncPaymentCallback<TLogicCallback>(const_cast<ITrustLogic&>(*this), task, paymentsManager, userId, cache, billSyncParams));
        operation.SetPaymentId(task.GetPaymentId()).SetBillingType(task.GetBillingType());
        AddOperation(std::move(operation));
    }
    return result;
}

bool ITrustLogic::DoRefund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session, NStorage::TTableRecord& /*newPaymentRecord*/) const {
    Y_UNUSED(account);
    Y_UNUSED(session);
    Y_UNUSED(payment);
    Y_UNUSED(sum);
    return false;
}

TString ITrustLogic::GetPassportUid(const TString& userId, const TDriveUserData* userData) const {
    if (userData) {
        return userData->GetPaymethodsUid(Settings);
    }
    auto gUser = UsersDB.FetchInfo(userId);
    auto data = gUser.GetResultPtr(userId);
    if (data) {
        return data->GetPaymethodsUid(Settings);
    }
    return TString();
}

TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> ITrustLogic::GetUserCards(const TString& userId) const {
    TString passportUid = GetPassportUid(userId);
    if (!passportUid) {
        WARNING_LOG << "User (" << userId << ") has empty uid to get user cards" << Endl;
        return TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>>();
    }

    TVector<NDrive::NTrustClient::TPaymentMethod> payMethods;
    auto product = TrustClient.GetTrustProduct(EBillingType::CarUsage);
    if (!product.Defined()) {
        return Nothing();
    }
    if (!TrustClient.GetPaymentMethods(product->GetTrustService(), passportUid, payMethods)) {
        return Nothing();
    }
    return payMethods;
}

void ITrustLogic::AddOperation(TBillingClient::TOperation&& context) const {
    TGuard<TMutex> g(OperationsLock);
    TrustOperations.emplace(std::forward<TBillingClient::TOperation>(context));
}

bool ITrustLogic::RunOperations(ui32 maxInFlight) {
    TGuard<TMutex> g(OperationsLock);
    if (TrustOperations.empty()) {
        return false;
    }
    for (ui32 i = 0; i < maxInFlight && !TrustOperations.empty(); ++i) {
        TrustClient.RunOperation(TrustOperations.front());
        TrustOperations.pop();
    }
    return true;
}

void ITrustLogic::WaitOperations() {
    TInstant start = TInstant::Now();
    ui32 maxInFlight = Settings.GetValueDef<ui32>("billing.trust_max_in_fly", MaxReqInFly);
    TDuration timeout = Settings.GetValueDef<TDuration>("billing.trust_waiting_timeout", TDuration::Seconds(600));
    if (maxInFlight) {
        RunOperations(maxInFlight);
    }

    while (TInstant::Now() < start + timeout) {
        if (!WaitAllTasks(maxInFlight ? maxInFlight - 1 : 0, start + timeout)) {
            break;
        }
        auto poolSize = PoolSize();
        TUnistatSignalsCache::SignalAdd("billing-cycle", "pool-size", poolSize);
        if (poolSize < maxInFlight) {
            auto run = RunOperations(maxInFlight - poolSize);
            if (!run && poolSize == 0) {
                break;
            }
        }
    }
    TUnistatSignalsCache::SignalAdd("billing-cycle", "wait", (TInstant::Now() - start).MilliSeconds());
}

void ITrustLogic::NotifyError(const TString& source, const TString& message) const {
    TStringStream ss;
    ss << "Trust Logic (" + source + "): " + message;
    if (Notifier) {
        if (Settings.GetValueDef<bool>("billing.telegram_errors", false, TInstant::Zero())) {
            Notifier->Notify(NDrive::INotifier::TMessage(ss.Str()));
        }
    } else {
        ERROR_LOG << ss.Str() << Endl;
    }
}
