#pragma once

#include "callback.h"
#include "client.h"

#include <drive/backend/billing/interfaces/logic.h>
#include <drive/backend/billing/interfaces/tasks.h>

#include <util/generic/queue.h>

class TBillingClient;
class TDriveUserData;

class ITrustLogicConfig : public ILogicConfig {
    R_READONLY(TBillingClientConfig, ClientConfig, {});
    R_READONLY(ui32, MaxInFly, 10);
public:
    virtual void Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* /*requestPolicy*/) override;
    virtual void ToString(IOutputStream& os) const override;
};

class ITrustLogic : public IChargeLogic, public TWaitGuard {
public:
    ITrustLogic(const ITrustLogicConfig& trustConfig, const ISettings& settings, TUsersDB& usersDB, TAtomicSharedPtr<NDrive::INotifier> notifier);
    virtual ~ITrustLogic() = default;

    const TBillingClient& GetTrustClient() const {
        return TrustClient;
    }

    virtual EProduceResult SyncPayments(const TPaymentsData& snapshot, const TPaymentsManager& manager, const NDrive::NBilling::TAccountsManager& accountsManager, NDrive::ITrustStorage* cache, const bool sync) const override;
    virtual bool DoRefund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session, NStorage::TTableRecord& /*newPaymentRecord*/) const override;

    TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> GetUserCards(const TString& userId) const;
    TString GetPassportUid(const TString& userId, const TDriveUserData* userData = nullptr) const;
    virtual TString GetPaymentPrefix() const override {
        return "";
    }

    void AddOperation(TBillingClient::TOperation&& context) const;
    virtual void WaitOperations() override;
    void NotifyError(const TString& source, const TString& message) const;

    EProduceResult SyncPaymentTask(const TPaymentTask& task, const TString& userId, const TPaymentsManager& manager, const ETrustOperatinType startOperation, const ETrustOperatinType syncOperation, NDrive::ITrustStorage* cache, const bool sync, const TBillSyncParams& billSyncParams) const;

    TSet<TString> GetNoFundsStatuses() const {
       return MakeSet(SplitString(Settings.GetValueDef<TString>("billing.no_funds_statuses", ""), ","));
    }

    TSet<TString> GetNoRetryStatuses() const {
       return MakeSet(SplitString(Settings.GetValueDef<TString>("billing.no_retry_statuses", ""), ","));
    }

protected:
    EProduceResult SyncPaymentsImpl(const TPaymentsData& snapshot, const TPaymentsManager& manager, const ETrustOperatinType startOperation, const ETrustOperatinType syncOperation, NDrive::ITrustStorage* cache, const bool sync) const;

private:
    bool RunOperations(ui32 maxInFlight);

protected:
    TBillingClient TrustClient;
    TUsersDB& UsersDB;
    TMutex Lock;

    mutable std::queue<TBillingClient::TOperation> TrustOperations;
    TMutex OperationsLock;
    ui32 MaxReqInFly;
    TAtomicSharedPtr<NDrive::INotifier> Notifier;
    R_FIELD(TDuration, OperationTimeout, TDuration::Hours(3));
};
