#include "client.h"

#include <drive/backend/logging/logging.h>

#include <rtline/library/json/builder.h>


bool ParseContent(const TString& reply, NJson::TJsonValue& report) {
    if (!NJson::ReadJsonFastTree(reply, &report)) {
        report["error"] = "Fail to parse content";
        report["content"] = reply;
        return false;
    }
    TString status;
    return report.Has("status") && report["status"].GetString(&status) && status == "success";
}

class TBillingCallbackWrapper: public NNeh::THttpAsyncReport::THandlerCallback {
private:
    using TBase = NNeh::THttpAsyncReport::THandlerCallback;

public:
    TBillingCallbackWrapper(ETrustOperatinType operation, IThreadPool& handler, TBillingCallback* cb, const TTrustAccessLogger& logger, const TBillingClient& client)
        : TBase(handler)
        , Client(client)
        , CallBack(cb)
        , Operation(operation)
        , Logger(logger)
    {}

    void Process(void*) final {
        CHECK_WITH_LOG(Reports.size() == 1);
        SetResult(Reports.front());
    }

    void SetResult(const THttpReplyData<TString>& reply) {
        NJson::TJsonValue report(NJson::JSON_NULL);
        const bool isOk = reply.HasReport() && ParseContent(*reply.GetReport(), report);
        SetResult(report, isOk, reply.GetHttpCode());
    }

    void SetResult(const NJson::TJsonValue& report, const bool isOk, const ui32 code) {
        Logger.LogRequest(Operation, Now() - StartTime, isOk, (HttpCodes)code);
        CallBack->OnResult(report, isOk, code, Client);
    }

private:
    const TBillingClient& Client;
    THolder<TBillingCallback> CallBack;
    ETrustOperatinType Operation;
    const TTrustAccessLogger& Logger;
    using TBase::StartTime;
};

TBillingClient::TBillingClient(const TBillingClientConfig& config, TDatabasePtr database)
    : Config(config)
    , Impl(Config, "trust_api")
    , HContext(database)
    , VirtualTerminals(HContext)
    , RepliesHandler(IThreadPool::TParams()
        .SetThreadName("billing_client")
    )
{
    RepliesHandler.Start(Config.GetThreads());
    Y_ENSURE_BT(VirtualTerminals.Start());
}

TBillingClient::~TBillingClient() {
    if (!VirtualTerminals.Stop()) {
        ERROR_LOG << "cannot stop VirtualTerminals manager" << Endl;
    }
}

TBillingClient::TGuard TBillingClient::RunOperation(TOperation& context) const {
    TString terminal;
    if (context.HasBillingType()) {
        auto product = GetTrustProduct(context.GetBillingTypeUnsafe());
        if (product.Defined()) {
            terminal = product->GetTrustService();
        } else {
            ERROR_LOG << "Incorrect terminal for " << context.GetBillingTypeUnsafe() << Endl;
            return {};
        }
    }

    TBillingCallback* callback = context.ReleaseCallback();
    CHECK_WITH_LOG(callback) << "No callback for " << context.GetType();
    callback->OnRequestStart();

    switch (context.GetType()) {
    case ETrustOperatinType::PaymentInfo:
        return GetPayment(terminal, context.GetPaymentId(), callback, false, true);
    case ETrustOperatinType::Paymethods:
        if (context.GetDuplicatePassIds().empty()) {
            return GetPaymentMethods(terminal, context.GetPassportId(), callback);
        }
        GetPaymentMethods(terminal, context.GetDuplicatePassIds(), callback);
        return TBillingClient::TGuard();
    case ETrustOperatinType::PaymentStart:
        return StartPayment(terminal, context.GetPaymentId(), callback);
    case ETrustOperatinType::PaymentCreate:
        return CreatePayment(terminal, context.GetPassportId(), context.GetPayment(), callback);
    case ETrustOperatinType::YAccountCreate:
        return CreateYAccount(terminal, context.GetPassportId(), context.GetPayment(), callback);
    case ETrustOperatinType::TopupCreate:
        return CreateTopup(terminal, context.GetPassportId(), context.GetPayment(), callback);
    case ETrustOperatinType::TopupInfo:
        return TopupInfo(terminal, context.GetPaymentId(), callback);
    case ETrustOperatinType::TopupStart:
        return StartTopup(terminal, context.GetPaymentId(), callback);
    default:
        FAIL_LOG("Not implemented");
    }
    return {};
}

void TBillingClient::GetPaymentMethods(const TString& terminal, const TVector<TString>& duplicatePassIds, TBillingCallback* callback) const {
    if (duplicatePassIds.empty()) {
        ERROR_LOG << "GetPaymentMethods: empty passport ids" << Endl;
        return;
    }

    NThreading::TFutures<NUtil::THttpReply> replies;
    {
        const TString& url = Config.GetPaymentMethodsPath();
        const TDuration timeout = Config.GetRequestTimeout();
        for (const auto& pass : duplicatePassIds) {
            replies.push_back(Impl->SendAsync(CreateRequest(terminal, pass).SetUri(url), Now() + timeout));
        }
    }

    auto callbackPtr = MakeAtomicShared<TBillingCallbackWrapper>(ETrustOperatinType::Paymethods, RepliesHandler, callback, Logger, *this);
    auto finalize = [callbackPtr, replies, passIds = duplicatePassIds](const NThreading::TFuture<void>& w) mutable {
        if (!callbackPtr) {
            ERROR_LOG << "GetPaymentMethods: inconsistent callback" << Endl;
            return;
        }
        ui32 code = HTTP_INTERNAL_SERVER_ERROR;
        NJson::TJsonValue report(NJson::JSON_ARRAY);
        if (!w.HasValue()) {
            const TString error = NThreading::GetExceptionMessage(w);
            ERROR_LOG << "GetPaymentMethods: failed reply: " << error << Endl;
            callbackPtr->SetResult(NJson::TMapBuilder("error", error), /* isOk = */ false, code);
            return;
        }
        if (replies.size() != passIds.size()) {
            const TString error = "wrong replies size " + ToString(replies.size()) + " expected " + ToString(passIds.size());
            ERROR_LOG << "GetPaymentMethods: " << error << Endl;
            callbackPtr->SetResult(NJson::TMapBuilder("error", error), /* isOk = */ false, code);
            return;
        }
        auto passIt = passIds.begin();
        for (auto it = replies.begin(); it != replies.end(); ++it) {
            NJson::TJsonValue samePerson;
            if (!it->HasValue()) {
                const TString error = NThreading::GetExceptionMessage(*it);
                ERROR_LOG << "GetPaymentMethods: failed reply: " << error << Endl;
                callbackPtr->SetResult(NJson::TMapBuilder("error", error), /* isOk = */ false, HTTP_INTERNAL_SERVER_ERROR);
                return;
            }
            NNeh::THttpReply reply = it->GetValue();
            code = reply.Code();
            if (!ParseContent(reply.Content(), samePerson)) {
                callbackPtr->SetResult(samePerson, /* isOk = */ false, code);
                return;
            }
            for (auto&& item : samePerson["bound_payment_methods"].GetArray()) {
                auto method = item;
                method["request_passport_id"] = *passIt;
                report.AppendValue(std::move(method));
            }
            ++passIt;
        }
        callbackPtr->SetResult(report, /* isOk = */ true, code);
    };
    NThreading::WaitExceptionOrAll(replies).Subscribe(finalize);
}

TBillingClient::TGuard TBillingClient::GetPaymentMethods(const TString& terminal, const TString& passportId, TBillingCallback* callback) const {
    const TString url = Config.GetPaymentMethodsPath();
    auto req = CreateRequest(terminal, passportId);
    req.SetUri(url);
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::Paymethods, RepliesHandler, callback, Logger, *this));
}

bool TBillingClient::GetPaymentMethods(const TString& terminal, const TString& passportId, TVector<NDrive::NTrustClient::TPaymentMethod>& results) const {
    auto req = CreateRequest(terminal, passportId);
    req.SetUri(Config.GetPaymentMethodsPath());
    TInstant startTs = Now();
    const TInstant deadline = startTs + Config.GetRequestTimeout();
    NUtil::THttpReply result = Impl->SendMessageSync(req, deadline);
    DEBUG_LOG << result.Content() << Endl;
    NJson::TJsonValue json;
    if (result.Code() != HTTP_OK || !NJson::ReadJsonFastTree(result.Content(), &json)) {
        Logger.LogRequest(ETrustOperatinType::Paymethods, Now() - startTs, false, (HttpCodes)result.Code());
        return false;
    }
    Logger.LogRequest(ETrustOperatinType::Paymethods, Now() - startTs, true, (HttpCodes)result.Code());

    if (json["status"].GetString() == "success") {
        for (auto&& payment : json["bound_payment_methods"].GetArray()) {
            NDrive::NTrustClient::TPaymentMethod method;
            method.FromJson(payment);
            results.push_back(method);
        }
        return true;
    }
    return false;
}

/*
TTrustOrder::TPtr TBillingClient::GetOrder(const TString& terminal, const TString& passportId, const TString& orderId)  const {
    auto req = CreateRequest(terminal, passportId);
    req.SetUri(Config.GetOrdersPath() + "/" + orderId);
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    TReplyData reply;
    THolder<TBillingCallbackWrapper> callback = MakeHolder<TBillingCallbackWrapper>(ETrustOperatinType::Paymethods, FakeHandler, new TReplyGetter(reply), Logger, *this);
    auto g = Impl->Send(req, deadline, callback.Release());
    g.Wait();

    if (reply.Success) {
        TTrustOrder::TPtr result(new TTrustOrder);
        result->FromJson(reply.Json);
        return result;
    }
    return nullptr;
}
*/

TBillingClient::TGuard TBillingClient::CreateOrder(const TString& terminal, const TString& passportId, const TTrustOrder& order, TBillingCallback* callback) {
    NJson::TJsonValue request;
    order.ToJson(request);

    auto req = CreateRequest(terminal, passportId, request);
    req.SetUri(Config.GetOrdersPath());
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::OrderCreate, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::CreatePayment(const TString& terminal, const TString& passportId, const NDrive::NTrustClient::TPayment& payment, TBillingCallback* callback) const {
    NJson::TJsonValue request;
    payment.ToJson(request);

    auto req = CreateRequest(terminal, passportId, request);
    req.SetUri(Config.GetPaymentPath());
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::PaymentCreate, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::GetPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback, bool sync, bool getProcessingInfo) const {
    auto req = CreateRequest(terminal, "");
    req.SetUri(Config.GetPaymentPath() + "/" + purchaseToken);
    if (getProcessingInfo) {
        req.AddCgiData("&show_processing_info=true");
    }
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    if (sync) {
        return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::PaymentInfo, FakeHandler, callback, Logger, *this));
    }
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::PaymentInfo, RepliesHandler, callback, Logger, *this));
}


TBillingClient::TGuard TBillingClient::StartPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, "", NJson::TJsonValue(NJson::JSON_MAP));
    req.SetUri(Config.GetPaymentPath() + "/" + purchaseToken + "/start");
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::PaymentStart, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::CancelPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, "", NJson::TJsonValue(NJson::JSON_MAP));
    req.SetUri(Config.GetPaymentPath() + "/" + purchaseToken + "/unhold");
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::PaymentCancel, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::ClearPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, "", NJson::TJsonValue(NJson::JSON_MAP));
    req.SetUri(Config.GetPaymentPath() + "/" + purchaseToken + "/clear");
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::Clearing, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::CreateRefund(const TString& terminal, const NDrive::NTrustClient::TRefund& refund, TBillingCallback* callback) const {
    NJson::TJsonValue request;
    refund.ToJson(request);
    auto req = CreateRequest(terminal, "", request);
    req.SetUri(Config.GetRefundsPath());
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::RefundCreate, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::StartRefund(const TString& terminal,const TString& refundId, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, "", NJson::TJsonValue(NJson::JSON_MAP));
    req.SetUri(Config.GetRefundsPath() + "/" + refundId + "/start");
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::RefundStart, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::ResizePayment(const TString& terminal, const TString& purchaseToken, const TString& orderId, ui32 leftSum, TBillingCallback* callback) const {
    NJson::TJsonValue request;
    request["amount"] = leftSum * 0.01;
    request["qty"] = 1;
    request["data"] = "";
    auto req = CreateRequest(terminal, "", request);
    req.SetUri(Config.GetPaymentPath() + "/" + purchaseToken + "/orders/" + orderId + "/resize");
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::Resize, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::GetRefund(const TString& terminal, const TString& refundId, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, "");
    req.SetUri(Config.GetRefundsPath() + "/" + refundId);
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::RefundInfo, RepliesHandler, callback, Logger, *this));
}

TMaybe<TVirtualTerminal> TBillingClient::GetTrustProduct(EBillingType type, const TInstant& actuality) const {
    return VirtualTerminals.GetTerminal(type, actuality);
}

TBillingClient::TGuard TBillingClient::CreateTopup(const TString& terminal, const TString& passportId, const NDrive::NTrustClient::TPayment& payment, TBillingCallback* callback) const {
    NJson::TJsonValue request;
    payment.ToJson(request);

    auto req = CreateRequest(terminal, passportId, request);
    req.SetUri(Config.GetTopupPath());
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::TopupCreate, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::TopupInfo(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback, bool sync) const {
    auto req = CreateRequest(terminal, "");
    req.SetUri(Config.GetTopupPath() + "/" + purchaseToken);
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    if (sync) {
        return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::TopupInfo, FakeHandler, callback, Logger, *this));
    }
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::TopupInfo, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::StartTopup(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, "", NJson::TJsonValue(NJson::JSON_MAP));
    req.SetUri(Config.GetTopupPath() + "/" + purchaseToken + "/start");
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::TopupStart, RepliesHandler, callback, Logger, *this));
}

TBillingClient::TGuard TBillingClient::CreateYAccount(const TString& terminal, const TString& passportId, const NDrive::NTrustClient::TPayment& payment, TBillingCallback* callback) const {
    auto req = CreateRequest(terminal, passportId, NJson::TMapBuilder("currency", payment.Currency));
    req.SetUri(Config.GetYAccountPath());
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->Send(req, deadline, new TBillingCallbackWrapper(ETrustOperatinType::YAccountCreate, RepliesHandler, callback, Logger, *this));
}
