#pragma once

#include "config.h"

#include <drive/backend/billing/interfaces/entities.h>
#include <drive/backend/billing/interfaces/signals.h>

#include <rtline/library/deprecated/async_impl/async_impl.h>
#include <rtline/library/unistat/signals.h>


enum class ETrustOperatinType {
    PaymentCreate = 0 /* "payment_create" */,
    PaymentStart = 1 /* "payment_start" */,
    Clearing = 2 /* "clearing" */,
    PaymentInfo = 3 /* "payment_info" */,
    RefundCreate = 4 /* "refund_create" */,
    RefundStart = 5 /* "refund_start" */,
    Paymethods = 6 /* "paymethods" */,
    Resize = 7 /* "resize" */,
    PaymentCancel = 8 /* "payment_cancel" */,
    RefundInfo = 9 /* "refund_info" */,
    OrderCreate = 10 /* "order_create" */,
    YAccountCreate = 11 /* "yaccount_create" */,
    TopupCreate = 12 /* "topup_create" */,
    TopupStart = 13 /* "topup_start" */,
    TopupInfo = 14 /* "topup_info" */,
};

struct TTrustOrder {
    using TPtr = TAtomicSharedPtr<TTrustOrder>;

    TTrustOrder() = default;

    void FromJson(const NJson::TJsonValue& json) {
        ProductId = json["product_id"].GetString();
        OrderId = json["order_id"].GetString();
    }

    void ToJson(NJson::TJsonValue& json) const {
        json["product_id"] = ProductId;
        json["order_id"] = OrderId;
    }

public:
    TString OrderId;
    TString ProductId;
};


class TTrustAccessLogger {
public:
    TTrustAccessLogger()
        : RequestsByType({ "drive-frontend-trust-requests" }, false)
        , ErrorsByType({ "drive-frontend-trust-errors" }, false)
        , TimesByType({ "drive-frontend-trust-times" }, NRTLineHistogramSignals::IntervalsRTLineReply)
        , TrustReplyCodes({ "drive-frontend-trust-reply-codes" }, false)
    {}

    void LogRequest(ETrustOperatinType requestType, TDuration duration, bool isOk, HttpCodes code) const {
        RequestsByType.Signal(requestType, 1);
        TrustReplyCodes.Signal(code, 1);
        TUnistatSignalsCache::SignalAdd("trust-reply-codes-" + ToString(requestType), ToString((ui32)code), 1);

        if (!isOk) {
            ErrorsByType.Signal(requestType, 1);
        }
        TimesByType.Signal(requestType, duration.MilliSeconds());
    }

private:
    TEnumSignal<ETrustOperatinType, double> RequestsByType;
    TEnumSignal<ETrustOperatinType, double> ErrorsByType;
    TEnumSignal<ETrustOperatinType, double> TimesByType;
    TEnumSignal<HttpCodes, double> TrustReplyCodes;
};

class TBillingClient;

class TBillingSyncGuard : public TWaitGuard {
public:
    TBillingSyncGuard(EProduceResult& result)
        : Result(result)
    {}

    void SetResult(EProduceResult result) {
        Result = result;
    }

private:
    EProduceResult& Result;
};

class TBillingCallback {
public:
    virtual ~TBillingCallback() {}
    virtual void OnResult(const NJson::TJsonValue& /*json*/, bool /*isOk*/, ui16 /*code*/, const TBillingClient& client) = 0;
    virtual void OnRequestStart() {}
    TBillingCallback() = default;

    virtual TBillingSyncGuard* GetGuard() {
        return nullptr;
    }

private:
    R_FIELD(TString, Request);
};

class TReplyGetter: public TBillingCallback {
public:
    TReplyGetter(NJson::TJsonValue& reply, bool& success)
        : Json(reply)
        , Success(success)
    {}

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& /*client*/) override {
        Json = json;
        Success = isOk;
    }

private:
    NJson::TJsonValue& Json;
    bool& Success;
};

class TBillingClient {
    using TGuard = NNeh::THttpClient::TGuard;
public:
    class TOperation {
    public:
        TBillingCallback* ReleaseCallback() {
            return Callback.Release();
        }

        TOperation(const ETrustOperatinType opType, TBillingCallback* callback)
            : Type(opType)
            , Callback(callback)
        {}

    private:
        R_READONLY(ETrustOperatinType, Type, ETrustOperatinType::PaymentInfo);
        R_FIELD(NDrive::NTrustClient::TPayment, Payment);
        R_FIELD(TString, PassportId);
        R_FIELD(TVector<TString>, DuplicatePassIds);
        R_FIELD(TString, PaymentId);
        R_OPTIONAL(EBillingType, BillingType);
        THolder<TBillingCallback> Callback;
    };

public:
    TBillingClient(const TBillingClientConfig& config, TDatabasePtr database);
    ~TBillingClient();

    TBillingClient::TGuard RunOperation(TOperation& context) const;

    TGuard GetPaymentMethods(const TString& terminal, const TString& passportId, TBillingCallback* callback) const;
    void GetPaymentMethods(const TString& terminal, const TVector<TString>& duplicatePassIds, TBillingCallback* callback) const;

    TGuard CreateOrder(const TString& terminal, const TString& passportId, const TTrustOrder& order, TBillingCallback* callback);
    TGuard CreatePayment(const TString& terminal, const TString& passportId, const NDrive::NTrustClient::TPayment& payment, TBillingCallback* callback) const;
    TGuard GetPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback, bool sync = false, bool getProcessingInfo = false) const;
    TGuard StartPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const;

    TGuard CancelPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const;
    TGuard ClearPayment(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const;
    TGuard ResizePayment(const TString& terminal, const TString& purchaseToken, const TString& orderId, ui32 leftSum, TBillingCallback* callback) const;

    TGuard CreateRefund(const TString& terminal, const NDrive::NTrustClient::TRefund& refund, TBillingCallback* callback) const;
    TGuard StartRefund(const TString& terminal, const TString& refundId, TBillingCallback* callback) const;
    TGuard GetRefund(const TString& terminal, const TString& refundId, TBillingCallback* callback) const;

    TGuard CreateTopup(const TString& terminal, const TString& passportId, const NDrive::NTrustClient::TPayment& payment, TBillingCallback* callback) const;
    TGuard TopupInfo(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback, bool sync = false) const;
    TGuard StartTopup(const TString& terminal, const TString& purchaseToken, TBillingCallback* callback) const;

    TGuard CreateYAccount(const TString& terminal, const TString& passportId, const NDrive::NTrustClient::TPayment& payment, TBillingCallback* callback) const;

    TMaybe<TVirtualTerminal> GetTrustProduct(EBillingType type, const TInstant& actuality = TInstant::Zero()) const;

    bool GetPaymentMethods(const TString& terminal, const TString& passportId, TVector<NDrive::NTrustClient::TPaymentMethod>& results) const;
//    TTrustOrder::TPtr GetOrder(const TString& terminal, const TString& passportId, const TString& orderId) const;
    const TVirtualTerminalsDB& GetTerminals() const {
        return VirtualTerminals;
    }

private:
    NNeh::THttpRequest CreateRequest(const TString& terminal, const TString& uid, const NJson::TJsonValue& postData = NJson::TJsonValue(NJson::JSON_NULL)) const {
        NNeh::THttpRequest req;
        req.AddHeader("X-Service-Token", terminal);

        if (!!uid) {
            req.AddHeader("X-Uid", uid);
        }

        if (postData.IsNull()) {
            req.SetRequestType("GET");
        } else {
            req.SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
        }
        return req;
    }

private:
    TBillingClientConfig Config;
    TSimpleAsyncRequestSender Impl;
    THistoryContext HContext;
    mutable TVirtualTerminalsDB VirtualTerminals;
    mutable TFakeThreadPool FakeHandler;
    mutable TThreadPool RepliesHandler;
    TTrustAccessLogger Logger;
};
