#include "config.h"

#include <library/cpp/mediator/global_notifications/system_status.h>

void TBillingConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
    UseDBJsonStatements = section->GetDirectives().Value("UseDBJsonStatements", UseDBJsonStatements);
    BatchUpdates = section->GetDirectives().Value("BatchUpdates", BatchUpdates);
    MinimalPayment = section->GetDirectives().Value("MinimalPayment", MinimalPayment);
    NotifierName = section->GetDirectives().Value("NotifierName", NotifierName);
    PaymentsCacheInterval = section->GetDirectives().Value<TDuration>("PaymentsCacheInterval", PaymentsCacheInterval);
    RefreshInterval = section->GetDirectives().Value<TDuration>("RefreshInterval", RefreshInterval);
    FailedTaskCheckInterval = section->GetDirectives().Value<TDuration>("FailedTaskCheckInterval", FailedTaskCheckInterval);
    ErrorTaskCheckInterval = section->GetDirectives().Value<TDuration>("ErrorTaskCheckInterval", ErrorTaskCheckInterval);
    MaxAutoCheckDebtDeep = section->GetDirectives().Value<TDuration>("MaxAutoCheckDebtDeep", MaxAutoCheckDebtDeep);
    ClearingTasksAtOnce = section->GetDirectives().Value<ui32>("ClearingTasksAtOnce", ClearingTasksAtOnce);
    UsePaymentsCache = section->GetDirectives().Value<bool>("UsePaymentsCache", UsePaymentsCache);

    auto sections = section->GetAllChildren();

    for (auto&& configSection : sections) {
        if (configSection.first == "AccountsHistoryConfig") {
            AccountsHistoryConfig.Init(configSection.second);
            continue;
        }

        if (configSection.first == "Logics") {
            TSet<TString> logics;
            TString modulesStr = configSection.second->GetDirectives().Value("Modules", TString());
            TVector<TString> modules = SplitString(modulesStr, ",");
            auto logicSections = configSection.second->GetAllChildren();
            for (auto&& logicItem : modules) {
                AssertCorrectConfig(logics.insert(logicItem).second, "Duplicate logic %s", logicItem.Data());
                NDrive::NBilling::EAccount logicName;
                AssertCorrectConfig(TryFromString(logicItem, logicName), "Unknown logic %s", logicItem.Data());
                ILogicConfig::TPtr config = ILogicConfig::TFactory::Construct(logicName);
                if (config) {
                    auto it = logicSections.find(logicItem);
                    if (it != logicSections.end()) {
                        config->Init(it->second, requestPolicy);
                    }
                    LogicConfigs.push_back(config);
                }
            }
        }

        if (configSection.first == "TrustStorage") {
            auto type = configSection.second->GetDirectives().Value("Type", TString());
            TrustStorageConfig.Reset(NDrive::ITrustStorageConfig::TFactory::Construct(type, type));
            AssertCorrectConfig(!!TrustStorageConfig, "incorrect trust storage type");
            TrustStorageConfig->Init(configSection.second);

            TString trustCacheStorageLogic;
            trustCacheStorageLogic = section->GetDirectives().Value("TrustCacheStorageLogic", ::ToString(TrustCacheStorageLogic));
            AssertCorrectConfig(TryFromString(trustCacheStorageLogic, TrustCacheStorageLogic), "incorrect cache logic");
        }
    }

    ActiveQueue = section->GetDirectives().Value("ActiveQueue", ActiveQueue);
    AssertCorrectConfig(MinimalPayment > 100, "MinimalPayment should be greater than 1 rub");
}

void TBillingConfig::ToString(IOutputStream& os) const {
    os << "BatchUpdates: " << BatchUpdates << Endl;
    os << "MinimalPayment: " << MinimalPayment << Endl;
    os << "PaymentsCacheInterval: " << PaymentsCacheInterval << Endl;
    os << "FailedTaskCheckInterval: " << FailedTaskCheckInterval << Endl;
    os << "ErrorTaskCheckInterval: " << ErrorTaskCheckInterval << Endl;
    os << "NotifierName: " << NotifierName << Endl;
    os << "MaxAutoCheckDebtDeep: " << MaxAutoCheckDebtDeep << Endl;
    os << "ClearingTasksAtOnce: " << ClearingTasksAtOnce << Endl;
    os << "ActiveQueue: " << ActiveQueue << Endl;
    os << "RefreshInterval: " << RefreshInterval << Endl;
    os << "UseDBJsonStatements: " << UseDBJsonStatements << Endl;
    os << "UsePaymentsCache: " << UsePaymentsCache << Endl;
    os << "<AccountsHistoryConfig>" << Endl;
    AccountsHistoryConfig.ToString(os);
    os << "</AccountsHistoryConfig>" << Endl;
    os << "<Logics>" << Endl;
    os << "Modules: " << Endl;
    for (auto&& logic : LogicConfigs) {
        os << logic->GetType() << ",";
    }
    os << Endl;
    for (auto&& logic : LogicConfigs) {
        os << "<" << logic->GetType() << ">" << Endl;
        logic->ToString(os);
        os << "</" << logic->GetType() << ">" << Endl;
    }
    os << "</Logics>" << Endl;

    if (TrustStorageConfig) {
        os << "<TrustStorage>" << Endl;
        TrustStorageConfig->ToString(os);
        os << "</TrustStorage>" << Endl;
        os << "TrustCacheStorageLogic: " << ::ToString(TrustCacheStorageLogic) << Endl;
    }
}
