#include "yt_nodes.h"

#include <rtline/util/types/uuid.h>
#include <drive/backend/billing/interfaces/payments.h>

TString NBillingExports::GetStringSum(ui32 sum, bool isRefund) {
    TString penny = (sum % 100 < 10) ? "0" + ::ToString(sum % 100) : ::ToString(sum % 100);
    if (isRefund && sum > 0) {
        return "-" + ::ToString(sum / 100) + "." + penny;
    }
    return ::ToString(sum / 100) + "." + penny;
}

TString NBillingExports::GetOrderType(EBillingType bType) {
    if (bType == EBillingType::CarUsage || bType == EBillingType::CarUsageAdditional || TBillingGlobals::CashbackBillingTypes.contains(bType) || bType == EBillingType::CarUsagePrepayment) {
        return "carsharing";
    }
    if (bType == EBillingType::Ticket) {
        return "fine";
    }
    if (bType == EBillingType::TicketGIBDD) {
        return "fine_gibdd";
    }
    if (bType == EBillingType::TollRoad) {
        return "toll_road";
    }
    if (bType == EBillingType::CarFuelling) {
        return "car_fuelling";
    }
    if (bType == EBillingType::CarParking) {
        return "car_parking";
    }
    if (bType == EBillingType::CarCleaning) {
        return "car_cleaning";
    }
    return "";
}

NYT::TNode NBillingExports::CreateTransactionNodeBase(const TString& orderId, const TString& clientId, const TString& type, const TString& paymentType, const TString& serviceOrderId) {
    NYT::TNode transactionRefundNode;
    transactionRefundNode["transaction_id"] = NUtil::CreateUUID();
    transactionRefundNode["orig_transaction_id"] = orderId;
    transactionRefundNode["service_order_id"] = serviceOrderId ? serviceOrderId : orderId;
    transactionRefundNode["client_id"] = clientId;
    transactionRefundNode["transaction_currency"] = "RUB";
    transactionRefundNode["payment_type"] = paymentType;
    transactionRefundNode["use_discount"] = 0;
    transactionRefundNode["type"] = type;
    return transactionRefundNode;
}

NYT::TNode NBillingExports::CreateRefundNode(const TString& orderId, const TString& serviceOrderId, const TString& clientId, ui32 refundSum, const TString& type, TInstant ts) {
    NYT::TNode transactionRefundNode = CreateTransactionNodeBase(orderId, clientId, type, "card", serviceOrderId);
    transactionRefundNode["dt"] = ToString(ts);
    transactionRefundNode["total_sum"] = GetStringSum(refundSum, true);
    transactionRefundNode["commission_sum"] = "0.0";
    transactionRefundNode["promocode_sum"] = "0.0";
    return transactionRefundNode;
}

NYT::TNode NBillingExports::CreateTransactionNode(const TString& sessionId, const TString& clientId, ui32 totalSum, ui32 bonusSum, const TInstant& ts, const TString& type,
    const bool isRefund, const TString& externalId /*= ""*/, const TString& paymentType /*= "card"*/) {
    NYT::TNode resultNode = CreateTransactionNodeBase(sessionId, clientId, type, paymentType);
    if (externalId) {
        resultNode["transaction_id"] = externalId;
    }
    resultNode["dt"] = ToString(ts);
    resultNode["total_sum"] = GetStringSum(totalSum + bonusSum, isRefund);
    resultNode["commission_sum"] = "0.0";
    resultNode["promocode_sum"] = GetStringSum(bonusSum, isRefund);
    return resultNode;
}

NYT::TNode NBillingExports::CreateB2BTransactionNode(const TString& sessionId, const TString& clientId, ui32 totalSum, ui32 bonusSum, const TInstant& ts, const TString& type,
    const bool isRefund, const TString& externalId /*= ""*/, const TString& paymentType /*= "card"*/) {
    NYT::TNode resultNode = CreateTransactionNodeBase(sessionId, clientId, type, paymentType);
    if (externalId) {
        resultNode["transaction_id"] = externalId;
    }
    resultNode["dt"] = ToString(ts);
    resultNode["transaction_type"] = isRefund ? "refund" : "payment";
    resultNode["total_sum"] = GetStringSum(totalSum + bonusSum, false);
    resultNode["commission_sum"] = "0.0";
    resultNode["promocode_sum"] = GetStringSum(bonusSum, false);
    return resultNode;
}

NYT::TNode NBillingExports::CreateCashbackTransactionNode(const TString& sessionId, const TString& clientId, ui32 totalSum, ui32 bonusSum, const TInstant& ts, const TString& type,
    const bool isRefund, const TString& externalId /*= ""*/, const TString& paymentType /*= "yandex_account_topup"*/) {
    NYT::TNode resultNode = CreateTransactionNodeBase(sessionId, clientId, type, paymentType);
    if (externalId) {
        resultNode["transaction_id"] = externalId;
    }
    resultNode["dt"] = ToString(ts);
    resultNode["total_sum"] = GetStringSum(totalSum, isRefund);
    resultNode["commission_sum"] = "0.0";
    resultNode["promocode_sum"] = GetStringSum(bonusSum, isRefund);
    return resultNode;
}

NYT::TNode NBillingExports::CreateSalaryNode(const TString& orderId, const ISalaryExportContext& context, const TString& clientId, TInstant ts, ui32 sum) {
    TString transactionId = NUtil::CreateUUID();
    return CreateSalaryNode(transactionId, orderId, context, clientId, ts, sum);
}

NYT::TNode NBillingExports::CreateSalaryNode(const TString& transactionId, const TString& orderId, const ISalaryExportContext& context, const TString& clientId, TInstant ts, ui32 sum) {
    NYT::TNode transactionNode;
    transactionNode["transaction_id"] = transactionId;
    transactionNode["transaction_type"] = "payment";
    transactionNode["service_id"] = context.GetBalanceServiceID();
    transactionNode["service_order_id"] = orderId;
    transactionNode["client_id"] = clientId;
    transactionNode["paysys_partner_id"] = context.GetBalancePartnerID();
    transactionNode["payment_id"] = transactionId;
    transactionNode["payment_type"] = context.GetBalancePaymentType();
    transactionNode["payload"] = "fake";
    transactionNode["product"] = context.GetBalanceProduct();
    transactionNode["value_currency"] = "RUB";
    transactionNode["value_amount"] = GetStringSum(sum, false);
    transactionNode["dt"] = ToString(ts);
    return transactionNode;
}
