#include "account.h"

namespace NDrive::NBilling {

    IBillingAccount::IBillingAccount(const TString& userId, const TAccountDescriptionRecord& description, TAccountRecord::TPtr accountRecord,
                                     IBillingAccount::TPtr parent, const TAccountsHistoryManager& historyManager)
        : AccountRecord(accountRecord)
        , Description(description)
        , HistoryManager(historyManager)
        , Parent(parent)
        , UserId(userId)
    {
        CHECK_WITH_LOG(AccountRecord);
    }

    NJson::TJsonValue IBillingAccount::GetNewUserReport(ELocalization locale, const IServerBase& server, const TVector<IBillingAccount::TPtr>& userAccounts, const IBillingAccount::TReportContext& /*context*/) const {
        NJson::TJsonValue result;
        result.InsertValue("group_type", ::ToString(GetGroupType()));
        result.InsertValue("name", GetName());

        auto local = server.GetLocalization();
        if (!Description.GetHiddenBalance()) {
            auto balance = GetBalance();
            if (Parent && Parent->GetBalance() >= 0 && Parent->GetBalance() < balance) {
                balance = Parent->GetBalance();
            }
            result.InsertValue("value", (balance < 0 ? "-" : "") + (local ? local->FormatRubles(locale, std::abs(balance) / 100, true) : ::ToString(std::abs(balance) / 100)));
        }
        result.InsertValue("type", ::ToString(GetType()));
        result.InsertValue("icon", Description.GetIcon());
        if (GetExcludeSwitchTypes()) {
            auto& excludeTypes = result.InsertValue("exclude_switch_types", NJson::JSON_ARRAY);
            for (const auto& acc : userAccounts) {
                if (GetExcludeSwitchTypes().contains(acc->GetType())) {
                    NJson::TJsonValue exclude;
                    acc->AddUniqueReport(exclude);
                    excludeTypes.AppendValue(exclude);
                }
            }
        }

        AddAccountToReport(result);
        return result;
    }

    void IBillingAccount::AddAccountToReport(NJson::TJsonValue& result) const {
        auto& requestParameters = result["request_parameters"];
        AddUniqueReport(requestParameters);
    }

    void IBillingAccount::AddUniqueReport(NJson::TJsonValue& result) const {
        result.InsertValue("account_id", GetUniqueName());
    }

    NJson::TJsonValue IBillingAccount::GetReport() const {
        NJson::TJsonValue result;
        if (AccountRecord) {
            result["details"] = AccountRecord->GetReport();
        } else {
            result["details"] = DoGetReport();
        }
        result["id"] = GetId();
        result["name"] = GetName();
        result["type_name"] = GetUniqueName();
        result["type"] = ::ToString(GetType());
        result["data_type"] = ::ToString(Description.GetDataType());
        result["balance"] = GetBalance();
        result["is_active"] = AccountRecord ? AccountRecord->GetActive() : false;
        result["is_actual"] = IsActual(Now());
        result["selectable"] = IsSelectable();
        result["disabled"] = Description.GetDisabled();
        result["soft_limit"] = GetSoftLimit();
        result["hard_limit"] = GetHardLimit();
        if (auto offersFilter = GetOffersFilter()) {
            result["offers_filter"] = offersFilter;
        }
        if (auto offersFilterName = GetOffersFilterName()) {
            result["offers_filter_name"] = offersFilterName;
        }
        if (auto customTollRoads = GetTollRoadsToPayFor()) {
            NJson::InsertField(result, "custom_toll_road", customTollRoads);
        }
        result["enable_toll_road_pay"] = EnableTollRoadsPay();
        result["version"] = GetVersion();
        result["spent"] = GetSpent();
        result["comment"] = AccountRecord ? AccountRecord->GetComment() : "";
        if (Parent) {
            result["parent"] = Parent->GetReport();
            if (Parent->GetRecord()) {
                result["parent"]["type_id"] = Parent->GetTypeId();
            }
        }
        return result;
    }

    TString IBillingAccount::GetCompanyName() const {
        if (Parent) {
            return Parent->GetCompanyName();
        }

        TString name;
        if (AccountRecord) {
            name = AccountRecord->GetCompany();
        }

        return name;
    }

    NJson::TJsonValue IBillingAccount::DoGetReport() const {
        FAIL_LOG("not valid");
        NJson::TJsonValue result;
        return result;
    }

    bool IBillingAccount::PatchAccountData(const NJson::TJsonValue& patch, const TString& historyUser, NDrive::TEntitySession& session) const {
        auto record = ReadAccountData(Description.GetDataType(), session);
        if (!record) {
            return false;
        }
        if (!record->PatchMeta(patch)) {
            return false;
        }
        return UpsertAccountData(*record, session, historyUser);
    }


    IBillingAccount::TPtr IBillingAccount::Construct(const TString& userId, const TAccountDescriptionRecord& description, TAccountRecord::TPtr accountRecord, IBillingAccount::TPtr parent, const TAccountsHistoryManager& historyManager) {
        if (accountRecord->GetAccountType() != description.GetType()) {
            return nullptr;
        }
        return TFactory::Construct(description.GetType(), userId, description, accountRecord, parent, historyManager);
    }

    TAccountRecord::TPtr IBillingAccount::ReadAccountData(EWalletDataType dataType, NDrive::TEntitySession& session) const {
        TAccountRecord::TPtr accountRec = TAccountRecord::TFactory::Construct(dataType);
        auto& db = session->GetDatabase();
        auto table = db.GetTable("billing_account");
        TRecordsSet records;
        if (!table->GetRows(NStorage::TRecordBuilder("account_id", GetId()), records, session.GetTransaction())->IsSucceed()
            || records.GetRecords().size() != 1) {
            return nullptr;
        }
        if (!accountRec->DeserializeFromTableRecord(records.GetRecords().front(), nullptr)) {
            return nullptr;
        }
        return accountRec;
    }

    bool IBillingAccount::UpsertAccountData(const TAccountRecord& record, NDrive::TEntitySession& session, const TString& historyUser, EObjectHistoryAction action) const {
        ui64 currentVersion = record.GetVersion();
        NStorage::TTableRecord accountUpdate = record.SerializeToTableRecord();
        accountUpdate.Remove("version");
        accountUpdate.Set("version", currentVersion + 1);

        auto& db = session->GetDatabase();
        auto table = db.GetTable("billing_account");

        NStorage::TObjectRecordsSet<TAccountData, IHistoryContext> affected(&HistoryManager.GetContext());
        auto res = table->UpdateRow(NStorage::TRecordBuilder("account_id", GetId())("version", currentVersion), accountUpdate, session.GetTransaction(), &affected);
        if (!res->IsSucceed() || affected.size() != 1) {
            return false;
        }
        return HistoryManager.AddHistory(affected.front(), historyUser, action, session);
    }
}
