#pragma once

#include "account_description.h"
#include "account_record.h"
#include "signals.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/entity/manager.h>

#include <util/generic/cast.h>

namespace NDrive::NBilling {

    class IBillingAccount {
    public:
        using TPtr = TAtomicSharedPtr<IBillingAccount>;
        using TConstPtr = TAtomicSharedPtr<const IBillingAccount>;

        IBillingAccount(const TString& userId,
                        const TAccountDescriptionRecord& description,
                        TAccountRecord::TPtr accountRecord,
                        IBillingAccount::TPtr parent,
                        const TAccountsHistoryManager& historyManager);

        EDriveOpResult Add(ui32 sum, NDrive::TEntitySession& session, TInstant deadline = TInstant::Max(), const TString& source = {}) const {
            if (sum > 0) {
                if (!Parent) {
                    return DoAdd(sum, session, deadline, source);
                }

                auto pStatus = Parent->Add(sum, session, deadline, source);
                if (pStatus == EDriveOpResult::Ok) {
                    return DoAdd(sum, session, deadline, source);
                }
                return pStatus;
            }
            return EDriveOpResult::LogicError;
        }

        EDriveOpResult Remove(ui32 sum, NDrive::TEntitySession& session) const {
            if (IsActive() && sum > 0) {
                if (!Parent) {
                    return DoRemove(sum, session);
                }
                auto pStatus = Parent->Remove(sum, session);
                if (pStatus == EDriveOpResult::Ok) {
                    return DoRemove(sum, session);
                }
                return pStatus;
            }
            return EDriveOpResult::LogicError;
        }

        TString GetCompanyName() const;

        TString GetUniqueName() const {
            return Description.GetName();
        }

        virtual TString GetName() const {
            return Description.GetHRName();
        }

        virtual TString GetDocName() const {
            return Description.GetDocName();
        }

        EWalletDataType GetDataType() const {
            return Description.GetDataType();
        }

        EAccount GetType() const {
            return Description.GetType();
        }

        bool IsActual(const TInstant timestamp) const {
            if (!IsActive()) {
                return false;
            }
            const auto& restrictions = AccountRecord->GetTimeRestrictions().Defined() ? AccountRecord->GetTimeRestrictions().GetRef() : Description.GetTimeRestrictions();
            return restrictions.Empty() || restrictions.IsActualNow(timestamp);
        }

        bool IsActive() const {
            if (Parent && !Parent->IsActive()) {
                return false;
            }
            if (Description.GetDisabled()) {
                return false;
            }
            return AccountRecord->GetActive();
        }

        bool IsSelectable() const {
            return Description.GetSelectable();
        }

        bool IsPersonal() const {
            return Description.GetIsPersonal();
        }

        i32 GetPriority() const {
            return Description.GetPriority();
        }

        ui64 GetSpent() const {
            return AccountRecord->GetSpent();
        }

        bool IsLimitedPolicy() const {
            return Description.IsLimitedPolicy();
        }

        virtual ui64 GetSoftLimit() const {
            if (AccountRecord->GetSoftLimit() == 0) {
                return Description.GetSoftLimit();
            }
            return AccountRecord->GetSoftLimit();
        }

        virtual ui64 GetHardLimit() const {
            if (AccountRecord->GetHardLimit() == 0) {
                return Description.GetHardLimit();
            }
            return AccountRecord->GetHardLimit();
        }

        virtual const TSet<TString>& GetInsuranceTypes() const {
            return Description.GetInsuranceTypes();
        }
        virtual TString GetOffersFilter() const {
            return AccountRecord->GetOffersFilter().Defined() ? AccountRecord->GetOffersFilter().GetRef() : Description.GetOffersFilter();
        }
        virtual TString GetOffersFilterName() const {
            return AccountRecord->GetOffersFilterName().Defined() ? AccountRecord->GetOffersFilterName().GetRef() : Description.GetOffersFilterName();
        }
        virtual TString GetInsuranceDescriptionFilter() const {
            return AccountRecord->GetInsuranceDescriptionFilter().Defined() ? AccountRecord->GetInsuranceDescriptionFilter().GetRef() : Description.GetInsuranceDescriptionFilter();
        }
        virtual bool EnableTollRoadsPay() const {
            return AccountRecord->EnableTollRoadsPay().Defined() ? AccountRecord->EnableTollRoadsPay().GetRef() : Description.GetEnableTollRoadsPay();
        }
        virtual TVector<TString> GetTollRoadsToPayFor() const {
            return AccountRecord->GetTollRoadsToPayFor().Defined() ? AccountRecord->GetTollRoadsToPayFor().GetRef() : Description.GetCustomTollRoads();
        }

        virtual ui64 GetId() const {
            return AccountRecord->GetAccountId();
        }

        virtual ui64 GetTypeId() const {
            return AccountRecord->GetTypeId();
        }

        ui64 GetVersion() const {
            return AccountRecord->GetVersion();
        }

        const TString& GetUserId() const {
            return UserId;
        }

        virtual i64 GetBalance() const = 0;

        virtual TSet<EAccount> GetExcludeSwitchTypes() const {
            return {};
        }

        TLimitedBalances GetLimitedBalance() const {
            return GetLimitedBalanceBySum(Max<ui32>());
        }

        bool GetHiddenBalance() const {
            return Description.GetHiddenBalance();
        }

        virtual TLimitedBalances GetLimitedBalanceBySum(ui32 /*sum*/) const {
            return {};
        }

        ERefreshStatus Refresh(const TString& historyUser, const TInstant timestamp, NDrive::TEntitySession& session) const {
            const TInstant nextRefresh = GetNextRefresh();
            if (nextRefresh == TInstant::Max() || nextRefresh > timestamp) {
                return ERefreshStatus::Skip;
            }

            TAccountRecord::TPtr record = ReadAccountData(Description.GetDataType(), session);
            if (!record) {
                return ERefreshStatus::Error;
            }
            NJson::TJsonValue metaJson;
            record->SaveMeta(metaJson);
            session.SetComment(metaJson.GetStringRobust());

            return (DoRefresh(record) && UpsertAccountData(*record, session, historyUser, EObjectHistoryAction::TagEvolve)) ? ERefreshStatus::Ok : ERefreshStatus::Error;
        }

        virtual bool DoRefresh(TAccountRecord::TPtr /*record*/) const {
            return true;
        }

        virtual TInstant GetNextRefresh() const {
            return TInstant::Max();
        }

        virtual NJson::TJsonValue GetUserReport() const {
            return NJson::TJsonValue(NJson::JSON_NULL);
        }

        virtual EPaymentGroupType GetGroupType() const {
            return EPaymentGroupType::Radio;
        }

        struct TReportContext {
            TMaybe<i32> ExternalBalance;
            TMaybe<TString> System;
        };
        virtual NJson::TJsonValue GetNewUserReport(ELocalization locale, const IServerBase& server, const TVector<IBillingAccount::TPtr>& userAccounts, const TReportContext& context) const;
        void AddAccountToReport(NJson::TJsonValue& result) const;
        void AddUniqueReport(NJson::TJsonValue& result) const;

        NJson::TJsonValue GetReport() const;
        virtual NJson::TJsonValue DoGetReport() const;

        TAccountRecord::TPtr GetRecord() const {
            return AccountRecord;
        }

        template <class TRecordType>
        const TRecordType* GetRecordAs() const {
            CHECK_WITH_LOG(AccountRecord);
            return dynamic_cast<const TRecordType*>(AccountRecord.Get());
        }

        virtual ~IBillingAccount() {}

        template <class T>
        const T* GetAsSafe() const {
            return VerifyDynamicCast<const T*>(this);
        }

        template <class T>
        T* GetAsSafe() {
            return VerifyDynamicCast<T*>(this);
        }

        template <class T>
        T* GetAs() {
            T* impl = dynamic_cast<T*>(this);
            if (!impl && Parent) {
                return Parent->GetAs<T>();
            }
            return impl;
        }

        IBillingAccount::TPtr GetParent() const {
            return Parent;
        }

        virtual bool IsSync() const {
            return true;
        }

        bool PatchAccountData(const NJson::TJsonValue& patch, const TString& historyUser,  NDrive::TEntitySession& session) const;

        static IBillingAccount::TPtr Construct(const TString& userId,
                                               const TAccountDescriptionRecord& description,
                                               TAccountRecord::TPtr accountRecord,
                                               IBillingAccount::TPtr parent,
                                               const TAccountsHistoryManager& historyManager);

        static IBillingAccount::TPtr GetRootAccount(const IBillingAccount::TPtr account) {
            Y_ENSURE(account);
            const auto& parent = account->GetParent();
            if (!parent) {
                return account;
            }

            return GetRootAccount(parent);
        }
        
    protected:
        using TFactory = NObjectFactory::TParametrizedObjectFactory<IBillingAccount,
                                                                    EAccount,
                                                                    TString,
                                                                    const TAccountDescriptionRecord&,
                                                                    TAccountRecord::TPtr,
                                                                    IBillingAccount::TPtr,
                                                                    const TAccountsHistoryManager&>;

    protected:
        TAccountRecord::TPtr ReadAccountData(EWalletDataType dataType, NDrive::TEntitySession& session) const;
        bool UpsertAccountData(const TAccountRecord& record, NDrive::TEntitySession& session, const TString& historyUser, EObjectHistoryAction action = EObjectHistoryAction::UpdateData) const;

        virtual EDriveOpResult DoRemove(ui32 sum, NDrive::TEntitySession& session) const = 0;
        virtual EDriveOpResult DoAdd(ui32 sum, NDrive::TEntitySession& session, TInstant deadline = TInstant::Max(), const TString& source = {}) const = 0;

    protected:
        TAccountRecord::TPtr AccountRecord;
        TAccountDescriptionRecord Description;
        const TAccountsHistoryManager& HistoryManager;
        IBillingAccount::TPtr Parent;
    private:
        TString UserId;
    };

    class ITrustAccount {
    public:
        virtual ~ITrustAccount() {}
        virtual TString GetDefaultCard() const = 0;
        virtual bool SetDefaultCard(const NDrive::NTrustClient::TPaymentMethod& defaultCard, const TString& historyUser, NDrive::TEntitySession& session) = 0;
    };
}
