#include "account_description.h"

#include "account.h"


NDrive::TScheme NDrive::NBilling::TAccountDescriptionRecord::GetScheme(const IServerBase* /*server*/) {
    NDrive::TScheme result;

    result.Add<TFSVariants>("type", "Тип кошелька").InitVariants<EAccount>();
    result.Add<TFSString>("name", "Имя");
    result.Add<TFSNumeric>("soft_limit", "Нижний лимит").SetDefault(500000);
    result.Add<TFSNumeric>("hard_limit", "Верхний лимит").SetDefault(500000);

    NDrive::TScheme meta;
    meta.Add<TFSString>("hr_name", "Название в репорте");
    meta.Add<TFSString>("hr_doc_name", "Название в документе");
    meta.Add<TFSString>("details", "Детали");
    meta.Add<TFSString>("short_description", "Краткое описание");
    meta.Add<TFSVariants>("refresh_policy", "Политика обновления").InitVariants<TRefreshSchedule::EPolicy>().SetDefault(::ToString(TRefreshSchedule::EPolicy::Month));
    meta.Add<TFSNumeric>("refresh_interval", "Частота обновления").SetDefault(1);
    meta.Add<TFSVariants>("data_type", "Тип данных").InitVariants<EWalletDataType>();
    meta.Add<TFSVariants>("insurance_types", "Типы страховок").SetMultiSelect(true).SetVariants({
        "full",
        "standart",
    });
    meta.Add<TFSBoolean>("selectable", "Доступен для выбора").SetDefault(false);
    meta.Add<TFSBoolean>("is_personal", "Для привязки пользователям").SetDefault(true);
    meta.Add<TFSBoolean>("disabled", "Выключен").SetDefault(false);
    meta.Add<TFSNumeric>("priority", "Приоритет в платежах (действует в рамках одного типа)").SetDefault(0);
    meta.Add<TFSNumeric>("parent_id", "Родительский кошелек");
    meta.Add<TFSNumeric>("max_links", "Максимальное число пользователей для одного кошелька").SetDefault(1);
    meta.Add<TFSText>("offers_filter", "Фильтр для тарифов (полный фильтр)");
    meta.Add<TFSString>("offers_filter_name", "Фильтр для тарифов (имя фильтра)");
    meta.Add<TFSString>("insurance_description_filter", "Фильтр для страховок (только для описания)");
    meta.Add<TFSBoolean>("enable_toll_roads", "Включить оплату платных дорог").SetDefault(true);
    meta.Add<TFSArray>("custom_toll_roads", "Оплата отдельных дорог").SetElement<TFSString>();
    meta.Add<TFSBoolean>("limited_policy", "Использовать лимитированную политику").SetDefault(false);
    meta.Add<TFSStructure>("time_restrictions", "Расписание активации").SetStructure(TTimeRestrictionsPool<class TTimeRestriction>::GetScheme());
    meta.Add<TFSBoolean>("hidden_balance", "Скрывать баланс в приложении").SetDefault(false);
    meta.Add<TFSString>("icon", "Иконка");

    result.Add<TFSStructure>("meta", "Дополнительные параметры").SetStructure(meta);
    return result;
}

NDrive::TScheme NDrive::NBilling::TAccountDescriptionRecord::GetPublicScheme(const IServerBase* /*server*/) {
    NDrive::TScheme result;

    result.Add<TFSVariants>("type", "Тип кошелька").InitVariants<EAccount>();
    result.Add<TFSString>("name", "Имя");
    result.Add<TFSNumeric>("soft_limit", "Нижний лимит").SetDefault(500000);
    result.Add<TFSNumeric>("hard_limit", "Верхний лимит").SetDefault(500000);

    NDrive::TScheme meta;
    meta.Add<TFSString>("hr_name", "Название в репорте");
    meta.Add<TFSString>("hr_doc_name", "Название в документе");
    meta.Add<TFSString>("details", "Детали");
    meta.Add<TFSString>("short_description", "Краткое описание");
    meta.Add<TFSVariants>("refresh_policy", "Политика обновления").InitVariants<TRefreshSchedule::EPolicy>().SetDefault(::ToString(TRefreshSchedule::EPolicy::Month));
    meta.Add<TFSNumeric>("refresh_interval", "Частота обновления").SetDefault(1);
    meta.Add<TFSVariants>("data_type", "Тип данных").InitVariants<EWalletDataType>();
    meta.Add<TFSVariants>("insurance_types", "Типы страховок").SetMultiSelect(true).SetVariants({
        "full",
        "standart",
    });
    meta.Add<TFSBoolean>("selectable", "Доступен для выбора").SetDefault(false);
    meta.Add<TFSBoolean>("is_personal", "Для привязки пользователям").SetDefault(true);
    meta.Add<TFSBoolean>("disabled", "Выключен").SetDefault(false);
    meta.Add<TFSNumeric>("parent_id", "Родительский кошелек");
    meta.Add<TFSNumeric>("max_links", "Максимальное число пользователей для одного кошелька").SetDefault(1);
    meta.Add<TFSText>("offers_filter", "Фильтр для тарифов (полный фильтр)");
    meta.Add<TFSString>("offers_filter_name", "Фильтр для тарифов (имя фильтра)");
    meta.Add<TFSString>("insurance_description_filter", "Фильтр для страховок (только для описания)");
    meta.Add<TFSBoolean>("enable_toll_roads", "Включить оплату платных дорог").SetDefault(true);
    meta.Add<TFSArray>("custom_toll_roads", "Оплата отдельных дорог").SetElement<TFSString>();
    meta.Add<TFSBoolean>("limited_policy", "Использовать лимитированную политику").SetDefault(false);
    meta.Add<TFSStructure>("time_restrictions", "Расписание активации").SetStructure(TTimeRestrictionsPool<class TTimeRestriction>::GetScheme());
    meta.Add<TFSBoolean>("hidden_balance", "Скрывать баланс в приложении").SetDefault(false);
    meta.Add<TFSString>("icon", "Иконка");

    result.Add<TFSStructure>("meta", "Дополнительные параметры").SetStructure(meta);
    return result;
}

bool NDrive::NBilling::TAccountDescriptionRecord::DeserializeWithDecoder(const TAccountDescriptionDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, Type);
    READ_DECODER_VALUE(decoder, values, Name);
    if (!Name) {
        return false;
    }
    READ_DECODER_VALUE(decoder, values, HardLimit);
    READ_DECODER_VALUE(decoder, values, SoftLimit);
    READ_DECODER_VALUE_JSON(decoder, values, Meta, Meta);
    HRName = Meta["hr_name"].GetStringRobust();

    JREAD_STRING_OPT(Meta, "offers_filter", OffersFilter);
    OffersFilter = StripInPlace(OffersFilter);
    JREAD_STRING_OPT(Meta, "offers_filter_name", OffersFilterName);
    JREAD_STRING_OPT(Meta, "short_description", ShortDescription);
    JREAD_STRING_OPT(Meta, "details", Details);
    JREAD_STRING_OPT(Meta, "hr_doc_name", HRDocName);

    JREAD_BOOL_OPT(Meta, "selectable", Selectable);
    JREAD_BOOL_OPT(Meta, "is_personal", IsPersonal);
    JREAD_BOOL_OPT(Meta, "disabled", Disabled);

    JREAD_UINT_OPT(Meta, "parent_id", ParentId);
    JREAD_INT_OPT(Meta, "priority", Priority);
    JREAD_UINT_OPT(Meta, "refresh_interval", RefreshInterval);
    JREAD_UINT_OPT(Meta, "max_links", MaxLinks);

    if (Meta.Has("refresh_policy")) {
        if (!TryFromString(Meta["refresh_policy"].GetString(), RefreshPolicy)) {
            return false;
        }
    }
    if (Meta.Has("data_type")) {
        if (!TryFromString(Meta["data_type"].GetString(), DataType)) {
            return false;
        }
    } else {
        if (Type == EAccount::Bonus) {
            DataType = EWalletDataType::Bonus;
        }
        if (Type == EAccount::Trust) {
            DataType = EWalletDataType::Trust;
        }
    }
    if (Meta.Has("time_restrictions")) {
        if (!TimeRestrictions.DeserializeFromJson(Meta["time_restrictions"])) {
            return false;
        }
    }
    JREAD_BOOL_OPT(Meta, "limited_policy", LimitedPolicy);
    JREAD_BOOL_OPT(Meta, "hidden_balance", HiddenBalance);
    JREAD_STRING_OPT(Meta, "icon", Icon);
    return
        NJson::ParseField(Meta["insurance_types"], InsuranceTypes)
        && NJson::ParseField(Meta, "insurance_description_filter", InsuranceDescriptionFilter)
        && NJson::ParseField(Meta, "enable_toll_roads", EnableTollRoadsPay, false)
        && NJson::ParseField(Meta, "custom_toll_roads", CustomTollRoads);
}

bool NDrive::NBilling::TAccountDescriptionRecord::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

NStorage::TTableRecord NDrive::NBilling::TAccountDescriptionRecord::SerializeToTableRecord() const {
    NStorage::TRecordBuilder builder;
    builder("name", Name)("type", Type)("hard_limit", HardLimit)("soft_limit", SoftLimit)("meta", Meta);
    if (Id != Max<ui32>()) {
        builder("id", Id);
    }
    return builder;
}

bool NDrive::NBilling::TAccountDescriptionRecord::FromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* /*errors*/) {
    return TBaseDecoder::DeserializeFromJson(*this, jsonValue);
}

void NDrive::NBilling::TAccountDescriptionRecord::FillReport(NJson::TJsonValue& report) const {
    report["id"] = Id;
    report["name"] = Name;
    report["type"] = ::ToString(Type);
    report["hard_limit"] = HardLimit;
    report["soft_limit"] = SoftLimit;
    report["meta"] = Meta;
}

NJson::TJsonValue NDrive::NBilling::TAccountDescriptionRecord::GetReport() const {
    NJson::TJsonValue report;
    FillReport(report);
    return report;
}

void NDrive::NBilling::TAccountDescriptionRecord::DoBuildReportItem(NJson::TJsonValue& json) const {
    FillReport(json);
}

ui32 NDrive::NBilling::TAccountDescriptionRecord::GetHistoryEventId() const {
    return Id;
}

const TString& NDrive::NBilling::TAccountDescriptionRecord::GetHistoryUserId() const {
    return Default<TString>();
}

const TString& NDrive::NBilling::TAccountDescriptionRecord::GetDocName() const {
    return !HRDocName.empty() ? HRDocName : HRName;
}

NDrive::NBilling::TAccountDescriptionHistory::TOptionalEvents NDrive::NBilling::TAccountDescriptionHistory::GetEventsByDescriptionId(const ui64& id, TRange<TInstant> timestampRange, const TMaybe<TString>& historyComment, NDrive::TEntitySession& session, ui64 limit, ui64 offset) const {
    TQueryOptions options(limit, true, offset);
    options.AddGenericCondition("id", ::ToString(id));
    if (historyComment) {
        options.AddGenericCondition("history_comment", *historyComment);
    }
    return TBase::GetEvents({}, timestampRange, session, options);
}

NDrive::NBilling::TAccountDescriptionHistory::TOptionalEvents NDrive::NBilling::TAccountDescriptionHistory::GetEventsByDescriptionName(const TString& name, TRange<TInstant> timestampRange, const TMaybe<TString>& historyComment, NDrive::TEntitySession& session, ui64 limit, ui64 offset) const {
    TQueryOptions options(limit, true, offset);
    options.AddGenericCondition("name", ::ToString(name));
    if (historyComment) {
        options.AddGenericCondition("history_comment", *historyComment);
    }
    return TBase::GetEvents({}, timestampRange, session, options);
}

NDrive::NBilling::TAccountsDescriptionDB::TAccountsDescriptionDB(const IHistoryContext& context, const THistoryConfig& hConfig)
    : TBase("billing_account_description", context, hConfig)
    , Database(context.GetDatabase())
    , DescriptionsTable(Database)
{
}

bool NDrive::NBilling::TAccountsDescriptionDB::DoRebuildCacheUnsafe() const {
    auto session = NDrive::TEntitySession(Database->CreateTransaction(true));
    auto table = Database->GetTable(GetTableName());

    NStorage::TObjectRecordsSet<TAccountDescriptionRecord> records;
    auto reqResult = table->GetRows("", records, session.GetTransaction());
    if (!reqResult->IsSucceed()) {
        ERROR_LOG << session.GetTransaction()->GetErrors().GetStringReport() << Endl;
        return false;
    }
    Objects.clear();
    ObjectByIds.clear();
    for (auto&& description : records) {
        Objects[description.GetName()] =  description;
        ObjectByIds[description.GetId()] = description;
    }
    return true;
}

bool NDrive::NBilling::TAccountsDescriptionDB::Upsert(const TAccountDescriptionRecord& entity, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord newRecord = entity.SerializeToTableRecord();

    NStorage::TTableRecord unique;
    unique.Set("name", entity.GetName());

    NStorage::ITableAccessor::TPtr table = Database->GetTable(GetTableName());
    auto transaction = session.GetTransaction();

    NStorage::TObjectRecordsSet<TAccountDescriptionRecord> affected;
    auto result = table->UpdateRow(unique.BuildCondition(*transaction), newRecord.BuildSet(*transaction), transaction, &affected);
    if (!result || !result->IsSucceed()) {
        session.SetErrorInfo("TAccountsDescriptionDB::Upsert", "UpdateRow failure", EDriveSessionResult::TransactionProblem);
        return false;
    }

    EObjectHistoryAction historyAction = EObjectHistoryAction::UpdateData;
    if (affected.empty()) {
        historyAction = EObjectHistoryAction::Add;
        result = table->AddIfNotExists(newRecord, transaction, unique, &affected);
        if (!result || !result->IsSucceed() || result->GetAffectedRows() != 1) {
            session.SetErrorInfo("TAccountsDescriptionDB::Upsert", "AddIfNotExists failure", EDriveSessionResult::TransactionProblem);
            return false;
        }
    }

    return HistoryManager->AddHistory(affected.front(), userId, historyAction, session);
}


bool NDrive::NBilling::TAccountsDescriptionDB::Remove(const ui32 typeId, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord unique;
    unique.Set("id", typeId);

    NStorage::ITableAccessor::TPtr table = Database->GetTable(GetTableName());
    auto transaction = session.GetTransaction();

    NStorage::TObjectRecordsSet<TAccountDescriptionRecord> affected;
    auto result = table->RemoveRow(unique.BuildCondition(*transaction), transaction, &affected);

    if (affected.size() != 1) {
        session.SetErrorInfo(GetTableName(), "remove", EDriveSessionResult::TransactionProblem);
        return false;
    }
    return HistoryManager->AddHistory(affected.front(), userId, EObjectHistoryAction::Remove, session);
}

TStringBuf NDrive::NBilling::TAccountsDescriptionDB::GetEventObjectId(const TObjectEvent<TAccountDescriptionRecord>& ev) const {
    return ev.GetName();
}

void NDrive::NBilling::TAccountsDescriptionDB::AcceptHistoryEventUnsafe(const TObjectEvent<TAccountDescriptionRecord>& ev) const {
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        Objects.erase(ev.GetName());
        ObjectByIds.erase(ev.GetId());
    }

    if (ev.GetHistoryAction() == EObjectHistoryAction::Add || ev.GetHistoryAction() == EObjectHistoryAction::UpdateData) {
        Objects[ev.GetName()] = ev;
        ObjectByIds[ev.GetId()] = ev;
    }
}


TMaybe<NDrive::NBilling::TAccountDescriptionRecord> NDrive::NBilling::TAccountsDescriptionDB::GetDescriptionById(ui32 typeId, TInstant lastInstant) const {
    if (!RefreshCache(lastInstant)) {
        return Nothing();
    }
    auto rg = MakeObjectReadGuard();
    auto it = ObjectByIds.find(typeId);
    if (it == ObjectByIds.end()) {
        return Nothing();
    }
    return it->second;
}

bool NDrive::NBilling::TAccountsDescriptionDB::GetCustomObjectsByTypeIds(TVector<TAccountDescriptionRecord>& result, const TSet<ui32>& typeIds, const TInstant reqActuality) {
    if (!RefreshCache(reqActuality)) {
        return false;
    }
    auto rg = MakeObjectReadGuard();
    for (auto&& i : typeIds) {
        auto it = ObjectByIds.find(i);
        if (it == ObjectByIds.end()) {
            continue;
        }
        result.emplace_back(it->second);
    }
    return true;
}

TString NDrive::NBilling::TAccountsDescriptionDB::GetTableName() const {
    return DescriptionsTable.GetTableName();
}

TInstant NDrive::NBilling::TRefreshSchedule::GetNextInstant(EPolicy policy, ui32 interval, const TInstant current) {
    struct tm tmNow;
    current.LocalTime(&tmNow);
    TDateTimeFields timeParts;
    switch (policy) {
    case EPolicy::Month:
        timeParts.Hour = 0;
        timeParts.Day = 1;
        timeParts.Month = (tmNow.tm_mon + interval) % 12 + 1;
        timeParts.SetLooseYear(tmNow.tm_mon + interval > 11 ? tmNow.tm_year + 1 : tmNow.tm_year);
        return timeParts.ToInstant(TInstant::Max());
    case EPolicy::Day:
        return TInstant::Days(current.Days()) + TDuration::Days(interval);
    case EPolicy::Week:
        return TInstant::Days(current.Days()) + TDuration::Days(7 - tmNow.tm_wday) + TDuration::Days(7) * (interval - 1);
    case EPolicy::None:
        return TInstant::Max();
    }
}

NDrive::NBilling::TAccountsDescriptionTable::TFetchResult NDrive::NBilling::TAccountsDescriptionDB::GetDescriptionsById(const TSet<ui32>& typeIds, NDrive::TEntitySession& session) const {
    return DescriptionsTable.FetchInfo(typeIds, session);
}

NDrive::NBilling::TAccountsDescriptionTable::TFetchResult NDrive::NBilling::TAccountsDescriptionDB::GetDescriptionsByType(NDrive::NBilling::EAccount type, NDrive::TEntitySession& session) const {
    return DescriptionsTable.FetchInfoByField(TSet<TString>({ ::ToString(type) }), "type", session);
}
