#pragma once

#include "entities.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/database/history/manager.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/parser.h>
#include <util/generic/cast.h>


namespace NDrive::NBilling {
    static const ui32 AccountLimit = 100000000;

    class TRefreshSchedule {
    public:
        enum class EPolicy {
            Month /* "month" */,
            Day /* "day" */,
            Week /* "week" */,
            None /* "none" */
        };

    public:
        static TInstant GetNextInstant(EPolicy policy, ui32 interval, const TInstant current);
    };

    enum class EAccount {
        Bonus = 0 /* "bonus" */,
        Wallet = 1 /* "wallet" */,
        Trust = 2 /* "card" */,
        Fake = 3 /* "fake" */,
        YMoney = 4 /* "ymoney" */, // deprecated
        Coins = 5 /* "coins" */,
        YAccount = 6 /* "yandex_account" */,
        YCashback = 7 /* "yandex_cashback" */,
        MobilePayment = 8 /* "mobile_payment" */,
    };

    enum class EWalletDataType {
        Bonus = 0 /* "bonus" */,
        Simple = 1 /* "simple" */,
        Trust = 2 /* "card" */,
        Salary = 3 /* "salary" */,
        Troika = 4 /* "troika" */, // deprecated
        YMoney = 5 /* "ymoney" */, // deprecated
        Coins = 6 /* "coins" */,
        YAccount = 7 /* "yandex_account" */,
        MobilePayment = 8 /* "mobile_payment" */,
    };

    class TAccountDescriptionDecoder : public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, HardLimit, -1);
        R_FIELD(i32, SoftLimit, -1);
        R_FIELD(i32, Meta, -1);

    public:
        TAccountDescriptionDecoder() = default;

        TAccountDescriptionDecoder(const TMap<TString, ui32>& decoderBase)
        {
            Id = GetFieldDecodeIndex("id", decoderBase);
            Type = GetFieldDecodeIndex("type", decoderBase);
            Name = GetFieldDecodeIndex("name", decoderBase);
            HardLimit = GetFieldDecodeIndex("hard_limit", decoderBase);
            SoftLimit = GetFieldDecodeIndex("soft_limit", decoderBase);
            Meta = GetFieldDecodeIndex("meta", decoderBase);
        }
    };

    class TAccountDescriptionRecord {
        R_READONLY(ui32, Id, Max<ui32>());
        R_READONLY(EAccount, Type, EAccount::Bonus);
        R_READONLY(EWalletDataType, DataType, EWalletDataType::Simple);
        R_READONLY(TString, Name);
        R_READONLY(TString, HRName);
        R_READONLY(TString, HRDocName);
        R_READONLY(ui64, HardLimit, AccountLimit);
        R_READONLY(ui64, SoftLimit, AccountLimit);
        R_READONLY(NJson::TJsonValue, Meta, NJson::JSON_MAP);
        R_READONLY(TRefreshSchedule::EPolicy, RefreshPolicy, TRefreshSchedule::EPolicy::None);
        R_READONLY(ui32, RefreshInterval, 1);
        R_READONLY(bool, Selectable, false);
        R_READONLY(bool, IsPersonal, true);
        R_READONLY(bool, Disabled, false);
        R_READONLY(i32, Priority, 0);
        R_READONLY(ui32, MaxLinks, 1);
        R_READONLY(ui32, ParentId, 0);
        R_READONLY(TString, Details);
        R_READONLY(TString, ShortDescription);
        R_READONLY(TString, OffersFilter);
        R_READONLY(TString, OffersFilterName);
        R_READONLY(TString, InsuranceDescriptionFilter);
        R_READONLY(TSet<TString>, InsuranceTypes);
        R_READONLY(bool, EnableTollRoadsPay, true);
        R_READONLY(TVector<TString>, CustomTollRoads);
        R_READONLY(bool, LimitedPolicy, false);
        R_READONLY(TTimeRestrictionsPool<class TTimeRestriction>, TimeRestrictions);
        R_READONLY(bool, HiddenBalance, false);
        R_READONLY(TString, Icon);

    public:
        using TDecoder = TAccountDescriptionDecoder;

        bool DeserializeWithDecoder(const TAccountDescriptionDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
        bool Parse(const NStorage::TTableRecord& row);
        NStorage::TTableRecord SerializeToTableRecord() const;

        bool FromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* /*errors*/);

        void FillReport(NJson::TJsonValue& report) const;
        NJson::TJsonValue GetReport() const;
        void DoBuildReportItem(NJson::TJsonValue& json) const;

        ui32 GetHistoryEventId() const;

        const TString& GetHistoryUserId() const;

        const TString& GetDocName() const;

        TInstant GetNextInstant(const TInstant current) const {
            return TRefreshSchedule::GetNextInstant(RefreshPolicy, RefreshInterval, current);
        }

        static NDrive::TScheme GetScheme(const IServerBase* server);
        static NDrive::TScheme GetPublicScheme(const IServerBase* server);
    };

    class TAccountsDescriptionTable : public TDBEntities<TAccountDescriptionRecord, true, ui32> {
    private:
        using TBase = TDBEntities<TAccountDescriptionRecord, true, ui32>;
        using TEntity = TAccountDescriptionRecord;

    public:
        using TBase::TBase;

        virtual TString GetTableName() const override {
            return "billing_account_description";
        }

        virtual TString GetColumnName() const override {
            return "id";
        }

        virtual ui32 GetMainId(const TAccountDescriptionRecord& e) const override {
            return e.GetId();
        }
    };

    class TAccountDescriptionHistory : public TIndexedAbstractHistoryManager<TAccountDescriptionRecord> {
    private:
        using TBase = TIndexedAbstractHistoryManager<TAccountDescriptionRecord>;
    public:
        TAccountDescriptionHistory(const IHistoryContext& context, const THistoryConfig& hConfig)
            : TBase(context, "billing_account_description_history", hConfig) {}

        TOptionalEvents GetEventsByDescriptionId(const ui64& id, TRange<TInstant> timestampRange, const TMaybe<TString>& historyComment, NDrive::TEntitySession& session, ui64 limit = 0, ui64 offset = 0) const;
        TOptionalEvents GetEventsByDescriptionName(const TString& name, TRange<TInstant> timestampRange, const TMaybe<TString>& historyComment, NDrive::TEntitySession& session, ui64 limit = 0, ui64 offset = 0) const;
    };

    class TAccountsDescriptionDB: public TDBCacheWithHistoryOwner<TAccountDescriptionHistory, TAccountDescriptionRecord> {
    private:
        using TBase = TDBCacheWithHistoryOwner<TAccountDescriptionHistory, TAccountDescriptionRecord>;
    public:
        TAccountsDescriptionDB(const IHistoryContext& context, const THistoryConfig& hConfig);

        virtual bool DoRebuildCacheUnsafe() const override;

        bool Upsert(const TAccountDescriptionRecord& entity, const TString& userId, NDrive::TEntitySession& session) const;
        bool Remove(const ui32 typeId, const TString& userId, NDrive::TEntitySession& session) const;

        virtual TStringBuf GetEventObjectId(const TObjectEvent<TAccountDescriptionRecord>& ev) const override;

        virtual void AcceptHistoryEventUnsafe(const  TObjectEvent<TAccountDescriptionRecord>& ev) const override;

        TMaybe<TAccountDescriptionRecord> GetDescriptionById(ui32 typeId, TInstant lastInstant) const;
        bool GetCustomObjectsByTypeIds(TVector<TAccountDescriptionRecord>& result, const TSet<ui32>& typeIds, const TInstant reqActuality);

        TAccountsDescriptionTable::TFetchResult GetDescriptionsById(const TSet<ui32>& typeIds, NDrive::TEntitySession& session) const;
        TAccountsDescriptionTable::TFetchResult GetDescriptionsByType(EAccount type, NDrive::TEntitySession& session) const;
    private:
        TString GetTableName() const;
    private:
        TDatabasePtr Database;
        mutable TMap<ui32, TAccountDescriptionRecord> ObjectByIds;
        TAccountsDescriptionTable DescriptionsTable;
    };
}
