#include "account_link.h"
#include <util/generic/algorithm.h>

bool NDrive::NBilling::TAccountLinkRecord::DeserializeWithDecoder(const TAccountLinkDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, AccountId);
    READ_DECODER_VALUE_DEF(decoder, values, TypeId, 0);
    return true;
}

NStorage::TTableRecord NDrive::NBilling::TAccountLinkRecord::SerializeToTableRecord() const {
    NStorage::TTableRecord record;
    record.Set("account_id", AccountId);
    record.Set("user_id", UserId);
    if (TypeId != 0) {
        record.Set("type_id", TypeId);
    }
    return record;
}

NDrive::NBilling::TAccounLinksHistory::TOptionalEvents NDrive::NBilling::TAccounLinksHistory::GetEventsByAccounts(const TSet<ui64>& ids, TRange<TInstant> timestampRange, NDrive::TEntitySession& session) const {
    return TBase::GetEvents({}, timestampRange, session, TQueryOptions().SetGenericCondition("account_id", ids));
}

TString NDrive::NBilling::TAccountsLinksDB::TableName = "billing_user_accounts";

NDrive::NBilling::TAccountsLinksDB::TAccountsLinksDB(const IHistoryContext& context)
    : IAutoActualization(TableName + "-cache", TDuration::Seconds(1))
    , Database(context.GetDatabase())
    , HistoryManager(context)
    , DefaultLifetime(TDuration::Seconds(120))
    , CacheHit({ TableName + "-cache-hit" }, false)
    , CacheMiss({ TableName + "-cache-miss" }, false)
    , CacheExpired({ TableName + "-cache-expired" }, false)
    , CacheInvalidated({ TableName + "-cache-invalidated" }, false)
    , ObjectCache(64 * 1024)
{
}

const TString& NDrive::NBilling::TAccountsLinksDB::GetTableName() const {
    return TableName;
}

bool NDrive::NBilling::TAccountsLinksDB::AddLink(const TString& userId, const ui64 accountId, ui32 typeId, const TString& historyUser, ui32 maxLinks, NDrive::TEntitySession& session) const {
    TAccountLinkRecord link;
    link.SetUserId(userId);
    link.SetAccountId(accountId);
    link.SetTypeId(typeId);

    NStorage::TObjectRecordsSet<TAccountLinkRecord> affected;
    TStringStream condition;
    if (maxLinks == 1) {
        condition << "NOT EXISTS (SELECT account_id FROM " << GetTableName() << " WHERE account_id=" << accountId << ")" << Endl;
    }
    Database->GetTable(GetTableName())->AddRow(link.SerializeToTableRecord(), session.GetTransaction(), condition.Str(), &affected);
    if (affected.size() != 1) {
        return false;
    }
    return HistoryManager.AddHistory(affected.front(), historyUser, EObjectHistoryAction::Add, session);
}

bool NDrive::NBilling::TAccountsLinksDB::RemoveLink(const TString& userId, const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const {
    TAccountLinkRecord link;
    link.SetUserId(userId);
    link.SetAccountId(accountId);

    NStorage::TObjectRecordsSet<TAccountLinkRecord> affected;
    Database->GetTable(GetTableName())->RemoveRow(link.SerializeToTableRecord(), session.GetTransaction(), &affected);
    if (affected.size() != 1) {
        return false;
    }
    return HistoryManager.AddHistory(affected.front(), historyUser, EObjectHistoryAction::Remove, session);
}

TMaybe<NDrive::NBilling::TAccountsLinksDB::TAccountLinks> NDrive::NBilling::TAccountsLinksDB::GetLinkAccounts(NSQL::TQueryOptions&& options, NDrive::TEntitySession& session) const {
    NStorage::TObjectRecordsSet<TAccountLinkRecord, IHistoryContext> affected;
    auto tx = session.GetTransaction();
    const TString query = options.PrintQuery(*tx, GetTableName());
    auto queryResult = tx->Exec(query, &affected);
    if (!ParseQueryResult(queryResult, session)) {
        return {};
    }
    return affected.DetachObjects();
}

TExpected<NDrive::NBilling::TAccountsLinksDB::TAccountLinks, TString> NDrive::NBilling::TAccountsLinksDB::GetCachedUserAccounts(const TString& userId, TInstant freshness) const {
    auto timedAccounts = ObjectCache.find(userId);
    if (timedAccounts) {
        if (timedAccounts->Timestamp + DefaultLifetime < Now()) {
            CacheExpired.Signal(1);
        } else {
            CacheHit.Signal(1);
            if (timedAccounts->Timestamp >= freshness) {
                return timedAccounts->Accounts;
            }
        }
    } else {
        CacheMiss.Signal(1);
    }
    auto session = HistoryManager.BuildSession(true);
    auto accounts = GetUserAccounts(userId, session);
    if (!accounts) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }
    return *accounts;
}

TMaybe<NDrive::NBilling::TAccountsLinksDB::TAccountLinks> NDrive::NBilling::TAccountsLinksDB::GetUserAccounts(const TString& userId, NDrive::TEntitySession& session) const {
    return GetUsersAccounts({ userId }, session);
}

TMaybe<NDrive::NBilling::TAccountsLinksDB::TAccountLinks> NDrive::NBilling::TAccountsLinksDB::GetUsersAccounts(const TSet<TString>& userIds, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions options;
    options.SetOrderBy({ "account_id" });
    options.SetGenericCondition("user_id", userIds);
    TInstant time = Now();
    auto accountLink = GetLinkAccounts(std::move(options), session);
    if (accountLink) {
        TMap<TString, TAccountLinks> userAccounts;
        ForEach(accountLink->begin(), accountLink->end(), [&userAccounts](const auto& link) {
            userAccounts[link.GetUserId()].emplace_back(link);
        });
        for (auto&& [userId, accounts] : userAccounts) {
            ObjectCache.update(userId, { accounts, time });
        }
    }
    return accountLink;
}

TMaybe<NDrive::NBilling::TAccountsLinksDB::TAccountLinks> NDrive::NBilling::TAccountsLinksDB::GetAccountsUsers(const TSet<ui64>& ids, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions options;
    options.SetOrderBy({ "account_id" });
    options.SetGenericCondition("account_id", ids);
    return GetLinkAccounts(std::move(options), session);
}


TMaybe<TSet<ui64>> NDrive::NBilling::TAccountsLinksDB::FilterAccounts(const TSet<ui64>& ids, bool used, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions options;
    options.SetGenericCondition("account_id", ids);
    auto accountRecords = GetLinkAccounts(std::move(options), session);
    if (!accountRecords) {
        return {};
    }

    TSet<ui64> accounts;
    Transform(accountRecords->begin(), accountRecords->end(), std::inserter(accounts, accounts.begin()), [](const auto& acc){ return acc.GetAccountId(); });
    if (used) {
        return accounts;
    }
    TSet<ui64> result;
    SetDifference(ids.begin(), ids.end(), accounts.begin(), accounts.end(), std::inserter(result, result.begin()));
    return result;
}

bool NDrive::NBilling::TAccountsLinksDB::Refresh() {
    auto session = HistoryManager.BuildSession(true);
    if (!LastEventId) {
        LastEventId = HistoryManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = HistoryManager.GetEvents({ since }, {}, session, NSQL::TQueryOptions().SetLimit(1000));
    if (!optionalEvents) {
        ERROR_LOG << TableName << ": cannot GetEvents since " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& event : *optionalEvents) {
        bool erased = ObjectCache.erase(event.GetUserId());
        if (erased) {
            CacheInvalidated.Signal(1);
        }
        LastEventId = std::max(LastEventId.GetOrElse(0), event.GetHistoryEventId());
    }
    return true;
}
