#pragma once

#include <drive/backend/database/history/manager.h>
#include <drive/library/cpp/threading/concurrent_cache.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/auto_actualization.h>
#include <rtline/util/types/expected.h>

#include <util/generic/cast.h>


namespace NDrive::NBilling {

    class TAccountLinkDecoder : public TBaseDecoder {
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, AccountId, -1);
        R_FIELD(i32, TypeId, -1);

    public:
        TAccountLinkDecoder() = default;

        TAccountLinkDecoder(const TMap<TString, ui32>& decoderBase) {
            AccountId = GetFieldDecodeIndex("account_id", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            TypeId = GetFieldDecodeIndex("type_id", decoderBase);
        }
    };

    class TAccountLinkRecord {
        R_FIELD(TString, UserId);
        R_FIELD(ui64, AccountId, Max<ui64>());
        R_FIELD(ui32, TypeId, 0);

    public:
        using TDecoder = TAccountLinkDecoder;

        bool DeserializeWithDecoder(const TAccountLinkDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

        NStorage::TTableRecord SerializeToTableRecord() const;
    };

    class TAccounLinksHistory : public TDatabaseHistoryManager<TAccountLinkRecord> {
    private:
        using TBase = TDatabaseHistoryManager<TAccountLinkRecord>;

    public:
        using THistoryEvent = TObjectEvent<TAccountLinkRecord>;

        TAccounLinksHistory(const IHistoryContext& context)
            : TBase(context, "billing_user_accounts_history")
        {}

        TOptionalEvents GetEventsByAccounts(const TSet<ui64>& ids, TRange<TInstant> timestampRange, NDrive::TEntitySession& session) const;
    };

    class TAccountsLinksDB : public IAutoActualization {
    public:
        using TAccountLinks = TVector<TAccountLinkRecord>;

        struct TTimedAccountLinks {
            TAccountLinks Accounts;
            TInstant Timestamp;
        };

    public:
        TAccountsLinksDB(const IHistoryContext& context);

        const TString& GetTableName() const;

        TExpected<TAccountLinks, TString> GetCachedUserAccounts(const TString& userId, TInstant freshness) const;

        bool AddLink(const TString& userId, const ui64 accountId, ui32 typeId, const TString& historyUser, ui32 maxLinks, NDrive::TEntitySession& session) const;
        bool RemoveLink(const TString& userId, const ui64 accountId, const TString& historyUser, NDrive::TEntitySession& session) const;

        TMaybe<TAccountLinks> GetUserAccounts(const TString& userId, NDrive::TEntitySession& session) const;
        TMaybe<TAccountLinks> GetUsersAccounts(const TSet<TString>& userIds, NDrive::TEntitySession& session) const;
        TMaybe<TSet<ui64>> FilterAccounts(const TSet<ui64>& ids, bool used, NDrive::TEntitySession& session) const;

        TMaybe<TAccountLinks> GetLinkAccounts(NSQL::TQueryOptions&& options, NDrive::TEntitySession& session) const;
        TMaybe<TAccountLinks> GetAccountsUsers(const TSet<ui64>& ids, NDrive::TEntitySession& session) const;

        const TAccounLinksHistory& GetHistoryManager() const {
            return HistoryManager;
        }

    private:
        bool Refresh() override;

    private:
        TDatabasePtr Database;
        TAccounLinksHistory HistoryManager;
        static TString TableName;

        const TDuration DefaultLifetime;
        IBaseSequentialTableImpl::TOptionalEventId LastEventId = 0;

        TUnistatSignal<double> CacheHit;
        TUnistatSignal<double> CacheMiss;
        TUnistatSignal<double> CacheExpired;
        TUnistatSignal<double> CacheInvalidated;
        mutable NUtil::TConcurrentCache<TString, TTimedAccountLinks> ObjectCache;
    };
}
