#pragma once

#include "account_description.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/time_restriction/time_restriction.h>

#include <util/generic/cast.h>


namespace NDrive::NBilling {

    class TAccountRecordDecoder : public TBaseDecoder {
        R_FIELD(i32, AccountId, -1);
        R_FIELD(i32, TypeId, -1);
        R_FIELD(i32, Version, -1);
        R_FIELD(i32, Spent, -1);
        R_FIELD(i32, CreatedAt, -1);
        R_FIELD(i32, Active, -1);
        R_FIELD(i32, Meta, -1);
        R_FIELD(i32, ExternalId, -1);
        R_FIELD(i32, SpravId, -1);

    public:
        TAccountRecordDecoder() = default;

        TAccountRecordDecoder(const TMap<TString, ui32>& decoderBase) {
            AccountId = GetFieldDecodeIndex("account_id", decoderBase);
            TypeId = GetFieldDecodeIndex("type_id", decoderBase);
            Version = GetFieldDecodeIndex("version", decoderBase);
            Spent = GetFieldDecodeIndex("spent", decoderBase);
            CreatedAt = GetFieldDecodeIndex("created_at", decoderBase);
            Active = GetFieldDecodeIndex("active", decoderBase);
            Meta = GetFieldDecodeIndex("account_meta", decoderBase);
            ExternalId = GetFieldDecodeIndex("external_id", decoderBase);
            SpravId = GetFieldDecodeIndex("sprav_id", decoderBase);
        }
    };

    class TLimitedBalance {
        R_FIELD(ui64, Balance, 0);
        R_FIELD(TInstant, Deadline, TInstant::Max());
        R_FIELD(TString, Source);
    public:
        TLimitedBalance() = default;
        TLimitedBalance(ui64 balance, TInstant deadline, const TString& source = {})
            : Balance(balance)
            , Deadline(deadline)
            , Source(source)
        {}

        bool operator <(const TLimitedBalance& balance) const {
            if (Deadline != balance.GetDeadline()) {
                return Deadline < balance.GetDeadline();
            }
            return Source < balance.GetSource();
        }

        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "deadline", Deadline, false) && NJson::ParseField(json, "sum", Balance, true) && NJson::ParseField(json, "source", Source, false);
        }

        NJson::TJsonValue ToJson() const {
            NJson::TJsonValue result;
            result.InsertValue("sum", Balance);
            if (Source) {
                result.InsertValue("source", Source);
            }
            if (Deadline != TInstant::Max()) {
                result.InsertValue("deadline", Deadline.Seconds());
            }
            return result;
        }

        static NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/) {
            NDrive::TScheme scheme;
            scheme.Add<TFSNumeric>("deadline", "Окончание действия").SetVisual(TFSNumeric::EVisualType::DateTime);
            scheme.Add<TFSNumeric>("sum", "Сумма").SetMin(0);
            scheme.Add<TFSString>("source", "Источник");
            return scheme;
        }
    };

    using TLimitedBalances = TSet<TLimitedBalance>;

    class TAccountRecord {
        R_READONLY(ui32, AccountId, Max<ui32>());
        R_FIELD(ui32, TypeId, Max<ui32>());
        R_READONLY(ui32, Version, 0);
        R_FIELD(ui64, Spent, 0);
        R_READONLY(TInstant, CreatedAt, TInstant::Now());
        R_READONLY(TString, Comment);
        R_READONLY(TString, Company);
        R_FIELD(bool, Active, false);
        R_FIELD(TString, ExternalId);
        R_FIELD(TString, SpravId);

    public:
        using TPtr = TAtomicSharedPtr<TAccountRecord>;

        bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/) {
            NJson::TJsonValue meta(NJson::JSON_MAP);
            if (!record.TryGet("account_id", AccountId)
                || !record.TryGet("type_id", TypeId)
                || !record.TryGet("spent", Spent)
                || !record.TryGet("version", Version)
                || !record.TryGet("created_at", CreatedAt)
                || !record.TryGet("account_meta", meta)
                || !record.TryGet("active", Active)) {
                return false;
            }
            Comment = meta["comment"].GetString();
            if (!ParseMeta(meta)) {
                return false;
            }
            if (record.Has("external_id")) {
                ExternalId = record.Get("external_id");
            }
            if (record.Has("sprav_id")) {
                SpravId = record.Get("sprav_id");
            }
            return true;
        }

        bool DeserializeWithDecoder(const TAccountRecordDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
            READ_DECODER_VALUE(decoder, values, AccountId);
            READ_DECODER_VALUE(decoder, values, TypeId);
            READ_DECODER_VALUE(decoder, values, Version);
            READ_DECODER_VALUE(decoder, values, Spent);
            READ_DECODER_VALUE_INSTANT(decoder, values, CreatedAt);
            READ_DECODER_VALUE(decoder, values, Active);

            if (decoder.GetExternalId() != -1) {
                READ_DECODER_VALUE(decoder, values, ExternalId);
            }

            if (decoder.GetSpravId() != -1) {
                READ_DECODER_VALUE(decoder, values, SpravId);
            }

            NJson::TJsonValue meta(NJson::JSON_MAP);
            READ_DECODER_VALUE_JSON(decoder, values, meta, Meta);

            if (!ParseMeta(meta)) {
                return false;
            }
            return true;
        }

        virtual bool ParseMeta(const NJson::TJsonValue& meta) {
            Comment = meta["comment"].GetString();
            Company = meta["company"].GetString();
            return true;
        }

        virtual bool PatchMeta(const NJson::TJsonValue& meta) {
            if (meta.Has("comment")) {
                Comment = meta["comment"].GetString();
            }
            if (meta.Has("company")) {
                Company = meta["company"].GetString();
            }
            return true;
        }

        virtual void SaveMeta(NJson::TJsonValue& meta) const {
            meta["comment"] = Comment;
            meta["company"] = Company;
        }

        virtual NJson::TJsonValue GetReport() const {
            NJson::TJsonValue report;
            SaveMeta(report);
            if (ExternalId) {
                report["external_id"] = ExternalId;
            }
            
            return report;
        }

        NStorage::TTableRecord SerializeToTableRecord() const {
            NStorage::TRecordBuilder builder;
            builder("type_id", TypeId)("version", Version)("spent", Spent)("created_at", CreatedAt.Seconds())("active", Active)("external_id", ExternalId)("sprav_id", SpravId);
            NJson::TJsonValue meta(NJson::JSON_MAP);
            SaveMeta(meta);
            builder("account_meta", meta);

            if (AccountId != Max<ui32>()) {
                builder("account_id", AccountId);
            }
            return builder;
        }

        virtual ~TAccountRecord() {}

        NDrive::TScheme GetScheme(const NDrive::IServer* server) const {
            NDrive::TScheme result = DoGetScheme(server);
            result.Add<TFSString>("comment", "Комментарий").SetRequired(false);
            result.Add<TFSString>("company", "Компания").SetRequired(false);
            return result;
        }

        virtual EAccount GetAccountType() const {
            return EAccount::Wallet;
        }

        virtual ui64 GetHardLimit() const {
            return 0;
        }

        virtual ui64 GetSoftLimit() const {
            return 0;
        }

        virtual TMaybe<TString> GetOffersFilter() const {
            return {};
        }
        virtual TMaybe<TString> GetOffersFilterName() const {
            return {};
        }
        virtual TMaybe<TString> GetInsuranceDescriptionFilter() const {
            return {};
        }
        virtual TMaybe<TTimeRestrictionsPool<class TTimeRestriction>> GetTimeRestrictions() const {
            return {};
        }
        virtual TMaybe<bool> EnableTollRoadsPay() const {
            return {};
        }
        virtual TMaybe<TVector<TString>> GetTollRoadsToPayFor() const {
            return {};
        }

        using TFactory = NObjectFactory::TObjectFactory<TAccountRecord, EWalletDataType>;
        using TDecoder = TAccountRecordDecoder;

    protected:
        virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const = 0;
    };

    class IAccountsContext : public IHistoryContext {
    public:
        virtual TMaybe<TAccountDescriptionRecord> GetDescriptionById(ui32 typeId, TInstant lastInstant) const = 0;
    };

    class TAccountData {
    public:
        bool DeserializeWithDecoder(const TAccountRecordDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
            CHECK_WITH_LOG(hContext);
            const IAccountsContext* context = VerifyDynamicCast<const IAccountsContext*>(hContext);
            ui32 typeId = 0;
            READ_DECODER_VALUE_TEMP(decoder, values, typeId, TypeId);
            auto description = context->GetDescriptionById(typeId, TInstant::Zero());
            if (!description.Defined()) {
                ERROR_LOG << "No description for " << typeId << Endl;
                return false;
            }

            Record = TAccountRecord::TFactory::Construct(description->GetDataType());
            if (!Record || !Record->DeserializeWithDecoder(decoder, values, hContext)) {
                return false;
            }
            if (description->GetType() != Record->GetAccountType()) {
                ERROR_LOG << "Incorrect AccountType " << description->GetType() << " for " << description->GetName() << Endl;
                return false;
            }
            return true;
        }

        ui32 GetAccountId() const {
            CHECK_WITH_LOG(Record);
            return Record->GetAccountId();
        }

        NStorage::TTableRecord SerializeToTableRecord() const {
            CHECK_WITH_LOG(Record);
            return Record->SerializeToTableRecord();
        }

        const TAccountRecord::TPtr GetRecord() const {
            CHECK_WITH_LOG(Record);
            return Record;
        }

        void DoBuildReportItem(NJson::TJsonValue& json) const {
            CHECK_WITH_LOG(Record);
            json.InsertValue("type_id", Record->GetTypeId());
            json.InsertValue("version", Record->GetVersion());
            json.InsertValue("spent", Record->GetSpent());
            json.InsertValue("created_at", Record->GetCreatedAt().Seconds());
            json.InsertValue("active", Record->GetActive());
            json.InsertValue("external_id", Record->GetExternalId());
            if (Record->GetSpravId()) {
                json.InsertValue("sprav_id", Record->GetSpravId());
            }
            NJson::TJsonValue meta;
            Record->SaveMeta(meta);
            json.InsertValue("account_meta", meta);
        }

        using TDecoder = TAccountRecordDecoder;

    private:
        TAccountRecord::TPtr Record;
    };

    class TAccountsHistoryManager : public TDatabaseHistoryManager<TAccountData> {
    private:
        using TBase = TDatabaseHistoryManager<TAccountData>;

    public:
        TAccountsHistoryManager(const IHistoryContext& context)
            : TBase(context, "billing_account_history")
            , HistoryContext(context) {}

        const IHistoryContext& GetContext() const {
            return HistoryContext;
        }

        TOptionalEvents GetEventsByAccountId(const ui32 id, TRange<TInstant> timestampRange, const TMaybe<TString>& historyComment, NDrive::TEntitySession& session, ui64 limit = 0, ui64 offset = 0) const;
    private:
        const IHistoryContext& HistoryContext;
    };

    class TAccountsStorage : public TAutoActualizingSnapshot<TAccountData, ui32>, public IStartStopProcess {
    public:
        using THistoryManager = TAccountsHistoryManager;
        using TAccountsHistoryReader = TCallbackSequentialTableImpl<TObjectEvent<TAccountData>, ui32>;

    private:
        using TBase = TAutoActualizingSnapshot<TAccountData, ui32>;

    public:
        TAccountsStorage(const IHistoryContext& context, const THistoryConfig& hConfig)
            : TBase("billing_account", MakeAtomicShared<TAccountsHistoryReader>(context, "billing_account_history", hConfig))
            , HistoryManager(MakeHolder<TAccountsHistoryManager>(context)) {}

        virtual bool DoStart() override {
            return RebuildCache()
                && HistoryReader->Start();
        }

        virtual bool DoStop() override {
            return HistoryReader->Stop();
        }

    protected:
        virtual bool DoRebuildCacheUnsafe() const override {
            auto session = NDrive::TEntitySession(HistoryManager->GetDatabase().CreateTransaction(true));
            auto table = HistoryManager->GetDatabase().GetTable("billing_account");

            NStorage::TObjectRecordsSet<TAccountData, IHistoryContext> records(&(HistoryManager->GetContext()));
            auto reqResult = table->GetRows("", records, session.GetTransaction());
            if (!reqResult->IsSucceed()) {
                ERROR_LOG << session.GetTransaction()->GetErrors().GetStringReport() << Endl;
                return false;
            }

            for (auto&& account : records) {
                auto result = Objects.emplace(account.GetRecord()->GetAccountId(), account);
                AccountsByType[account.GetRecord()->GetTypeId()].emplace(account.GetRecord()->GetAccountId(), &result.first->second);
            }
            return true;
        }

        bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TAccountData>>& dbEvent, const bool isNewEvent) override {
            if (isNewEvent) {
                if (dbEvent->GetHistoryAction() == EObjectHistoryAction::Remove) {
                    auto it = AccountsByType.find(dbEvent->GetRecord()->GetTypeId());
                    if (it != AccountsByType.end()) {
                        it->second.erase(dbEvent->GetAccountId());
                    }
                    Objects.erase(dbEvent->GetRecord()->GetAccountId());

                } else {
                    TAccountData& accData = Objects[dbEvent->GetRecord()->GetAccountId()];
                    accData = *dbEvent;
                    AccountsByType[dbEvent->GetRecord()->GetTypeId()].emplace(dbEvent->GetRecord()->GetAccountId(), &accData);
                }
            }
            return true;
        }

        ui32 GetEventObjectId(const TObjectEvent<TAccountData>& ev) const override {
            return ev.GetRecord()->GetAccountId();
        }

    public:
        template <class TFunc>
        bool ListAccountsByTypeId(const ui32 typeId, TFunc& func, const TInstant actuality) const {
            if (!RefreshCache(actuality)) {
                return false;
            }

            auto g = MakeObjectReadGuard();
            auto it = AccountsByType.find(typeId);
            if (it != AccountsByType.end()) {
                for (auto&& data : it->second) {
                    func(*data.second);
                }
            }
            return true;
        }

        const THistoryManager& GetHistoryManager() const {
            CHECK_WITH_LOG(HistoryManager);
            return *HistoryManager;
        }

    private:
        THolder<THistoryManager> HistoryManager;
        using TBase::Objects;
        mutable TMap<ui32, TMap<ui32, TAccountData*>> AccountsByType;
    };
}
