#include "compiled.h"
#include "payments.h"

#include <drive/backend/proto/snapshot.pb.h>

#include <library/cpp/string_utils/base64/base64.h>

namespace NDrive::NBilling {
    bool TFiscalItem::FromProto(const NDrive::NProto::TFiscalItem& itemProto) {
        AccountId = itemProto.GetAccountId();
        Sum = itemProto.GetSum();
        UniqueName = itemProto.GetUniqueName();
        Name = itemProto.GetName();
        TransactionId = itemProto.GetTransactionId();
        TString typeStr = itemProto.GetType();
        return TryFromString(typeStr, Type);
    }

    void TFiscalItem::ToProto(NDrive::NProto::TFiscalItem& itemProto) const {
        itemProto.SetAccountId(AccountId);
        itemProto.SetSum(Sum);
        itemProto.SetType(::ToString(Type));
        itemProto.SetUniqueName(UniqueName);
        itemProto.SetName(Name);
        itemProto.SetTransactionId(TransactionId);
    }

    void TFiscalDetails::ToProto(NDrive::NProto::TCompiledBillMeta& proto) const {
        for (auto&& item : Items) {
            auto itemProto = proto.AddItems();
            item.ToProto(*itemProto);
        }
    }

    bool TFiscalDetails::FromProto(const NDrive::NProto::TCompiledBillMeta& proto) {
        for (auto&& itemProto : proto.GetItems()) {
            TFiscalItem item;
            if (!item.FromProto(itemProto)) {
                return false;
            }
            Items.emplace_back(std::move(item));
        }
        return true;
    }

    TCompiledImplDecoder::TCompiledImplDecoder(const TMap<TString, ui32>& decoderBase, const bool strict /*= true*/) : TBaseDecoder(strict) {
        SessionId = GetFieldDecodeIndex("session_id", decoderBase);
        UserId = GetFieldDecodeIndex("user_id", decoderBase);
        Bill = GetFieldDecodeIndex("bill", decoderBase);
        BillingType = GetFieldDecodeIndex("billing_type", decoderBase);
        Details = GetFieldDecodeIndex("details", decoderBase);
        BillingWallets = GetFieldDecodeIndex("billing_wallets", decoderBase);
    }

    bool TCompiledImpl::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, SessionId);
        if (decoder.GetUserId() > 0) {
            READ_DECODER_VALUE_DEF(decoder, values, UserId, "");
        }
        READ_DECODER_VALUE(decoder, values, Bill);
        READ_DECODER_VALUE(decoder, values, BillingType);

        if (decoder.GetBillingWallets() > 0) {
            TString walletsStr;
            READ_DECODER_VALUE_TEMP(decoder, values, walletsStr, BillingWallets);
            if (walletsStr) {
                BillingWallets = StringSplitter(walletsStr).SplitBySet(",").ToList<TString>();
            }
        }

        TString encodedDetails;
        READ_DECODER_VALUE_TEMP(decoder, values, encodedDetails, Details);
        NDrive::NProto::TCompiledBillMeta proto;
        if (!proto.ParseFromString(Base64Decode(encodedDetails))) {
            return false;
        }
        if (!ParseMetaProto(proto)) {
            return false;
        }
        return Details.FromProto(proto);
    }

    NStorage::TTableRecord TCompiledImpl::SerializeToTableRecord() const {
        NStorage::TRecordBuilder builder;
        builder("session_id", SessionId)("bill", Bill)("billing_type", BillingType);
        if (UserId) {
            builder("user_id", UserId);
        }
        if (auto billingWallets = GetBillingWallets()) {
            builder("billing_wallets", JoinSeq(",", billingWallets));
        }
        NDrive::NProto::TCompiledBillMeta detailsProto;
        Details.ToProto(detailsProto);
        SaveMetaProto(detailsProto);
        builder("details", Base64Encode(detailsProto.SerializeAsString()));
        return builder;
    }

    bool TCompiledImpl::Check() const {
        ui64 itemsSum = 0;
        for (auto&& item : Details.GetItems()) {
            itemsSum += item.GetSum();
        }
        return itemsSum == Bill;
    }

    TVector<TString> TCompiledImpl::GetBillingWallets() const {
        if (BillingWallets) {
            return *BillingWallets;
        }
        TVector<TString> result;
        for (auto&& item : Details.GetItems()) {
            if (item.GetType() == EAccount::Wallet) {
                result.push_back(item.GetUniqueName());
            }
        }
        return result;
    }

    bool TCompiledImpl::ParseMetaProto(const NDrive::NProto::TCompiledBillMeta& detailsProto) {
        RealSessionId = detailsProto.GetRealSessionId();
        Comment = detailsProto.GetComment();
        return true;
    }

    void TCompiledImpl::SaveMetaProto(NDrive::NProto::TCompiledBillMeta& detailsProto) const {
        detailsProto.SetRealSessionId(RealSessionId);
        detailsProto.SetComment(Comment);
    }
}

TSet<TString> TCompiledBill::GetAccounts() const {
    TSet<TString> result;
    for (auto&& item : GetDetails().GetItems()) {
        result.emplace(item.GetUniqueName());
    }
    return result;
}

bool TCompiledBill::HasAccount(TStringBuf name) const {
    for (auto&& item : GetDetails().GetItems()) {
        if (item.GetUniqueName() == name) {
            return true;
        }
    }
    return false;
}

bool TCompiledBill::ParseMetaProto(const NDrive::NProto::TCompiledBillMeta& detailsProto) {
    LastPaymentId = detailsProto.GetLastPaymentId();
    if (detailsProto.HasFinal()) {
        Final = detailsProto.GetFinal();
    }
    if (detailsProto.HasCashback()) {
        Cashback = detailsProto.GetCashback();
    }
    return TBase::ParseMetaProto(detailsProto);
}

void TCompiledBill::SaveMetaProto(NDrive::NProto::TCompiledBillMeta& detailsProto) const {
    TBase::SaveMetaProto(detailsProto);
    detailsProto.SetLastPaymentId(LastPaymentId);
    detailsProto.SetFinal(Final);
    detailsProto.SetCashback(Cashback);
}

NJson::TJsonValue TCompiledBill::GetReport() const {
    NJson::TJsonValue report(NJson::JSON_ARRAY);
    for (auto&& item : GetDetails().GetItems()) {
        NJson::TJsonValue& accountReport = report.AppendValue(NJson::JSON_MAP);
        accountReport.InsertValue("account_id", item.GetAccountId());
        accountReport.InsertValue("name", item.GetName());
        accountReport.InsertValue("unique_name", item.GetUniqueName());
        accountReport.InsertValue("type", ::ToString(item.GetType()));
        accountReport.InsertValue("sum", item.GetSum());
        accountReport.InsertValue("transaction_id", item.GetTransactionId());
    }
    return report;
}

NJson::TJsonValue TCompiledBill::GetFullReport() const {
    NJson::TJsonValue report;
    report.InsertValue("items", GetReport());
    report.InsertValue("final", Final);
    report.InsertValue("cashback", Cashback);
    return report;
}

TMap<ui32, ui32> TCompiledBill::GetSumByAccounts() const {
    TMap<ui32, ui32> sumByAccountId;
    for (const auto& accountSum : GetDetails().GetItems()) {
        auto it = sumByAccountId.find(accountSum.GetAccountId());
        if (it == sumByAccountId.end()) {
            sumByAccountId[accountSum.GetAccountId()] = accountSum.GetSum();
        } else {
            it->second += accountSum.GetSum();
        }
    }
    return sumByAccountId;
}

bool TCompiledBill::AddBillItem(const TObjectEvent<TCompiledBill>& item, NDrive::TInfoEntitySession& session) {
    if (TBillingGlobals::CashbackBillingTypes.contains(item.GetBillingType())) {
        MutableCashback() += item.GetCashback();
    } else {
        if (GetFinal()) {
            session.SetErrorInfo("AddBill", "Already full " + GetSessionId() + " " + item.GetReport().GetStringRobust());
            return false;
        }
        SetFinal(item.GetFinal());
        SetFinalTime(Max<TInstant>(GetFinalTime(), item.GetHistoryInstant()));
    }
    if (!GetRealSessionId() && item.GetRealSessionId()) {
        if (GetSessionId()) {
            session.SetErrorInfo("AddBill", "Inconsistent real session " + GetSessionId() + " " + item.GetReport().GetStringRobust());
            return false;
        }
        SetRealSessionId(item.GetRealSessionId());
    }
    if (!GetUserId() && item.GetUserId()) {
        if (GetSessionId()) {
            session.SetErrorInfo("AddBill", "Inconsistent user " + GetSessionId() + " " + item.GetReport().GetStringRobust());
            return false;
        }
        SetUserId(item.GetUserId());
    }
    if (!TBillingGlobals::CashbackBillingTypes.contains(item.GetBillingType())) {
        if (item.GetLastPaymentId() < GetLastPaymentId()) {
            session.SetErrorInfo("AddBill", "Inconsistent payments " + GetSessionId() + " " + item.GetReport().GetStringRobust());
            return false;
        }
        SetLastPaymentId(item.GetLastPaymentId());
    }

    if (!GetSessionId()) {
        if (!item.GetSessionId()) {
            session.SetErrorInfo("AddBill", "SessionId absent " + item.GetReport().GetStringRobust());
            return false;
        }
        SetSessionId(item.GetSessionId());
        SetBillingType(item.GetBillingType());
    }
    if (item.GetComment()) {
        if (GetComment()) {
            MutableComment() += "\n";
        }
        MutableComment() += item.GetComment();
    }
    MutableBill() += item.GetBill();
    TMap<ui32, NDrive::NBilling::TFiscalItem> sumByAccountId;
    for (const auto& accountSum : GetDetails().GetItems()) {
        auto it = sumByAccountId.find(accountSum.GetAccountId());
        if (it == sumByAccountId.end()) {
            sumByAccountId[accountSum.GetAccountId()] = accountSum;
        } else {
            it->second.MutableSum() += accountSum.GetSum();
        }
    }
    for (const auto& accountSum : item.GetDetails().GetItems()) {
        auto it = sumByAccountId.find(accountSum.GetAccountId());
        if (it == sumByAccountId.end()) {
            sumByAccountId[accountSum.GetAccountId()] = accountSum;
        } else {
            it->second.MutableSum() += accountSum.GetSum();
        }
    }
    MutableDetails().SetItems(MakeVector(NContainer::Values(sumByAccountId)));
    if (!Check()) {
        session.SetErrorInfo("AddBill", "Inconsistent sum " + GetSessionId() + " " + item.GetReport().GetStringRobust());
        return false;
    }
    return true;
}
