#pragma once

#include "account.h"
#include "entities.h"

#include <rtline/library/storage/structured.h>

namespace NDrive::NProto {
    class TCompiledBillMeta;
    class TFiscalItem;
}

namespace NDrive::NBilling {
    class TFiscalItem {
    public:
        R_FIELD(ui32, AccountId, 0);
        R_FIELD(ui32, Sum, 0);
        R_FIELD(EAccount, Type, EAccount::Wallet);
        R_FIELD(TString, UniqueName);
        R_FIELD(TString, Name);
        R_FIELD(TString, TransactionId);

    public:
        bool FromProto(const NDrive::NProto::TFiscalItem& itemProto);
        void ToProto(NDrive::NProto::TFiscalItem& itemProto) const;
    };

    class TFiscalDetails {
    public:
        void ToProto(NDrive::NProto::TCompiledBillMeta& proto) const;
        bool FromProto(const NDrive::NProto::TCompiledBillMeta& proto);

        TFiscalDetails& AddItem(TFiscalItem&& item) {
            Items.emplace_back(item);
            return *this;
        }

        R_FIELD(TVector<TFiscalItem>, Items);
    };

    class TCompiledImplDecoder : public TBaseDecoder {
    public:
        R_FIELD(i32, SessionId, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, Bill, -1);
        R_FIELD(i32, BillingType, -1);
        R_FIELD(i32, Details, -1);
        R_FIELD(i32, BillingWallets, -1);

    public:
        TCompiledImplDecoder() = default;
        TCompiledImplDecoder(const TMap<TString, ui32>& decoderBase, const bool strict = true);
    };

    class TCompiledImpl {
    public:
        using TDecoder = TCompiledImplDecoder;

    public:
        virtual ~TCompiledImpl() = default;

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
        NStorage::TTableRecord SerializeToTableRecord() const;

        bool Check() const;
        TVector<TString> GetBillingWallets() const;

    protected:
        virtual bool ParseMetaProto(const NDrive::NProto::TCompiledBillMeta& detailsProto);
        virtual void SaveMetaProto(NDrive::NProto::TCompiledBillMeta& detailsProto) const;

    public:
        R_FIELD(TString, SessionId);
        R_FIELD(TString, UserId);
        R_FIELD(TString, RealSessionId);
        R_FIELD(TString, Comment);
        R_FIELD(ui32, Bill, 0);
        R_FIELD(EBillingType, BillingType, EBillingType::CarUsage);
        R_FIELD(NDrive::NBilling::TFiscalDetails, Details);
        R_FIELD(TInstant, FinalTime);
        R_OPTIONAL(TVector<TString>, BillingWallets);
    };

    template<class TCompiled>
    bool GetEventsFromDB(const TInstant since, const TInstant until, TVector<TObjectEvent<NDrive::NBilling::TCompiledImpl>>& result, NDrive::TEntitySession& session) {
        auto table = session->GetDatabase().GetTable(TCompiled::GetTableName());
        NStorage::TObjectRecordsSet<TObjectEvent<NDrive::NBilling::TCompiledImpl>> records;
        auto reqResult = table->GetRows("history_timestamp >= " + ::ToString(since.Seconds()) + " AND history_timestamp < " + ::ToString(until.Seconds()), records, session.GetTransaction());
        if (!reqResult->IsSucceed()) {
            ERROR_LOG << session.GetTransaction()->GetErrors().GetStringReport() << Endl;
            return false;
        }
        result = records.GetObjects();
        return true;
    }
}

class TCompiledBill : public NDrive::NBilling::TCompiledImpl {
public:
    using TBase = NDrive::NBilling::TCompiledImpl;
    using TDecoder = TBase::TDecoder;

public:
    R_FIELD(ui64, LastPaymentId, 0);
    R_FIELD(bool, Final, true);
    R_FIELD(ui64, Cashback, 0);

public:
    TCompiledBill() = default;

    TSet<TString> GetAccounts() const;
    bool HasAccount(TStringBuf name) const;

    virtual bool ParseMetaProto(const NDrive::NProto::TCompiledBillMeta& detailsProto);
    virtual void SaveMetaProto(NDrive::NProto::TCompiledBillMeta& detailsProto) const;
    NJson::TJsonValue GetReport() const;
    NJson::TJsonValue GetFullReport() const;

    static TString GetTableName() {
        return "compiled_bills";
    }

    bool AddBillItem(const TObjectEvent<TCompiledBill>& item, NDrive::TInfoEntitySession& session);
    TMap<ui32, ui32> GetSumByAccounts() const;
};

class TCompiledRefund : public NDrive::NBilling::TCompiledImpl {
public:
    using TBase = NDrive::NBilling::TCompiledImpl;
    using TDecoder = TBase::TDecoder;

public:
    TCompiledRefund() = default;

    static TString GetTableName() {
        return "compiled_refunds";
    }
};
