#include "db_accessors.h"

TOptionalBillingTask NDrive::NBilling::TActiveTasksManager::GetTask(const TString& sessionId, NDrive::TEntitySession& session) const {
    auto optionalTasks = GetSessionsTasks({ sessionId }, session);
    if (!optionalTasks) {
        return {};
    }
    if (optionalTasks->size() > 1) {
        session.SetErrorInfo("BillingManager::GetBillingTask", "multiple billing tasks exist");
        return {};
    }
    if (!optionalTasks->empty()) {
        return std::move(optionalTasks->front());
    } else {
        return TBillingTask();
    }
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::GetUsersTasks(TConstArrayRef<TString> users, NDrive::TEntitySession& session) const {
    return GetTasks("user_id", users, session);
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::GetSessionsTasks(TConstArrayRef<TString> sessions, NDrive::TEntitySession& session) const {
    return GetTasks("session_id", sessions, session);
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::GetTasks(const TString& field, TConstArrayRef<TString> ids, NDrive::TEntitySession& session) const {
    TStringStream condition;
    if (ids.empty()) {
        return TBillingTasks();
    }
    condition << field << " IN (" << session->Quote(ids) << ")";
    return GetTasks(session, condition.Str());
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::GetTasks(NDrive::TEntitySession& session, const TString& condition) const {
    if (!Table) {
        session.SetErrorInfo("GetTasks", "incorrect table entity", EDriveSessionResult::InternalError);
        return {};
    }
    NStorage::TObjectRecordsSet<TBillingTask> records;
    auto queryResult = Table->GetRows(condition, records, session.GetTransaction());
    if (!queryResult || !queryResult->IsSucceed()) {
        session.SetErrorInfo("GetTasks", "cannot execute query", EDriveSessionResult::TransactionProblem);
        return {};
    }
    return records.DetachObjects();
}

bool NDrive::NBilling::TActiveTasksManager::BoostSession(const TString& sessionId, const TString& historyUserId, const TBillingTask::TExecContext& context, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord condition;
    condition.Set("session_id", sessionId);
    auto tasks = BoostTasks(std::move(condition), historyUserId, context, session);
    if (!tasks) {
        return false;
    }
    if (tasks->empty()) {
        session.SetErrorInfo("BoostSession", "cannot find task " + sessionId, EDriveSessionResult::IncorrectRequest);
        return false;
    }
    return true;
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::BoostUserSessions(const TString& userId, const TString& historyUserId, const TBillingTask::TExecContext& context, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord condition;
    condition.Set("user_id", userId);
    return BoostTasks(std::move(condition), historyUserId, context, session);
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::BoostTasks(NStorage::TTableRecord&& condition, const TString& historyUserId, const TBillingTask::TExecContext& context, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord update;
    update.Set("last_status_update", 0);
    update.Set("exec_context", context.Serialize().GetStringRobust());
    update.Set("event_id", "nextval('billing_tasks_event_id_seq')");
    return UpdateWithHistory(std::move(condition), std::move(update), historyUserId, session);
}

bool NDrive::NBilling::TActiveTasksManager::CancelTask(const TString& sessionId, const TString& historyUserId, NDrive::TEntitySession& session, ui32 cancelSum) const {
    auto restored = GetTask(sessionId, session);
    if (!restored) {
        return false;
    }
    if (!restored.GetRef()) {
        session.SetErrorInfo("CancelTask", "Unknown session", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    return CancelTask(*restored, historyUserId, session, cancelSum);
}

bool NDrive::NBilling::TActiveTasksManager::CancelTask(const TBillingTask& task, const TString& historyUserId, NDrive::TEntitySession& session, ui32 cancelSum) const {
    if (!task) {
        session.SetErrorInfo("CancelTask", "Unknown session", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    if (!task.IsFinished()) {
        session.SetErrorInfo("CancelTask", "Can't cancel not finished task", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    if (!Table) {
        session.SetErrorInfo("CancelTask", "incorrect table entity", EDriveSessionResult::InternalError);
        return false;
    }

    NStorage::TTableRecord tCond;
    tCond.Set("session_id", task.GetId());
    tCond.Set("state", "finished");

    NStorage::TTableRecord update;
    update.Set("state", "canceled");
    update.Set("event_id", "nextval('billing_tasks_event_id_seq')");
    if (cancelSum) {
        update.Set("bill", "LEAST(bill," + ::ToString(cancelSum) + ")");
    }
    auto tasks = UpdateWithHistory(std::move(tCond), std::move(update), historyUserId, session);
    if (!tasks) {
        return false;
    }
    if (tasks->empty()) {
        session.SetErrorInfo("CancelTask", "cannot find task " + task.GetId(), EDriveSessionResult::IncorrectRequest);
        return false;
    }
    return true;
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::Update(const NStorage::TTableRecord& condition, const NStorage::TTableRecord& update, NDrive::TEntitySession& session) const {
    if (!Table) {
        session.SetErrorInfo("UpdateWithHistory", "incorrect table entity", EDriveSessionResult::InternalError);
        return {};
    }
    NStorage::TObjectRecordsSet<TBillingTask> affected;
    auto queryResult = Table->UpdateRow(condition, update, session.GetTransaction(), &affected);
    if (!queryResult || !queryResult->IsSucceed()) {
        return {};
    }
    return affected.DetachObjects();
}

TOptionalBillingTasks NDrive::NBilling::TActiveTasksManager::UpdateWithHistory(const NStorage::TTableRecord& condition, const NStorage::TTableRecord& update, const TString& historyUserId, NDrive::TEntitySession& session) const {
    auto tasks = Update(condition, update, session);
    if (!tasks) {
        return {};
    }
    if (!HistoryManager.AddHistory(*tasks, historyUserId, EObjectHistoryAction::UpdateData, session)) {
        return {};
    }
    return tasks;
}


TRefundsDB::TRefundsDB(const IHistoryContext& context)
    : IBaseSequentialTableImpl(context, "drive_refunds")
{
}

bool TRefundsDB::GetSessionRefunds(const TString& sessionId, TVector<TRefundTask>& refunds, NDrive::TEntitySession& session) const {
    TMap<TString, TVector<TRefundTask>> result;
    if (!GetSessionsRefunds(NContainer::Scalar(sessionId), result, session)) {
        return false;
    }
    refunds = result[sessionId];
    return true;
}

bool TRefundsDB::GetSessionsRefunds(TConstArrayRef<TString> sessionIds, TMap<TString, TVector<TRefundTask>>& refunds, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = GetDatabase().GetTable(GetTableName());
    NStorage::TObjectRecordsSet<TRefundTask> records;
    if (!table->GetRows("session_id in ('" + JoinStrings(sessionIds.begin(), sessionIds.end(), "','") + "')", records, session.GetTransaction())->IsSucceed()) {
        return false;
    }
    for (auto&& task : records) {
        refunds[task.GetSessionId()].emplace_back(std::move(task));
    }
    return true;
}

TVector<TRefundTask> TRefundsDB::GetRefundsFromDB(const TSet<TString>& statuses, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = GetDatabase().GetTable(GetTableName());
    NStorage::TObjectRecordsSet<TRefundTask> records;
    TStringStream request;
    request << "status IN ('" << JoinStrings(statuses.begin(), statuses.end(), "','") << "')";
    table->GetRows(request.Str(), records, session.GetTransaction());
    return records.GetObjects();
}
