#include "entities.h"

#include <util/string/vector.h>
#include <drive/backend/abstract/base.h>

NJson::TJsonValue GetPaymentMethodUserReport(const NDrive::NTrustClient::TPaymentMethod& payMethod, ELocalization locale, const IServerBase& server) {
    const auto& settings = server.GetSettings();
    NJson::TJsonValue json;
    if (payMethod.GetSuffix()) {
        json.InsertValue("name", payMethod.GetSuffix());
    } else {
        json.InsertValue("name", payMethod.GetAccount());
    }
    json.InsertValue("group_type", ::ToString(EPaymentGroupType::Radio));
    json["request_parameters"].InsertValue("card", payMethod.GetId());
    json.InsertValue("type", "credit_card");
    json.InsertValue("payment_system", payMethod.GetPSystem());
    TString icon;
    auto local = server.GetLocalization();
    if (payMethod.OptionalPayerInfo() && payMethod.OptionalPayerInfo()->OptionalFamilyInfo()) {
        auto& familyInfo = payMethod.OptionalPayerInfo()->GetFamilyInfoRef();
        bool unlimited = (familyInfo.GetLimit() >= settings.GetValueDef<i64>("billing.family_card_max_limit", 1000000000));
        i64 balance = Max<i64>(0, familyInfo.GetLimit() - familyInfo.GetExpenses()) / 100;
        auto amount = std::abs(balance);
        auto balanceStr = (balance < 0 ? "-" : "") + (local ? local->FormatNumeral(amount, locale, true, " ", "units.full." + to_lower(familyInfo.GetCurrency()), "", "", "") : ToString(amount));
        auto getLocal = [&](const TString& key, const TString& defaultStr) -> TString {
            auto localKey = local ? local->GetLocalString(locale, key, defaultStr) : defaultStr;
            SubstGlobal(localKey, "_balance_", balanceStr);
            SubstGlobal(localKey, "_suffix_", payMethod.GetSuffix());
            return localKey;
        };
        auto name = getLocal("family_card_member_title", "");
        auto value = getLocal("family_card_member_value", "_balance_");
        auto description = getLocal("family_card_member_description", "");
        if (name) {
            json["name"] = name;
            json["display_name"] = name;
        }
        if (!unlimited) {
            json["value"] = value;
        }
        if (description) {
            json["description"] = description;
        }
        icon = settings.GetValueDef<TString>("billing.family_card_member_icon", "");
        json.InsertValue("family_card", true);
        json.InsertValue("editable", false);
    }
    auto getLocal = [&](const TString& key, const TString& defaultStr) -> TString {
        auto localKey = local ? local->GetLocalString(locale, key, defaultStr) : defaultStr;
        SubstGlobal(localKey, "_suffix_", payMethod.GetSuffix());
        return localKey;
    };
    if (payMethod.GetFamilyCardOwner()) {
        auto name = getLocal("family_card_owner_title", "");
        auto description = getLocal("family_card_owner_description", "");
        if (name) {
            json["name"] = name;
            json["display_name"] = name;
        }
        if (description) {
            json["description"] = description;
        }
        icon = settings.GetValueDef<TString>("billing.family_card_owner_icon", "");
        json.InsertValue("family_card", true);
        json.InsertValue("family_card_owner", true);
    }

    bool isYabankCard = false;
    if (payMethod.OptionalPartnerInfo()) {
        isYabankCard = payMethod.GetPartnerInfoRef().OptionalYaBankCard()
            && payMethod.GetPartnerInfoRef().GetYaBankCardRef()
            && payMethod.GetPartnerInfoRef().OptionalYaBankCardOwner()
            && payMethod.GetPartnerInfoRef().GetYaBankCardOwnerRef();
    }
    json.InsertValue("is_yabank_card", isYabankCard);

    auto systemName = getLocal("custom_" + payMethod.GetPSystem() + "_title", "");
    auto systemDescription = getLocal("custom_" + payMethod.GetPSystem() + "_description", "");
    if (systemName) {
        json["display_name"] = systemName;
    }
    if (systemDescription) {
        json["description"] = systemDescription;
    }
    auto bank = settings.GetJsonValue("billing.bank_by_bin")[::ToString(payMethod.GetBIN())].GetString();
    if (!icon && bank) {
        icon = settings.GetJsonValue("billing.bank_icons")[bank].GetString();
    }
    if (!icon) {
        icon = settings.GetJsonValue("billing.system_icons")[payMethod.GetPSystem()].GetString();
    }
    json.InsertValue("icon", icon);
    return json;
}

TVirtualTerminalsDB::TVirtualTerminalsDB(const IHistoryContext& context)
    : TBase(context, THistoryConfig().SetDeep(TDuration::Days(0)))
{
}

TMaybe<TVirtualTerminal> TVirtualTerminalsDB::GetTerminal(EBillingType type, const TInstant& reqActuality) const {
    if (!TBase::RefreshCache(reqActuality)) {
        return Nothing();
    }
    auto rg = MakeObjectReadGuard();
    auto it = Objects.find(::ToString(type));
    if (it == Objects.end()) {
        return Nothing();
    }
    return it->second;
}

bool TVirtualTerminal::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, TrustProduct);
    READ_DECODER_VALUE(decoder, values, TrustService);
    READ_DECODER_VALUE_DEF(decoder, values, TrustServiceId, 0);
    READ_DECODER_VALUE(decoder, values, FiscalTitle);
    READ_DECODER_VALUE(decoder, values, FiscalNDS);
    READ_DECODER_VALUE(decoder, values, BillingType);
    READ_DECODER_VALUE(decoder, values, CashbackType);
    READ_DECODER_VALUE(decoder, values, Terminal);
    if (decoder.GetRevision() > -1) {
        ui32 revision;
        READ_DECODER_VALUE_TEMP(decoder, values, revision, Revision);
        Revision = revision;
    }
    TerminalId = ::ToString(BillingType);
    return true;
}

NStorage::TTableRecord TVirtualTerminal::SerializeToTableRecord() const {
    NStorage::TTableRecord record;
    record.Set("trust_product", TrustProduct);
    record.Set("trust_service", TrustService);
    if (TrustServiceId) {
        record.Set("trust_service_id", TrustServiceId);
    }
    record.Set("fiscal_title", FiscalTitle);
    record.Set("fiscal_nds", FiscalNDS);
    record.Set("billing_type", ::ToString(BillingType));
    record.Set("cashback_type", CashbackType);
    record.Set("terminal", Terminal);
    if (HasRevision()) {
        record.Set("revision", *Revision);
    }
    return record;
}

NJson::TJsonValue TVirtualTerminal::GetReport() const {
    NJson::TJsonValue report;
    report["trust_product"] = TrustProduct;
    report["trust_service"] = TrustService;
    if (TrustServiceId) {
        report["trust_service_id"] = TrustServiceId;
    }
    report["fiscal_title"] = FiscalTitle;
    report["fiscal_nds"] = FiscalNDS;
    report["billing_type"] = ::ToString(BillingType);
    if (CashbackType) {
        report["cashback_type"] = CashbackType;
    }
    if (Terminal) {
        report["terminal"] = Terminal;
    }
    report["terminal_id"] = TerminalId;
     if (HasRevision()) {
        report["revision"] = Revision.GetRef();
    }
    return report;
}

TVirtualTerminal::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    TrustProduct = GetFieldDecodeIndex("trust_product", decoderBase);
    TrustService = GetFieldDecodeIndex("trust_service", decoderBase);
    TrustServiceId = GetFieldDecodeIndex("trust_service_id", decoderBase);
    FiscalTitle = GetFieldDecodeIndex("fiscal_title", decoderBase);
    FiscalNDS = GetFieldDecodeIndex("fiscal_nds", decoderBase);
    BillingType = GetFieldDecodeIndex("billing_type", decoderBase);
    CashbackType = GetFieldDecodeIndex("cashback_type", decoderBase);
    Terminal = GetFieldDecodeIndex("terminal", decoderBase);
    Revision = GetFieldDecodeIndex("revision", decoderBase);
}

template <>
NJson::TJsonValue NJson::ToJson(const EBillingType& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, EBillingType& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NTrustClient::EPaymentStatus& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NTrustClient::EPaymentStatus& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}
