#pragma once
#include <drive/library/cpp/trust/entity.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/json/merge.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/string/cast.h>


enum class EBillingQueue {
    Active = 0 /* "active" */,
    Prestable = 1 /* "prestable" */,
    Tests = 2 /* "tests" */,
};

enum class EBillingType {
    CarUsage = 0 /* "car_usage" */,
    Ticket = 1 /* "ticket" */,
    Deposit = 2 /* "deposit" */,                // DEPRECATED
    TicketGIBDD = 3 /* "ticket_gibdd" */,
    Fueling = 4 /* "fueling" */,
    Service = 5 /* "service" */,                // OLD AND UNUSED
    TicketDTP = 6 /* "ticket_DTP" */,
    TicketEva = 7 /* "ticket_eva" */,
    TicketDTPD = 8 /* "ticket_DTP_deductible" */,
    TollRoad = 9 /* "toll_road" */,
    YCashback = 10 /* "ycashback" */,
    CarUsageAdditional = 11 /* "car_usage_additional" */,  // DRIVEBACK-925
    CarFuelling = 12 /* "car_fuelling" */,
    CarParking = 13 /* "car_parking" */,
    CarCleaning = 14 /* "car_cleaning" */,
    CarUsagePrepayment = 15 /* "car_usage_prepayment" */,
    NonTransactionCashback = 16 /* "non_transaction_cashback" */,
};

enum class EPaymentGroupType {
    Radio /* "radio" */,
    Switch /* "switch" */,
    Link /* "link" */,
};

struct TChargeInfo {
    TChargeInfo(EBillingType type)
        : Type(type)
    {}

    ui32 Sum = 0;
    ui32 BaseSum = 0;
    EBillingType Type = EBillingType::CarUsage;
    TMaybe<bool> IsPlusUser;
    TMaybe<ui32> PaymentTaskId;
    TString OfferType;
    TString OfferName;
    TString SessionId;
    TString PayloadType;
    TMaybe<NJson::TJsonValue> PayloadExtra;
};

NJson::TJsonValue GetPaymentMethodUserReport(const NDrive::NTrustClient::TPaymentMethod& payMethod, ELocalization locale, const IServerBase& server);

class TVirtualTerminal {
    R_READONLY(TString, TerminalId);
    R_READONLY(TString, TrustProduct);
    R_READONLY(TString, TrustService);
    R_READONLY(ui32, TrustServiceId, 0);
    R_READONLY(TString, FiscalTitle);
    R_READONLY(TString, FiscalNDS, "nds_none");
    R_READONLY(EBillingType, BillingType, EBillingType::CarUsage);
    R_READONLY(TString, CashbackType);
    R_READONLY(TString, Terminal);
    R_OPTIONAL(ui32, Revision);

public:
    using TId = TString;

    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, TrustProduct, -1);
        R_FIELD(i32, TrustService, -1);
        R_FIELD(i32, TrustServiceId, -1);
        R_FIELD(i32, FiscalTitle, -1);
        R_FIELD(i32, FiscalNDS, -1);
        R_FIELD(i32, BillingType, -1);
        R_FIELD(i32, CashbackType, -1);
        R_FIELD(i32, Terminal, -1);
        R_FIELD(i32, Revision, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

public:
    TVirtualTerminal() = default;

    static TString GetTableName() {
        return "trust_products";
    }

    static TString GetHistoryTableName() {
        return GetTableName() + "_history";
    }

    TString GetInternalId() const {
        return GetTerminalId();
    }

    TString GetName() const {
        return GetTerminalId();
    }

    bool operator!() const {
        return !TerminalId;
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& /*errors*/, const IHistoryContext* hContext) {
        return DeserializeWithDecoder(decoder, values, hContext);
    }

    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue GetReport() const;
};

class TTerminalConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "billing_type IN (" + session->Quote(ids) + ")";
    }
    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("billing_type", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TVirtualTerminal& object) {
        return BuildCondition(object.GetTerminalId());
    }
};

class TVirtualTerminalHistoryManager : public TDBEntitiesHistoryManager<TVirtualTerminal> {
    using TBase = TDBEntitiesHistoryManager<TVirtualTerminal>;
public:
    using TBase::TBase;

    virtual bool IsNotUniqueTableSequential() const override {
        return true;
    }
};

class TVirtualTerminalsDB : public TDBEntitiesManager<TVirtualTerminal, TTerminalConditionConstructor, TVirtualTerminalHistoryManager> {
    using TBase = TDBEntitiesManager<TVirtualTerminal, TTerminalConditionConstructor, TVirtualTerminalHistoryManager>;
public:
    TVirtualTerminalsDB(const IHistoryContext& context);
    TMaybe<TVirtualTerminal> GetTerminal(EBillingType type, const TInstant& reqActuality) const;
};
