#include "history.h"

TMaybe<TVector<TBillingHistoryEvent>> TBillingHistoryManager::GetUserHistory(const TString& userId, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const {
    return GetEvents({}, tsRange, session, TQueryOptions().AddGenericCondition("user_id", userId));
}

TMaybe<TVector<TBillingHistoryEvent>> TBillingHistoryManager::GetSessionHistory(const TString& sessionId, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const {
    return GetSessionsHistory(NContainer::Scalar(sessionId), tsRange, session);
}

TMaybe<TVector<TBillingHistoryEvent>> TBillingHistoryManager::GetSessionsHistory(TConstArrayRef<TString> sessionIds, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const {
    if (sessionIds.empty()) {
        return TVector<TBillingHistoryEvent>();
    }
    return GetEvents({}, tsRange, session, TQueryOptions().SetGenericCondition("session_id", MakeSet<TString>(sessionIds)));
}

TMaybe<TVector<TBillingHistoryEvent>> TBillingHistoryManager::GetRealSessionHistory(const TString& realSessionId, const TString& userId, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const {
    if (UseDBJsonStatements) {
        auto options = TQueryOptions().AddGenericCondition("meta::json->>'real_session_id'", realSessionId);
        if (userId) {
            options.AddGenericCondition("user_id", userId);
        }
        return GetEvents({}, tsRange, session, options);
    }
    NStorage::TObjectRecordsSet<TBillingHistoryEvent> records;
    auto query = TStringBuilder() << "SELECT *";
    query << " FROM " << GetTableName() << R"( WHERE meta LIKE '%"real_session_id":")" << realSessionId << "\"%'";
    if (userId) {
        query << " AND user_id=" << session->Quote(userId);
    }
    auto transaction = session.GetTransaction();
    auto queryResult = transaction->Exec(query, &records);
    if (!ParseQueryResult(queryResult, session)) {
        return {};
    }
    return records.DetachObjects();
}

TMaybe<TMap<TString, TBillingTask>> TBillingHistoryManager::GetFinishedBillingTasks(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const {
    auto tasks = GetSessionsHistory(sessionIds, {}, session);
    if (!tasks) {
        return {};
    }
    TMap<TString, TBillingTask> result;
    for (auto&& historyTask : Reversed(*tasks)) {
        if (historyTask.IsFinished()) {
            result.emplace(historyTask.GetId(), std::move(historyTask));
        }
    }
    return result;
}

TFiscalHistoryManager::TOptionalObjectEvents TFiscalHistoryManager::GetBillsFromDB(const TString& sessionId, NDrive::TEntitySession& session) const {
    return GetBillsFromDB(NContainer::Scalar(sessionId), session);
}

TFiscalHistoryManager::TOptionalObjectEvents TFiscalHistoryManager::GetBillsFromDB(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const {
    if (sessionIds.empty()) {
        return TBase::TEvents();
    }
    TBase::TQueryOptions options;
    options.SetGenericCondition("session_id", MakeSet<TString>(sessionIds));
    return TBase::GetEvents({}, {}, session, options);
}

TMaybe<TMap<TString, TCompiledBill>> TFiscalHistoryManager::GetFullBills(const TVector<TObjectEvent<TCompiledBill>>& events, NDrive::TEntitySession& session) const {
    TMap<TString, TCompiledBill> fullBills;
    for (const auto& event : events) {
        auto it = fullBills.find(event.GetSessionId());
        if (it == fullBills.end()) {
            it = fullBills.emplace(event.GetSessionId(), TCompiledBill().SetFinal(false)).first;
        }
        if (!it->second.AddBillItem(event, session)) {
            return {};
        }
    }
    return fullBills;
}

TMaybe<TMap<TString, TCompiledBill>> TFiscalHistoryManager::GetFullBillsFromDB(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const {
    auto events = GetBillsFromDB(sessionIds, session);
    if (!events) {
        return {};
    }
    return GetFullBills(*events, session);
}

TCompiledBill TFiscalHistoryManager::GetFullBill(const TString& sessionId, const TMap<TString, TCompiledBill>& fullBills) {
    auto it = fullBills.find(sessionId);
    if (it == fullBills.end()) {
        return TCompiledBill().SetFinal(false);
    }
    return it->second;
}

TMaybe<TCompiledBill> TFiscalHistoryManager::GetFullBillFromDB(const TString& sessionId, NDrive::TEntitySession& session) const {
    auto events = GetFullBillsFromDB(NContainer::Scalar(sessionId), session);
    if (!events) {
        return {};
    }
    return GetFullBill(sessionId, *events);
}

TFiscalHistoryManager::TOptionalObjectEvents TFiscalHistoryManager::GetUserBillsFromDB(const TString& userId, TMaybe<TInstant> since, TMaybe<TInstant> until, NDrive::TEntitySession& session, TMaybe<EBillingType> billingType, TMaybe<ui32> last, TMaybe<ui32> offset) const {
    TBase::TQueryOptions options;
    options.SetOrderBy({
            "history_event_id",
            "history_timestamp"
        });
    if (last) {
        options.SetLimit(*last);
        options.SetDescending(true);
    }
    if (offset) {
        options.SetOffset(*offset);
    }
    options.AddGenericCondition("history_user_id", userId);
    if (billingType) {
        options.AddGenericCondition("billing_type", ::ToString(*billingType));
    }
    return TBase::GetEvents({}, { since, until }, session, options);
}

TMaybe<TMap<TString, TCompiledBill>> TFiscalHistoryManager::GetUserFullBillsFromDB(const TString& userId, TMaybe<TInstant> since, TMaybe<TInstant> until, NDrive::TEntitySession& session, TMaybe<EBillingType> billingType, TMaybe<ui32> last) const {
    TVector<TString> finalEvents;
    if (last) {
        ui32 offset = 0;
        while (finalEvents.size() < *last) {
            auto events = GetUserBillsFromDB(userId, since, until, session, billingType, *last * 7, offset);
            if (!events) {
                return {};
            }
            if (events->empty()) {
                break;
            }
            offset += events->size();
            for (auto&& event : *events) {
                if (event.GetFinal()) {
                    finalEvents.push_back(event.GetSessionId());
                    if (finalEvents.size() == *last) {
                        break;
                    }
                }
            }
        }
    } else {
        auto events = GetUserBillsFromDB(userId, since, until, session, billingType);
        if (!events) {
            return {};
        }
        for (auto&& event : *events) {
            if (event.GetFinal()) {
                finalEvents.push_back(event.GetSessionId());
            }
        }
    }
    return GetFullBillsFromDB(finalEvents, session);
}

TFiscalRefundsHistoryManager::TOptionalObjectEvents TFiscalRefundsHistoryManager::GetSessionRefundsFromDB(const TString& sessionId, NDrive::TEntitySession& session) const {
    return GetRefundsFromDB(NContainer::Scalar(sessionId), session);
}

TFiscalRefundsHistoryManager::TOptionalObjectEvents TFiscalRefundsHistoryManager::GetRefundsFromDB(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const {
    if (sessionIds.empty()) {
        return TBase::TEvents();
    }
    TBase::TQueryOptions options;
    options.SetGenericCondition("session_id", MakeSet<TString>(sessionIds));
    return TBase::GetEvents({}, {}, session, options);
}
