#pragma once

#include "promocodes.h"
#include "tasks.h"

#include <drive/backend/database/history/manager.h>

class TBillingHistoryManager: public TDatabaseHistoryManager<TBillingTask> {
private:
    using TBase =  TDatabaseHistoryManager<TBillingTask>;

    bool UseDBJsonStatements;

public:
    TBillingHistoryManager(const IHistoryContext& context, bool useDBJsonStatements)
        : TBase(context, "billing_tasks_history")
        , UseDBJsonStatements(useDBJsonStatements)
    {
    }

    TMaybe<TVector<TBillingHistoryEvent>> GetUserHistory(const TString& userId, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const;
    TMaybe<TVector<TBillingHistoryEvent>> GetSessionHistory(const TString& sessionId, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const;
    TMaybe<TVector<TBillingHistoryEvent>> GetSessionsHistory(TConstArrayRef<TString> sessionIds, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const;
    TMaybe<TVector<TBillingHistoryEvent>> GetRealSessionHistory(const TString& realSessionId, const TString& userId, const TRange<TInstant>& tsRange, NDrive::TEntitySession& session) const;
    TMaybe<TMap<TString, TBillingTask>> GetFinishedBillingTasks(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const;
};

class TFiscalHistoryManager : public TDatabaseHistoryManager<TCompiledBill> {
private:
    using TBase = TDatabaseHistoryManager<TCompiledBill>;
    using TOptionalObjectEvents = TOptionalObjectEvents<TCompiledBill>;

public:
    TFiscalHistoryManager(const IHistoryContext& context)
        : TBase(context, TCompiledBill::GetTableName())
    {
    }

    TOptionalObjectEvents GetBillsFromDB(const TString& sessionId, NDrive::TEntitySession& session) const;
    TOptionalObjectEvents GetBillsFromDB(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const;
    TOptionalObjectEvents GetUserBillsFromDB(const TString& userId, TMaybe<TInstant> since, TMaybe<TInstant> until, NDrive::TEntitySession& session, TMaybe<EBillingType> billingType = {}, TMaybe<ui32> last = {}, TMaybe<ui32> offset = {}) const;

    TMaybe<TMap<TString, TCompiledBill>> GetFullBills(const TVector<TObjectEvent<TCompiledBill>>& events, NDrive::TEntitySession& session) const;
    TMaybe<TMap<TString, TCompiledBill>> GetFullBillsFromDB(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const;
    TMaybe<TCompiledBill> GetFullBillFromDB(const TString& sessionId, NDrive::TEntitySession& session) const;
    TMaybe<TMap<TString, TCompiledBill>> GetUserFullBillsFromDB(const TString& userId, TMaybe<TInstant> since, TMaybe<TInstant> until, NDrive::TEntitySession& session, TMaybe<EBillingType> billingType = {}, TMaybe<ui32> last = {}) const;

    static TCompiledBill GetFullBill(const TString& sessionId, const TMap<TString, TCompiledBill>& fullBills);
};

class TFiscalRefundsHistoryManager : public TDatabaseHistoryManager<TCompiledRefund> {
private:
    using TBase = TDatabaseHistoryManager<TCompiledRefund>;
    using TOptionalObjectEvents = TOptionalObjectEvents<TCompiledRefund>;

public:

    TFiscalRefundsHistoryManager(const IHistoryContext& context)
        : TBase(context, TCompiledRefund::GetTableName())
    {
    }

    TOptionalObjectEvents GetSessionRefundsFromDB(const TString& sessionId, NDrive::TEntitySession& session) const;
    TOptionalObjectEvents GetRefundsFromDB(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const;
};

class TPromocodeAccountLinksHistoryManager : public TDatabaseHistoryManager<TPromocodeAccountLink> {
private:
    using TBase = TDatabaseHistoryManager<TPromocodeAccountLink>;

public:
    TPromocodeAccountLinksHistoryManager(const IHistoryContext& context)
        : TBase(context, TPromocodeAccountLink::GetTableName())
    {}

    TOptionalEvents GetAccountsByPromocodesFromDB(TConstArrayRef<TString> promocodes, NDrive::TEntitySession& session) const {
        return TBase::GetEvents({}, {}, session, TQueryOptions().SetGenericCondition("promocode", MakeSet<TString>(promocodes)));
    }

    TOptionalEvents GetPromocodeByAccountFromDB(const TVector<ui32>& accountIds, NDrive::TEntitySession& session) const {
        TSet<TString> accountStr;
        for(const auto& id : accountIds) {
            accountStr.emplace(::ToString(id));
        }
        return TBase::GetEvents({}, {}, session, TQueryOptions().SetGenericCondition("account_id", std::move(accountStr)));
    }
};
