#include "logic.h"

#include <drive/backend/billing/accounts_manager.h>

#include <rtline/library/unistat/cache.h>

EProduceResult IChargeLogic::SyncPayments(const TPaymentsData& /*snapshot*/, const TPaymentsManager& /*manager*/, const NDrive::NBilling::TAccountsManager& /*accountsManager*/, NDrive::ITrustStorage* /*cache*/, const bool /*sync*/) const {
    return EProduceResult::Ok;
}

TChargeInfo IChargeLogic::CorrectCharge(const NDrive::NBilling::IBillingAccount::TPtr /*account*/, const TPaymentsData& /*snapshot*/, const TChargeInfo& chargeInfo) const {
    return chargeInfo;
}

TChargeInfo IChargeLogic::CorrectChargeForBonus(const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const {
    auto cardSum = Settings.GetValue<ui64>("billing.card_pay_min_sum").GetOrElse(0);
    TChargeInfo newCharge = chargeInfo;
    if (snapshot.GetSnapshot().GetCardSum() < cardSum && (snapshot.GetBillingTask().GetDeposit() > snapshot.GetSnapshot().GetHeldSum() || snapshot.GetBillingTask().IsFinished() || snapshot.GetOnFinalization())) {
        if (newCharge.Sum <= cardSum) {
            newCharge.Sum = 0;
        } else {
            newCharge.Sum -= cardSum;
        }
    }
    return newCharge;
}

TExpectedCharge IChargeLogic::ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager,
    const TChargeInfo& chargeInfo, TLazySession& session, const bool /*sync*/) const {
    if (account->GetType() != GetType() || !account->IsActive()) {
        return MakeUnexpected(EProduceResult::Error);
    }

    TChargeInfo charge = CorrectCharge(account, snapshot, chargeInfo);
    if (charge.Sum == 0) {
        return charge;
    }

    if (account->Remove(charge.Sum, session.Get()) != EDriveOpResult::Ok) {
        return MakeUnexpected(EProduceResult::TransactionError);
    }

    TPaymentTask payment;
    if (!BuildPaymentTask(account, paymentsManager, snapshot.GetBillingTask(), charge, payment, session.Get()) || !session.Commit()) {
        return MakeUnexpected(EProduceResult::TransactionError);
    }
    charge.PaymentTaskId = payment.GetId();
    return charge;
}

bool IChargeLogic::Refund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsManager& manager, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord update;
    if (!DoRefund(account, payment, sum, session, update)) {
        session.AddErrorMessage("fiscal_refund", "Can't refund (" + ToString(sum) + ") " + payment.GetPaymentId());
        return false;
    }
    update.Set("cleared", ToString(payment.GetCleared() - sum));
    update.Set("last_update_ts", ModelingNow().Seconds());
    update.Set("id", "nextval('drive_payments_id_seq')");

    TPaymentTask newPayment;
    return manager.UpdatePayment(update, payment, newPayment, session) == EDriveOpResult::Ok;
}

bool IChargeLogic::BuildPaymentTask(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsManager& manager, const TBillingTask& task, const TChargeInfo& charge, TPaymentTask& payment, NDrive::TEntitySession& session) const {
    ui64 ts = ModelingNow().Seconds();
    NStorage::TTableRecord record;
    record.Set("account_id", account ? account->GetId() : 0);
    record.Set("sum", charge.Sum);
    record.Set("cleared", charge.Sum);
    record.Set("billing_type", ToString(charge.Type));
    record.Set("payment_type", ToString(GetType()));
    record.Set("last_update_ts", ts);
    record.Set("created_at_ts", ts);
    record.Set("session_id", task.GetId());
    record.Set("status", ::ToString(NDrive::NTrustClient::EPaymentStatus::Cleared));
    record.Set("pay_method", ToString(GetType()));
    record.Set("payment_id", GetPaymentPrefix() + CreateGuidAsString());
    record.Set("user_id", task.GetUserId());
    if (DoBuildPaymentTask(account, manager, task, charge, record) && manager.RegisterPayment(record, payment, session) == EDriveOpResult::Ok) {
        TUnistatSignalsCache::SignalAdd("drive-frontend-billing-authorized" + (account ? ::ToString(account->GetType()) : "unknown"), "ok", charge.Sum);
        return true;
    }
    return false;
}
