#pragma once

#include "account.h"
#include "payments.h"

#include <rtline/library/metasearch/simple/config.h>

#include <util/generic/cast.h>


namespace NDrive::NBilling {
    class TAccountsManager;
}

class TUsersDB;
namespace NDrive {
    class ITrustStorage;
}

class IChargeLogic {
public:
    using TPtr = TAtomicSharedPtr<IChargeLogic>;

    IChargeLogic(NDrive::NBilling::EAccount type, const ISettings& settings)
        : LogicType(type)
        , Settings(settings)
    {}

    virtual ~IChargeLogic() {};

    virtual TExpectedCharge ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager, const TChargeInfo& chargeInfo, TLazySession& session, const bool /*sync*/) const;
    virtual EProduceResult SyncPayments(const TPaymentsData& snapshot, const TPaymentsManager& manager, const NDrive::NBilling::TAccountsManager& accountsManager, NDrive::ITrustStorage* cache, const bool sync) const;

    bool Refund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsManager& manager, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session) const;

    NDrive::NBilling::EAccount GetType() const {
        return LogicType;
    }

    virtual bool IsSync() const {
        return true;
    }

    virtual void WaitOperations() {
    }

    template <class T>
    const T* GetAsSafe() const {
        return VerifyDynamicCast<const T*>(this);
    }
private:
    virtual bool DoRefund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session, NStorage::TTableRecord& patchPaymentRecord) const = 0;
    virtual bool DoBuildPaymentTask(NDrive::NBilling::IBillingAccount::TPtr /*account*/, const TPaymentsManager& /*manager*/, const TBillingTask& /*task*/, const TChargeInfo& /*charge*/, NStorage::TTableRecord& /*paymentRecord*/) const {
        return true;
    }

protected:
    bool BuildPaymentTask(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsManager& manager, const TBillingTask& task, const TChargeInfo& charge, TPaymentTask& payment, NDrive::TEntitySession& session) const;
    TChargeInfo CorrectChargeForBonus(const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const;
    virtual TChargeInfo CorrectCharge(const NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const;
    virtual TString GetPaymentPrefix() const = 0;
private:
    NDrive::NBilling::EAccount LogicType;
protected:
    const ISettings& Settings;
};

class ILogicConfig {
public:
    using TPtr = TAtomicSharedPtr<ILogicConfig>;
    virtual ~ILogicConfig() {};

    virtual void Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) = 0;
    virtual void ToString(IOutputStream& os) const = 0;
    virtual NDrive::NBilling::EAccount GetType() const = 0;
    virtual IChargeLogic::TPtr ConstructLogic(const ISettings& settings, TUsersDB& userDB, TAtomicSharedPtr<NDrive::INotifier> notifier) const = 0;
    virtual ILogicData::TPtr ConstructData(const NJson::TJsonValue& /*data*/) const {
        return nullptr;
    }


    using TFactory = NObjectFactory::TObjectFactory<ILogicConfig, NDrive::NBilling::EAccount>;
};

