#include "payments.h"
#include "entities.h"

#include <drive/backend/billing/client/charge_logic.h>

#include <util/generic/adaptor.h>
#include <util/string/vector.h>


TSet<NDrive::NTrustClient::EPaymentStatus> TBillingGlobals::AuthStatuses = { NDrive::NTrustClient::EPaymentStatus::Authorized, NDrive::NTrustClient::EPaymentStatus::NotAuthorized, NDrive::NTrustClient::EPaymentStatus::Cleared, NDrive::NTrustClient::EPaymentStatus::Canceled, NDrive::NTrustClient::EPaymentStatus::Refunded };
TSet<NDrive::NTrustClient::EPaymentStatus> TBillingGlobals::FinalStatuses = { NDrive::NTrustClient::EPaymentStatus::Cleared, NDrive::NTrustClient::EPaymentStatus::Canceled, NDrive::NTrustClient::EPaymentStatus::Refunded };
TSet<NDrive::NTrustClient::EPaymentStatus> TBillingGlobals::HeldStatuses = { NDrive::NTrustClient::EPaymentStatus::Authorized, NDrive::NTrustClient::EPaymentStatus::Cleared, NDrive::NTrustClient::EPaymentStatus::Refunded };
TSet<NDrive::NTrustClient::EPaymentStatus> TBillingGlobals::FailedStatuses = { NDrive::NTrustClient::EPaymentStatus::NotAuthorized };
TSet<NDrive::NBilling::EAccount> TBillingGlobals::BonusTypes = { NDrive::NBilling::EAccount::Bonus, NDrive::NBilling::EAccount::Coins, NDrive::NBilling::EAccount::YAccount };
TSet<EBillingType> TBillingGlobals::CashbackBillingTypes = { EBillingType::YCashback, EBillingType::NonTransactionCashback };

TSet<TString> TBillingGlobals::SupportedPayMethods = { "card" };
TSet<TString> TBillingGlobals::SupportedCashbackMethods = { "yandex_account" };
TSet<TString> TBillingGlobals::SupportedCashbackCurrency = { "RUB" };


bool TCachedPayments::TErrorInfo::operator<(const TErrorInfo& other) const {
    return PaymentMethod < other.PaymentMethod;
}

bool TCachedPayments::Init(const TVector<TPaymentTask>& payments) {
    if (payments.empty()) {
        return true;
    }

    for (auto&& pay : payments) {
        if (!Update(pay)) {
            return false;
        }
    }
    return true;
}

bool TCachedPayments::Update(const TPaymentTask& payment, const bool skipFailed) {
    if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Unknown) {
        ERROR_LOG << "Billing error. Unknown payment status found for " << payment.GetSessionId() << Endl;
        return false;
    }
    /* if (Version >= payment.GetId()) {
        return true;
    }
    */
    if (!SessionId) {
        SessionId = payment.GetSessionId();
    }
    LastStatus = payment.GetStatus();
    Version = Max<ui64>(payment.GetId(), Version);
    FirstPaymentTs = Min<TInstant>(payment.GetCreatedAt(), FirstPaymentTs);
    auto it = Payments.find(payment);
    if (it != Payments.end()) {
        const TPaymentTask& paymentOld = *it;
        if (TBillingGlobals::HeldStatuses.contains(paymentOld.GetStatus())) {
            ui32 changeSum = 0;
            if (paymentOld.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared || paymentOld.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Refunded) {
                changeSum = paymentOld.GetCleared();
            } else {
                changeSum = paymentOld.GetSum();
            }
            if (paymentOld.GetPaymentType() == NDrive::NBilling::EAccount::YCashback) {
                auto cashbackIt = CashbackSumByTypes.find(paymentOld.GetBillingType());
                if (cashbackIt != CashbackSumByTypes.end()) {
                    if (cashbackIt->second > changeSum) {
                        cashbackIt->second -= changeSum;
                    } else {
                        CashbackSumByTypes.erase(cashbackIt);
                    }
                }
                if (CashbackSum >= changeSum) {
                    CashbackSum -= changeSum;
                } else {
                    CashbackSum = 0;
                }
            } else {
                HeldSum -= changeSum;
                switch (payment.GetPaymentType()) {
                case NDrive::NBilling::EAccount::Trust:
                case NDrive::NBilling::EAccount::MobilePayment:
                    CardSum -= changeSum;
                    break;
                case NDrive::NBilling::EAccount::YAccount:
                    YandexAccountSum -= changeSum;
                    break;
                case NDrive::NBilling::EAccount::Bonus:
                case NDrive::NBilling::EAccount::Coins:
                    BonusSum -= changeSum;
                    break;
                case NDrive::NBilling::EAccount::Fake:
                    CanceledSum -= changeSum;
                    break;
                default:
                    break;
                }
            }
        }
        Payments.erase(it);
    }

    if (skipFailed) {
        TErrorInfo error = {payment.GetPayMethod(), payment.GetCreatedAt(), payment.GetPaymentError()};
        auto it = FailedPayMethods.find(error);
        if (it != FailedPayMethods.end() && it->Date < error.Date) {
            error.Count += it->Count;
            FailedPayMethods.erase(it);
        }
        if (TBillingGlobals::FailedStatuses.contains(payment.GetStatus())) {
            FailedPayMethods.emplace(std::move(error));
            return true;
        }
    }

    Payments.emplace(payment);

    if (TBillingGlobals::HeldStatuses.contains(payment.GetStatus())) {
        ui32 changeSum = 0;
        if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared || payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Refunded) {
            changeSum = payment.GetCleared();
        } else {
            changeSum = payment.GetSum();
        }
        if (payment.GetPaymentType() == NDrive::NBilling::EAccount::YCashback) {
            auto cashbackIt = CashbackSumByTypes.find(payment.GetBillingType());
            if (cashbackIt == CashbackSumByTypes.end()) {
                cashbackIt = CashbackSumByTypes.emplace(payment.GetBillingType(), 0).first;
            }
            cashbackIt->second += changeSum;
            CashbackSum += changeSum;
        } else {
            HeldSum += changeSum;
            switch (payment.GetPaymentType()) {
            case NDrive::NBilling::EAccount::Trust:
            case NDrive::NBilling::EAccount::MobilePayment:
                CardSum += changeSum;
                break;
            case NDrive::NBilling::EAccount::YAccount:
                YandexAccountSum += changeSum;
                break;
            case NDrive::NBilling::EAccount::Bonus:
                BonusSum += changeSum;
                break;
            case NDrive::NBilling::EAccount::Fake:
                CanceledSum += changeSum;
                break;
            default:
                break;
            }
        }
    }
    return true;
}

TString TCachedPayments::TErrorInfo::GetPayMethod(const TErrorInfo& info) {
    return info.PaymentMethod;
}

TSet<TString> TCachedPayments::GetFailedPayMethods() const {
    TSet<TString> result;
    Transform(FailedPayMethods.begin(), FailedPayMethods.end(), std::inserter(result, result.begin()), TErrorInfo::GetPayMethod);
    return result;
}

const TPaymentTask* TCachedPayments::GetProcessingPayment() const {
    for (auto&& payment : Payments) {
        if (!TBillingGlobals::AuthStatuses.contains(payment.GetStatus())) {
            return &payment;
        }
    }
    return nullptr;
}

TVector<TPaymentTask> TCachedPayments::GetTimeline() const {
    TVector<TPaymentTask> sorted(Payments.begin(), Payments.end());
    const auto compare = [](const TPaymentTask& left, const TPaymentTask& right) -> bool {
        return left.GetCreatedAt() < right.GetCreatedAt() ||  (left.GetCreatedAt() == right.GetCreatedAt() && left.GetId() < right.GetId());
    };

    Sort(sorted.begin(), sorted.end(), compare);
    return sorted;
}

bool TCorrectedSessionPayments::Init(const TCachedPayments& snapshot, ui32 minCardSum) {
    ui32 walletSum = 0;
    if (snapshot.GetHeldSum() > snapshot.GetCardSum() + snapshot.GetCanceledSum()) {
        walletSum = snapshot.GetHeldSum() - (snapshot.GetCardSum() + snapshot.GetCanceledSum());
    }
    for (auto&& payment : snapshot.GetPayments()) {
        ui32 changeSum = 0;
        if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Authorized) {
            changeSum = payment.GetSum();
        } else if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared || payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Refunded) {
            changeSum = payment.GetCleared();
        }
        if (payment.GetPaymentType() != NDrive::NBilling::EAccount::YCashback && changeSum > 0) {
            SumByAccount[payment.GetAccountId()] += changeSum;
        }
    }
    if (snapshot.GetHeldSum() > Bill + snapshot.GetCanceledSum()) {
        auto refunds = snapshot.CalculateRefunds(snapshot.GetHeldSum() - Bill - snapshot.GetCanceledSum(), {}, minCardSum);
        if (!refunds) {
            return false;
        }
        for (auto&& refund : *refunds) {
            if (refund.GetPayment().GetPaymentType() != NDrive::NBilling::EAccount::YCashback) {
                const auto& accountId = refund.GetPayment().GetAccountId();
                auto accountIt = SumByAccount.find(accountId);
                if (accountIt == SumByAccount.end() || refund.GetSum() > accountIt->second) {
                    return false;
                }
                accountIt->second -= refund.GetSum();
                if (accountIt->second == 0) {
                    SumByAccount.erase(accountIt);
                }
            }
        }
    }
    CashbackBaseSum = Max<i32>(0, (i32)Bill - walletSum);
    return true;
}

bool TCachedPayments::WaitAnyPayments() const {
    if (Payments.empty()) {
        return false;
    }
    return !!GetProcessingPayment();
}

TExpected<TVector<TRefundIssueNew>, TString> TCachedPayments::CalculateRefunds(ui32 refundSum, const TVector<TRefundTask>& refunds, ui32 minCardSum) const {
    TMap<TString, ui32> refundsByPayment;
    ui32 refundedSum = 0;
    for (auto&& refundTask : refunds) {
        if (refundTask.GetStatus() == "draft") {
            return MakeUnexpected<TString>("draft refund " + ToString(refundTask.GetPaymentId()));
        }
        if (refundTask.GetFinished()) {
            refundsByPayment[refundTask.GetPaymentId()] += refundTask.GetSum();
            refundedSum += refundTask.GetSum();
        }
    }

    TVector<TRefundIssueNew> result;
    TVector<TPaymentTask> sorted = GetTimeline();

    ui32 overhead = refundSum + minCardSum;
    ui32 nonBonusRefund = 0;

    ui32 cashbackSum = 0;
    for (auto&& pay : Reversed(sorted)) {
        if (pay.GetPaymentType() == NDrive::NBilling::EAccount::Fake || pay.GetPaymentType() == NDrive::NBilling::EAccount::YCashback) {
            continue;
        }
        if (!TBillingGlobals::HeldStatuses.contains(pay.GetStatus())) {
            continue;
        }
        if (overhead == 0) {
            break;
        }
        auto refundsIt = refundsByPayment.find(pay.GetPaymentId());
        ui32 refunded = (refundsIt == refundsByPayment.end()) ? 0 : refundsIt->second;
        ui32 paymentsSum = (pay.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared) ? pay.GetCleared() : pay.GetSum();

        if (paymentsSum < refunded) {
            return MakeUnexpected<TString>("refunded more than payment " + ToString(paymentsSum) + "/" + ToString(refunded));
        }

        ui32 sum = Min<ui32>(paymentsSum - refunded, overhead);
        if (sum > 0) {
            if (!TBillingGlobals::BonusTypes.contains(pay.GetPaymentType())) {
                nonBonusRefund += sum;
            }
            result.push_back(TRefundIssueNew(pay, sum));
            overhead -= sum;
            auto meta = dynamic_cast<const TTrustPaymentMeta*>(pay.GetMeta().Get());
            if (meta) {
                cashbackSum += sum * meta->GetCashbackPercent();
            }
        }
    }

    if (overhead > minCardSum) {
        return MakeUnexpected<TString>("more refund expected then payment of " + ToString(overhead - minCardSum));
    }

    ui32 cardNoRefund = Min<ui32>(minCardSum - overhead, nonBonusRefund);
    ui32 bonusNoRefund = minCardSum - overhead - cardNoRefund;

    Reverse(result.begin(), result.end());
    for (auto it = result.begin(); it != result.end();) {
        if (!TBillingGlobals::BonusTypes.contains(it->GetPayment().GetPaymentType())) {
            if (cardNoRefund > 0) {
                if (cardNoRefund >= it->GetSum()) {
                    cardNoRefund -= it->GetSum();
                    it = result.erase(it);
                    continue;
                } else {
                    it->MutableSum() -= cardNoRefund;
                    cardNoRefund = 0;
                }
            }
        } else {
            if (bonusNoRefund > 0) {
                if (bonusNoRefund >= it->GetSum()) {
                    bonusNoRefund -= it->GetSum();
                    it = result.erase(it);
                    continue;
                } else {
                    it->MutableSum() -= bonusNoRefund;
                    bonusNoRefund = 0;
                }
            }
        }
        ++it;
    }

    if (cardNoRefund > 0 || bonusNoRefund > 0) {
        return MakeUnexpected<TString>("cannot calculate non refunded sum");
    }
    Reverse(result.begin(), result.end());

    const ui32 minTrustPayment = 100;
    for (const auto& refundIssue : result) {
        const auto payment = refundIssue.GetPayment();
        if (payment.GetPaymentType() != NDrive::NBilling::EAccount::Trust) {
            continue;
        }
        const auto refundsIt = refundsByPayment.find(payment.GetPaymentId());
        const ui32 refunded = (refundsIt == refundsByPayment.end()) ? 0 : refundsIt->second;
        const ui32 paymentsSum = (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared) ? payment.GetCleared() : payment.GetSum();
        const auto paymentAfterRefund = paymentsSum - refunded - refundIssue.GetSum();
        if (paymentsSum > refunded + refundIssue.GetSum() && paymentAfterRefund < minTrustPayment) {
            return MakeUnexpected<TString>("final payment cannot be less than " + ToString(minTrustPayment));
        }
    }

    cashbackSum /= 100;
    if (cashbackSum > 0) {
        for (auto&& pay : Reversed(sorted)) {
            if (pay.GetPaymentType() != NDrive::NBilling::EAccount::YCashback) {
                continue;
            }
            if (!TBillingGlobals::HeldStatuses.contains(pay.GetStatus())) {
                continue;
            }
            if (cashbackSum == 0) {
                break;
            }
            auto refundsIt = refundsByPayment.find(pay.GetPaymentId());
            ui32 refunded = (refundsIt == refundsByPayment.end()) ? 0 : refundsIt->second;
            ui32 paymentsSum = (pay.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Authorized) ? pay.GetSum() : pay.GetCleared();

            if (paymentsSum < refunded) {
                return MakeUnexpected<TString>("refunded cashback more then payment " + ToString(paymentsSum) + "/" + ToString(refunded));
            }

            ui32 sum = Min<ui32>(paymentsSum - refunded, cashbackSum);
            if (sum > 0) {
                result.push_back(TRefundIssueNew(pay, sum));
                cashbackSum -= sum;
            }
        }
    }

    return result;
}

ui32 TPaymentsData::GetActualCashback() const {
    if (BillingTask.GetCashback() < 0.1) {
        return 0;
    }
    ui32 cashbackByPayments = 0;
    bool hasFake = false;
    for (const auto& payment : Snapshot.GetPayments()) {
        if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Fake) {
            hasFake = true;
        } else if (payment.GetPaymentType() != NDrive::NBilling::EAccount::YCashback && TBillingGlobals::HeldStatuses.contains(payment.GetStatus())) {
            ui32 changeSum = 0;
            if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared || payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Refunded) {
                changeSum = payment.GetCleared();
            } else {
                changeSum = payment.GetSum();
            }
            auto meta = dynamic_cast<const TTrustPaymentMeta*>(payment.GetMeta().Get());
            if (meta) {
                cashbackByPayments += changeSum * meta->GetCashbackPercent();
            }
        }
    }
    cashbackByPayments = std::round(cashbackByPayments / 10000.) * 100;
    if (hasFake) {
        return cashbackByPayments;
    }
    if (cashbackByPayments + 200 < BillingTask.GetCashback()) {
        ALERT_LOG << "Incorrect cashback " << BillingTask.GetId() << ":" << cashbackByPayments << "/" << BillingTask.GetCashback() << Endl;
    }
    return BillingTask.GetCashback();
}

EDriveOpResult TPaymentsManager::RegisterPayment(const NStorage::TTableRecord& record, TPaymentTask& newPayment, NDrive::TEntitySession& session, const TMaybe<ui64> oldPaymentId) const {
    auto table = Database->GetTable("drive_payments");
    NStorage::TObjectRecordsSet<TPaymentTask> affected;
    if (!table->AddRow(record, session.GetTransaction(), "", &affected)->IsSucceed()) {
        return EDriveOpResult::TransactionError;
    }

    if (affected.size() != 1) {
        return EDriveOpResult::TransactionError;
    }

    newPayment = affected.front();
    auto ctx = session.GetContextAs<TBillingSessionContext>();
    if (ctx) {
        ctx->SetVersion(newPayment.GetSessionId(), newPayment.GetId());
    }

    TMaybe<EPaymentStatus> taskStatus;
    if (newPayment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared && newPayment.GetPaymentType() != NDrive::NBilling::EAccount::Fake) {
        taskStatus = EPaymentStatus::Ok;
    }

    return UpdatePaymentStatus(newPayment, taskStatus, session, oldPaymentId);
}

EDriveOpResult TPaymentsManager::UpdatePayment(const NStorage::TTableRecord& record, const TPaymentTask& payment, TPaymentTask& newPayment, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable("drive_payments");

    NStorage::TTableRecord condition;
    condition.Set("payment_id", payment.GetPaymentId());
    condition.Set("id", payment.GetId());

    NStorage::TObjectRecordsSet<TPaymentTask> affected;
    if (!table->UpdateRow(condition, record, session.GetTransaction(), &affected)->IsSucceed()) {
        return EDriveOpResult::TransactionError;
    }

    if (affected.size() != 1) {
        session.AddErrorMessage("update_payment", "Conflict for " + payment.GetPaymentId());
        return EDriveOpResult::TransactionError;
    }
    newPayment = affected.front();
    if (UpdatePaymentStatus(newPayment, TMaybe<EPaymentStatus>(), session) != EDriveOpResult::TransactionError) {
        return EDriveOpResult::Ok;
    }
    return EDriveOpResult::TransactionError;
}

bool TPaymentsManager::MarkStarted(const TString& orderId, const TString& paymentId, bool updateBillingTask) const {
    auto session = BuildSession(false);
    auto table = session->GetDatabase().GetTable("drive_payments");

    NStorage::TTableRecord update;
    update.Set("status", NDrive::NTrustClient::EPaymentStatus::Started);
    update.Set("id", "nextval('drive_payments_id_seq')");
    update.Set("last_update_ts", ModelingNow().Seconds());
    if (orderId) {
        update.Set("order_id", orderId);
    }

    NStorage::TTableRecord condition;
    condition.Set("payment_id", paymentId);
    NStorage::TObjectRecordsSet<TPaymentTask> affected;
    auto result = table->UpdateRow(condition, update, session.GetTransaction(), &affected);
    if (result && result->IsSucceed() && affected.size() == 1) {
        TPaymentTask& payment = affected.front();
        auto ctx = session.GetContextAs<TBillingSessionContext>();
        if (ctx) {
            ctx->SetVersion(payment.GetSessionId(), payment.GetId());
        }

        if (updateBillingTask && UpdatePaymentStatus(payment, TMaybe<EPaymentStatus>(), session) != EDriveOpResult::Ok) {
            return false;
        }

        return session.Commit();
    }
    return false;
}

EDriveOpResult TPaymentsManager::UpdateTaskStatus(const TString& sessionId, TMaybe<EPaymentStatus> paymentStatus, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable("billing_tasks");
    TInstant ts = ModelingNow();
    TBillingTask::TExecContext context;
    context.SetLastUpdate(ts);

    NStorage::TTableRecord update;
    update.Set("last_status_update", ts.Seconds());
    update.Set("exec_context", context.Serialize().GetStringRobust());
    update.Set("event_id", "nextval('billing_tasks_event_id_seq')");
    if (paymentStatus.Defined()) {
        update.Set("task_status", paymentStatus.GetRef());
    }
    NStorage::TTableRecord condition;
    condition.Set("session_id", sessionId);
    auto results = table->UpdateRow(condition, update, session.GetTransaction());
    if (!results->IsSucceed()) {
        return EDriveOpResult::TransactionError;
    }

    if (results->GetAffectedRows() != 1) {
        return EDriveOpResult::LogicError;
    }

    return EDriveOpResult::Ok;
}

EDriveOpResult TPaymentsManager::UpdatePaymentStatus(const TPaymentTask& paymentTask, TMaybe<EPaymentStatus> paymentStatus, NDrive::TEntitySession& session, const TMaybe<ui64> oldPaymentId) const {
    auto table = Database->GetTable("billing_tasks");

    NStorage::TTableRecord update;
    if (paymentStatus.Defined()) {
        update.Set("task_status", paymentStatus.GetRef());
    }
    TInstant ts = ModelingNow();
    update.Set("last_status_update", ts.Seconds());
    update.Set("last_payment_update", paymentTask.GetLastUpdate().Seconds());
    update.Set("last_payment_id", paymentTask.GetId());

    TBillingTask::TExecContext context;
    context.SetLastUpdate(ts);
    update.Set("exec_context", context.Serialize().GetStringRobust());

    NStorage::TTableRecord condition;
    condition.Set("session_id", paymentTask.GetSessionId());
    if (oldPaymentId) {
        condition.Set("last_payment_id", oldPaymentId);
    }

    auto results = table->UpdateRow(condition, update, session.GetTransaction());
    if (!results->IsSucceed()) {
        return EDriveOpResult::TransactionError;
    }

    if (results->GetAffectedRows() != 1) {
        return EDriveOpResult::LogicError;
    }
    auto ctx = session.GetContextAs<TBillingSessionContext>();
    if (ctx) {
        ctx->SetVersion(paymentTask.GetSessionId(), paymentTask.GetId());
    }
    return EDriveOpResult::Ok;
}

ui64 TPaymentsManager::GetPaymentVersion(const TString& sessionId) const {
    TReadGuard g(VersionsLock);
    auto  it = VersionBySession.find(sessionId);
    if (it != VersionBySession.end()) {
        return it->second;
    }
    return Max<ui64>();
}


void TPaymentsManager::UpdatePaymentVersion(const TString& sessionId, ui64 version) const {
    TWriteGuard g(VersionsLock);
    VersionBySession[sessionId] = version;
}


TPaymentsManager::TPaymentsManager(const IHistoryContext& context, const NDrive::NBilling::TActiveTasksManager& activeTasksManager, bool useCache)
    : TBase("payments_manager")
    , Database(context.GetDatabase())
    , UpdatesWatcher(context)
    , Refunds(context)
    , ActiveTasksManager(activeTasksManager)
    , ReloadErrorsCount({ "drive-frontend-billing-reload-errors-count" }, false)
    , UseCache(useCache)
{
    UpdatesWatcher.RegisterCallback(this);
}

TPaymentsManager::~TPaymentsManager() {
    UpdatesWatcher.UnregisterCallback(this);
}

bool TPaymentsManager::GetPayments(TMap<TString, TCachedPayments>& payments, const TString& condition, NDrive::TEntitySession& session, bool skipFailed) const {
    NStorage::TObjectRecordsSet<TPaymentTask> records;
    auto table = Database->GetTable("drive_payments");
    auto reqResult = table->GetRows(condition, records, session.GetTransaction());
    if (!reqResult || !reqResult->IsSucceed()) {
        ERROR_LOG << session.GetTransaction()->GetErrors().GetStringReport() << Endl;
        return false;
    }
    for (auto&& rec : records) {
        payments[rec.GetSessionId()].Update(rec, skipFailed);
    }
    return true;
}

bool TPaymentsManager::GetPayments(TCachedPayments& payments, const TString& sessionId, NDrive::TEntitySession& session, bool skipFailed) const {
    TMap<TString, TCachedPayments> result;
    if (!GetPayments(result, NContainer::Scalar(sessionId), session, skipFailed)) {
        return false;
    }
    auto it = result.find(sessionId);
    if (it != result.end()) {
        payments = std::move(it->second);
    }
    return true;
}

bool TPaymentsManager::GetPayments(TMap<TString, TCachedPayments>& payments, TConstArrayRef<TString> sessions, NDrive::TEntitySession& session, bool skipFailed) const {
    if (sessions.empty()) {
        return true;
    }
    NSQL::TQueryOptions options;
    options.SetOrderBy({"id"});
    options.SetGenericCondition("session_id", MakeSet<TString>(sessions));
    return GetPayments(payments, options.PrintConditions(*session.GetTransaction()), session, skipFailed);
}

TOptionalPayments TPaymentsManager::GetPayments(NDrive::TEntitySession& session, const NSQL::TQueryOptions& options) const {
    if (options.Empty()) {
        return TVector<TPaymentTask>();
    }
    NStorage::TObjectRecordsSet<TPaymentTask> records;
    auto table = Database->GetTable("drive_payments");
    auto reqResult = table->GetRows(options.PrintConditions(*session.GetTransaction()), records, session.GetTransaction());
    if (!reqResult || !reqResult->IsSucceed()) {
        return {};
    }
    return records.GetObjects();
}

TOptionalPayments TPaymentsManager::GetPayments(TConstArrayRef<TString> paymentIds, NDrive::TEntitySession& session) const {
    return GetPayments(session, NSQL::TQueryOptions().SetGenericCondition("payment_id", MakeSet<TString>(paymentIds)));
}

TMaybe<TMap<TString, TPaymentsData>> TPaymentsManager::GetSessionsPayments(TConstArrayRef<TBillingTask> tasks, TInstant greaterThan, bool ignoreErrors) const {
    TSet<TStringBuf> ids;
    TMap<TString, const TBillingTask*> tasksMap;
    for (auto&& billingTask : tasks) {
        ids.emplace(billingTask.GetId());
        tasksMap[billingTask.GetId()] = &billingTask;
    }

    TVector<TCachedPayments> snapshots;
    if (!GetCustomObjectsFromCache(snapshots, ids, greaterThan)) {
        return {};
    }
    TMap<TString, TPaymentsData> result;
    for (auto&& snapshot : snapshots) {
        const TBillingTask* billingTask = tasksMap[snapshot.GetSessionId()];
        CHECK_WITH_LOG(billingTask);
        auto payments = TPaymentsData::BuildPaymentsData(*billingTask, std::move(snapshot), ignoreErrors);
        if (!payments) {
            continue;
        }
        result.emplace(billingTask->GetId(), *payments);
    }

    for (auto&& billingTask : tasks) {
        if (!result.contains(billingTask.GetId())) {
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments(), ignoreErrors);
            if (!payments) {
                continue;
            }
            result.emplace(billingTask.GetId(), *payments);
        }
    }
    return result;
}


TMaybe<TMap<TString, TPaymentsData>> TPaymentsManager::GetSessionsPayments(TConstArrayRef<TBillingTask> tasks, NDrive::TEntitySession& session, bool ignoreErrors) const {
    TVector<TString> ids;
    TMap<TString, const TBillingTask*> tasksMap;
    for (auto&& billingTask : tasks) {
        ids.emplace_back(billingTask.GetId());
        tasksMap[billingTask.GetId()] = &billingTask;
    }

    TMap<TString, TCachedPayments> snapshots;
    if (!GetPayments(snapshots, ids, session)) {
        return {};
    }
    TMap<TString, TPaymentsData> result;
    for (auto&& billingTask : tasks) {
        TExpected<TPaymentsData, TString> payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments(), ignoreErrors);
        auto it = snapshots.find(billingTask.GetId());
        if (it != snapshots.end()) {
            payments = TPaymentsData::BuildPaymentsData(billingTask, std::move(it->second), ignoreErrors);
            if (!payments) {
                continue;
            }
        }
        result.emplace(billingTask.GetId(), std::move(*payments));
    }
    return result;
}

TMaybe<TPaymentsData> TPaymentsManager::GetFinishedPayments(const TString& sessionId, NDrive::TEntitySession& session) const {
    auto payments = GetFinishedPayments(NContainer::Scalar(sessionId), session);
    if (!payments) {
        return {};
    }
    if (!payments->size()) {
        session.SetErrorInfo("GetFinishedPayments", "cannot fetch stable payments");
        return {};
    }
    return payments->begin()->second;
}

TMaybe<TMap<TString, TPaymentsData>> TPaymentsManager::GetFinishedPayments(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const {
    auto billingTasks = ActiveTasksManager.GetHistoryManager().GetFinishedBillingTasks(sessionIds, session);
    if (!billingTasks) {
        return {};
    }
    auto payments = GetSessionsPayments(MakeVector(NContainer::Values(billingTasks.GetRef())), session, true);
    if (!payments) {
        return {};
    }
    for (auto&& [sessionId, payment] : *payments) {
        if (payment.GetBillingTask().GetLastPaymentId() > 0 && payment.GetBillingTask().GetLastPaymentId() > payment.GetSnapshot().GetVersion()) {
            session.SetErrorInfo("GetFinishedPayments", "incorrect payments version for " + payment.GetBillingTask().GetId());
            return {};
        }
    }
    return *payments;
}

bool TPaymentsManager::RebuildCacheUnsafe() const {
    auto session = BuildSession(true);
    auto table = Database->GetTable("drive_payments");

    NStorage::TObjectRecordsSet<TPaymentTask> records;
    auto reqResult = table->GetRows("session_id in (SELECT session_id from billing_tasks) order by id", records, session.GetTransaction());
    NOTICE_LOG << "Payments before rebuild " << records.size() << "/" << Objects.size() << Endl;
    if (!reqResult->IsSucceed()) {
        NOTICE_LOG << session.GetTransaction()->GetErrors().GetStringReport() << Endl;
        return false;
    }
    Objects.clear();
    for (auto&& rec : records) {
        AcceptEventUnsafe(rec);
    }
    ui64 paymentsCount = 0;
    for (auto&& data : Objects) {
        paymentsCount += data.second.GetPayments().size();
    }
    NOTICE_LOG << "Payments after rebuild " << paymentsCount << "/" << Objects.size() << Endl;
    return true;
}


TStringBuf TPaymentsManager::GetEventObjectId(const TPaymentTask& ev) const {
    return ev.GetPaymentId();
}

void TPaymentsManager::AcceptEventUnsafe(const TPaymentTask& ev) const {
    auto it = Objects.find(ev.GetSessionId());
    if (it != Objects.end()) {
        it->second.Update(ev);
    } else {
        TCachedPayments snapshot;
        if (snapshot.Update(ev)) {
            Objects.emplace(ev.GetSessionId(), std::move(snapshot));
        }
    }
}


bool TPaymentsManager::DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TPaymentTask>& ev, const bool isNewEvent) {
    if (!isNewEvent) {
        return true;
    }
    AcceptEventUnsafe(*ev);
    return true;
}

TBillingSessionContext::TBillingSessionContext(const TPaymentsManager& manager)
    : PaymentsManager(manager)
{}

void TBillingSessionContext::OnAfterCommit() {
    for (auto&& update : Updates) {
        PaymentsManager.UpdatePaymentVersion(update.first, update.second);
    }
}
