#pragma once
#include "db_accessors.h"
#include "entities.h"
#include "history.h"
#include "signals.h"
#include "tasks.h"

#include <drive/backend/database/history/cache.h>
#include <drive/backend/users/user.h>

#include <rtline/util/types/expected.h>

using TExpectedCharge = TExpected<TChargeInfo, EProduceResult>;

struct TBillingGlobals {
    static TSet<NDrive::NTrustClient::EPaymentStatus> AuthStatuses;
    static TSet<NDrive::NTrustClient::EPaymentStatus> FinalStatuses;
    static TSet<NDrive::NTrustClient::EPaymentStatus> HeldStatuses;
    static TSet<NDrive::NTrustClient::EPaymentStatus> FailedStatuses;
    static TSet<TString> SupportedPayMethods;
    static TSet<TString> SupportedCashbackMethods;
    static TSet<TString> SupportedCashbackCurrency;
    static TSet<NDrive::NBilling::EAccount> BonusTypes;
    static TSet<EBillingType> CashbackBillingTypes;
};

class TRefundIssueNew {
    R_READONLY(TPaymentTask, Payment, {});
    R_FIELD(ui32, Sum, 0);
public:
    TRefundIssueNew(const TPaymentTask& payment, ui32 sum)
        : Payment(payment)
        , Sum(sum)
    {}
};

class TCachedPayments {
public:
    struct TErrorInfo {
        TString PaymentMethod;
        TInstant Date = TInstant::Zero();
        TString PaymentError;
        ui32 Count = 1;

        bool operator<(const TErrorInfo& other) const;
        static TString GetPayMethod(const TErrorInfo& info);
        TErrorInfo(const TString& method, TInstant date = TInstant::Zero(), const TString& paymentError = "")
           : PaymentMethod(method)
           , Date(date)
           , PaymentError(paymentError)
        {}
    };

    bool Init(const TVector<TPaymentTask>& payments);
    bool Update(const TPaymentTask& payment, const bool skipFailed = true);
    bool WaitAnyPayments() const;
    TSet<TString> GetFailedPayMethods() const;

    TMaybe<TErrorInfo> GetPayMethodError(const TString& method) const {
        auto it = FailedPayMethods.find({ method });
        if (it != FailedPayMethods.end()) {
            return *it;
        }
        return {};
    }

    const TPaymentTask* GetProcessingPayment() const;

    bool Empty() const {
        return !SessionId;
    };

    TVector<TPaymentTask> GetTimeline() const;
    TExpected<TVector<TRefundIssueNew>, TString> CalculateRefunds(ui32 refundSum, const TVector<TRefundTask>& refunds, ui32 minCardSum) const;

private:
    using TCashbackSumType = TMap<EBillingType, ui32>;
    R_READONLY(TSet<TPaymentTask>, Payments);
    R_READONLY(ui32, HeldSum, 0);
    R_READONLY(ui32, CardSum, 0);
    R_READONLY(ui64, Version, 0);
    R_READONLY(ui32, BonusSum, 0);
    R_READONLY(ui32, CashbackSum, 0);
    R_READONLY(TCashbackSumType, CashbackSumByTypes);
    R_READONLY(ui32, YandexAccountSum, 0);
    R_READONLY(ui32, CanceledSum, 0);
    R_READONLY(TInstant, FirstPaymentTs, TInstant::Max());
    R_READONLY(TString, SessionId);
    R_READONLY(NDrive::NTrustClient::EPaymentStatus, LastStatus, NDrive::NTrustClient::EPaymentStatus::Cleared);

    TSet<TErrorInfo> FailedPayMethods;

    struct TErrorInfoLess {
        bool operator() (const TCachedPayments::TErrorInfo& left, const TCachedPayments::TErrorInfo& right) const {
            return left.Date < right.Date;
        }
    };

public:
    template <class TActionFilter>
    TVector<TString> GetFailedPayMethodsQueue(TActionFilter& action) const {
        std::multiset<TErrorInfo, TErrorInfoLess> queue;
        CopyIf(FailedPayMethods.cbegin(), FailedPayMethods.cend(), std::inserter(queue, queue.begin()), action);
        TVector<TString> result;
        result.reserve(queue.size());
        Transform(queue.begin(), queue.end(), std::back_inserter(result), TErrorInfo::GetPayMethod);
        return result;
    }
};

struct TCorrectedSessionPayments {
    TMap<ui32, ui32> SumByAccount;

    TCorrectedSessionPayments(const TString& userId, ui32 bill)
        : UserId(userId)
        , Bill(bill)
    {}

    bool Init(const TCachedPayments& snapshot, ui32 minCardSum);

private:
    R_READONLY(TString, UserId);
    R_READONLY(ui32, Bill, 0);
    R_FIELD(ui32, CashbackBaseSum, 0);
};

class TPaymentsData {
    R_READONLY(TBillingTask, BillingTask, {});
    R_FIELD(bool, OnFinalization, false);
    TCachedPayments Snapshot;

public:
    using TPtr = TAtomicSharedPtr<TPaymentsData>;

    const TCachedPayments& GetSnapshot() const {
        return Snapshot;
    }

    EBillingType GetType() const {
        return BillingTask.GetBillingType();
    }

    ui32 GetDebt() const {
        if (!BillingTask.IsDebt()) {
            return 0;
        }

        ui32 bill = (BillingTask.IsFinished()) ? BillingTask.GetBillCorrected() : Max<ui32>(BillingTask.GetBillCorrected(), BillingTask.GetDeposit());
        ui32 heldSum = Snapshot.GetHeldSum();
        if (bill > heldSum) {
            return bill - heldSum;
        }
        return 0;
    }

    bool IsClosed() const {
        return BillingTask.IsFinished() && !OnFinalization;
    }

    ui32 GetActualCashback() const;

    static TExpected<TPaymentsData, TString> BuildPaymentsData(const TBillingTask& bTask, TCachedPayments&& snapshot, bool ignoreErrors = false) {
        if (!ignoreErrors && !snapshot.Empty() && bTask.GetLastPaymentId() != 0 && bTask.GetLastPaymentId() != snapshot.GetVersion()) {
            return MakeUnexpected<TString>(TStringBuilder() << bTask.GetLastPaymentId() << "/" << snapshot.GetVersion());
        }
        return TPaymentsData(bTask, std::move(snapshot));
    }

private:
    TPaymentsData(const TBillingTask& bTask, TCachedPayments&& snapshot)
        : BillingTask(bTask)
        , Snapshot(std::move(snapshot))
    {
    }
};

class TPaymentsManager;

class TBillingSessionContext : public NDrive::ISessionContext {
public:
    TBillingSessionContext(const TPaymentsManager& manager);
    virtual void OnAfterCommit() override;

    void SetVersion(const TString& sessionId, ui64 version) {
        Updates[sessionId] = version;
    }

    TMutex& GetMutex() {
        return Mutex;
    }

private:
    const TPaymentsManager& PaymentsManager;
    TMap<TString, ui64> Updates;
    TMutex Mutex;
};

class TPaymentsUpdatesWatcher : public TCallbackSequentialTableImpl<TPaymentTask, TStringBuf, TCachedPayments> {
private:
    using TBase = TCallbackSequentialTableImpl<TPaymentTask, TStringBuf, TCachedPayments>;
public:
    TPaymentsUpdatesWatcher(const IHistoryContext& context)
        : TBase(context, "drive_payments", THistoryConfig()
            .SetDeep(TDuration::Zero())
        )
    {
    }

    virtual TString GetTimestampFieldName() const override {
        return "last_update_ts";
    }

    virtual TString GetSeqFieldName() const override {
        return "id";
    }
};

class TPaymentsManager : public IHistoryCallback<TPaymentTask, TStringBuf, TCachedPayments>, public IStartStopProcess {
private:
    using TBase = IHistoryCallback<TPaymentTask, TStringBuf, TCachedPayments>;

public:
    TPaymentsManager(const IHistoryContext& context, const NDrive::NBilling::TActiveTasksManager& activeTasksManager, bool useCache);
    ~TPaymentsManager();

    virtual bool DoStart() override {
        if (UseCache) {
            if (!UpdatesWatcher.Start()) {
                return false;
            }
            auto g = MakeObjectWriteGuard();
            return RebuildCacheUnsafe();
        }
        return true;
    }

    virtual bool DoStop() override {
        if (UseCache) {
            return UpdatesWatcher.Stop();
        }
        return true;
    }

    const NDrive::NBilling::TActiveTasksManager& GetActiveTasksManager() const {
        return ActiveTasksManager;
    }

    TInstant GetLastUpdateTimestamp() const override {
        return UpdatesWatcher.GetCurrentInstant();
    }

    NDrive::TEntitySession BuildSession(bool readonly = false) const {
        NDrive::TEntitySession session(Database->CreateTransaction(readonly), new TBillingSessionContext(*this));
        return session;
    }

    bool GetPayments(TCachedPayments& payments, const TString& sessionId, NDrive::TEntitySession& session, bool skipFailed = true) const;
    bool GetPayments(TMap<TString, TCachedPayments>& payments, TConstArrayRef<TString> sessions, NDrive::TEntitySession& session, bool skipFailed = true) const;
    bool GetPayments(TMap<TString, TCachedPayments>& payments, const TString& condition, NDrive::TEntitySession& session, bool skipFailed = true) const;
    TOptionalPayments GetPayments(TConstArrayRef<TString> paymentIds, NDrive::TEntitySession& session) const;
    TOptionalPayments GetPayments(NDrive::TEntitySession& session, const NSQL::TQueryOptions& options) const;

    TMaybe<TMap<TString, TPaymentsData>> GetSessionsPayments(TConstArrayRef<TBillingTask> tasks, TInstant reqActuality, bool ignoreErrors = false) const;
    TMaybe<TMap<TString, TPaymentsData>> GetSessionsPayments(TConstArrayRef<TBillingTask> tasks, NDrive::TEntitySession& session, bool ignoreErrors = false) const;
    TMaybe<TPaymentsData> GetFinishedPayments(const TString& sessionId, NDrive::TEntitySession& session) const;
    TMaybe<TMap<TString, TPaymentsData>> GetFinishedPayments(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& session) const;

    EDriveOpResult RegisterPayment(const NStorage::TTableRecord& record, TPaymentTask& newPayment, NDrive::TEntitySession& session, const TMaybe<ui64> oldPaymentId = {}) const;
    EDriveOpResult UpdatePayment(const NStorage::TTableRecord& record, const TPaymentTask& payment, TPaymentTask& newPayment, NDrive::TEntitySession& session) const;
    bool MarkStarted(const TString& orderId, const TString& paymentId, bool updateBillingTask) const;

    EDriveOpResult UpdateTaskStatus(const TString& sessionId, TMaybe<EPaymentStatus> paymentStatus, NDrive::TEntitySession& session) const;
    EDriveOpResult UpdatePaymentStatus(const TPaymentTask& paymentTask, TMaybe<EPaymentStatus> paymentStatus, NDrive::TEntitySession& session, const TMaybe<ui64> oldPaymentId = {}) const;

    ui64 GetPaymentVersion(const TString& sessionId) const;
    void UpdatePaymentVersion(const TString& sessionId, ui64 version) const;

    const TRefundsDB& GetRefundsDB() const {
        return Refunds;
    }

    bool GetUseCache() const {
        return UseCache;
    }

    bool CheckSnapshotVersion(const TString& sessionId, const TCachedPayments& payments) const {
        ui64 currentVersion = GetPaymentVersion(sessionId);
        if (currentVersion == Max<ui64>()) {
            return true;
        }
        if (payments.GetVersion() == Max<ui64>()) {
            return false;
        }
        if (payments.GetVersion() >= currentVersion) {
            return true;
        }
        return false;
    }

protected:
    bool RefreshCache(const TInstant reqActuality, const bool /*doActualizeHistory*/) const override {
        if (UseCache) {
            return UpdatesWatcher.Update(reqActuality);
        }
        return true;
    }
    bool RebuildCacheUnsafe() const;
    virtual TStringBuf GetEventObjectId(const TPaymentTask& ev) const override;
    virtual bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TPaymentTask>& dbEvent, const bool isNewEvent) override;
    void AcceptEventUnsafe(const TPaymentTask& ev) const;

private:
    TDatabasePtr Database;
    TPaymentsUpdatesWatcher UpdatesWatcher;
    TRefundsDB Refunds;
    const NDrive::NBilling::TActiveTasksManager& ActiveTasksManager;
    TUnistatSignal<double> ReloadErrorsCount;
    TRWMutex VersionsLock;
    mutable TMap<TString, ui64> VersionBySession;
    bool UseCache;
};
