#include "tasks.h"

#include "entities.h"
#include "logic.h"

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TBillingTask::TCashbackInfo& result) {
    return
        NJson::ParseField(value, "value", result.OptionalValue()) &&
        NJson::ParseField(value, "billing_type", NJson::Stringify(result.MutableBillingType())) &&
        NJson::ParseField(value, "payload", result.MutablePayload(), true);
}

template <>
NJson::TJsonValue NJson::ToJson(const TBillingTask::TCashbackInfo& object) {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "value", object.OptionalValue());
    NJson::InsertField(result, "billing_type", NJson::Stringify(object.GetBillingType()));
    NJson::InsertField(result, "payload", object.GetPayload());
    return result;
}

bool TBillingTask::TCashbacksInfo::Parse(const NJson::TJsonValue& json) {
    if (json.IsMap()) {
        TCashbackInfo info;
        info.SetPayload(json);
        Parts.emplace_back(std::move(info));
        return true;
    }
    return NJson::ParseField(json, Parts, true);
}

NJson::TJsonValue TBillingTask::TCashbacksInfo::Serialize() const {
    if (Parts.size() == 1 && !Parts.front().HasValue()) {
        return Parts.front().GetPayload();
    }
    return NJson::ToJson(Parts);
}

void TBillingTask::TCashbacksInfo::CorrectCashback(const TMap<EBillingType, ui32>& alreadyPaid, ui32& cashbackSum, TMaybe<NJson::TJsonValue>& payload, EBillingType& type) const {
    int commonInfoInd = -1;
    TMap<EBillingType, ui32> sum;
    for (size_t i = 0; i < Parts.size(); ++i) {
        auto& info = Parts[i];
        if (info.HasValue()) {
            auto sumIt = sum.find(info.GetBillingType());
            if (sumIt == sum.end()) {
                sumIt = sum.emplace(info.GetBillingType(), 0).first;
            }
            sumIt->second += info.GetValueRef();
            ui32 alreadyPaidSum = alreadyPaid.Value(info.GetBillingType(), 0);
            if (sumIt->second > alreadyPaidSum) {
                cashbackSum = Min<ui32>(cashbackSum, sumIt->second - alreadyPaidSum);
                payload = info.GetPayload();
                type = info.GetBillingType();
                return;
            }
        } else {
            commonInfoInd = i;
        }
    }
    if (commonInfoInd >= 0) {
        payload = Parts[commonInfoInd].GetPayload();
        type = Parts[commonInfoInd].GetBillingType();
    }
}

bool TBillingTask::DeserializeWithDecoder(const TBillingTaskDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Bill);
    READ_DECODER_VALUE(decoder, values, TaskStatus);
    READ_DECODER_VALUE(decoder, values, Deposit);
    READ_DECODER_VALUE(decoder, values, BillingType);
    READ_DECODER_VALUE(decoder, values, Discretization);
    if (decoder.GetLastPaymentUpdate() != -1) {
        READ_DECODER_VALUE_INSTANT(decoder, values, LastPaymentUpdate);
    }

    READ_DECODER_VALUE(decoder, values, State);
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE_DEF(decoder, values, LastPaymentId, 0);

    if (decoder.GetMeta() != -1) {
        NJson::TJsonValue meta(NJson::JSON_MAP);
        READ_DECODER_VALUE_JSON(decoder, values, meta, Meta);
        if (meta["accounts"].IsArray()) {
            for (auto&& account : meta["accounts"].GetArray()) {
                ChargableAccounts.emplace(account.GetStringRobust());
            }
        } else {
            // Old tasks compatibility
            ChargableAccounts.emplace("card");
            if (BillingType == EBillingType::CarUsage) {
                ChargableAccounts.emplace("bonus");
            }
        }
        if (meta.Has("max_bonus")) {
            AvailableBonus = meta["max_bonus"].GetInteger();
        }
        if (meta.Has("max_yandex_bonus")) {
            AvailableYandexBonus = meta["max_yandex_bonus"].GetUInteger();
        }
        if (!NJson::ParseField(meta, "comment", Comment, false)
          || !NJson::ParseField(meta, "real_session_id", RealSessionId, false)
          || !NJson::ParseField(meta, "offer_type", OfferType, false)
          || !NJson::ParseField(meta, "offer_name", OfferName, false)
        ) {
            return false;
        }
        if (meta.Has("cashback_percent")) {
            CashbackPercent = meta["cashback_percent"].GetUInteger();
        }

        if (meta.Has("is_plus_user")) {
            IsPlusUser = meta["is_plus_user"].GetBoolean();
        }

        if (meta.Has("use_only_selected_paymethod")) {
            UseOnlySelectedPaymethod = meta["use_only_selected_paymethod"].GetBoolean();
        }

        if (!NJson::ParseField(meta, "session_finish_time", SessionFinishTime, false)) {
            return false;
        }

        for (auto&& item : meta.GetMap()) {
            NDrive::NBilling::EAccount logicType;
            if (TryFromString(item.first, logicType)) {
                auto builder = ILogicConfig::TFactory::Construct(logicType);
                if (builder) {
                    auto logicData = builder->ConstructData(item.second);
                    if (logicData) {
                        LogicsData.insert({ logicType, logicData });
                    }
                }
            }
        }

        if (meta["mobile_paymethods"].IsArray()) {
            for (auto&& methodJson : meta["mobile_paymethods"].GetArray()) {
                TMobilePaymethod method;
                if (!method.Parse(methodJson)) {
                    return false;
                }
                MobilePaymethods.push_back(method);
            }
        }

        if (meta.Has("cashback_info")) {
            CashbacksInfo.ConstructInPlace();
            if (!CashbacksInfo->Parse(meta["cashback_info"])) {
                return false;
            }
        }
    }

    if (decoder.GetExecContext() != -1) {
        NJson::TJsonValue context(NJson::JSON_MAP);
        READ_DECODER_VALUE_JSON(decoder, values, context, ExecContext);
        if (!ExecContext.Parse(context)) {
            return false;
        }
        LastUpdate = ExecContext.GetLastUpdate();
        if (!LastUpdate) {
            READ_DECODER_VALUE_INSTANT(decoder, values, LastUpdate);
        }
    }

    if (MobilePaymethods || ExecContext.GetMobilePaymethod()) {
        ChargableAccounts.emplace("mobile_payment");
    }

    if (decoder.GetEventId() != -1) {
        READ_DECODER_VALUE_DEF(decoder, values, EventId, 0);
    }

    if (decoder.GetCashback() != -1) {
        READ_DECODER_VALUE_DEF(decoder, values, Cashback, 0);
    }

    READ_DECODER_VALUE_DEF(decoder, values, Queue, EBillingQueue::Active);
    READ_DECODER_VALUE_DEF(decoder, values, NextQueue, EBillingQueue::Active);
    return true;
}

NJson::TJsonValue TBillingTask::SerializeMetaToJson() const {
    NJson::TJsonValue meta(NJson::JSON_MAP);
    NJson::TJsonValue& accountsJson = meta.InsertValue("accounts", NJson::JSON_ARRAY);
    for (auto&& account : ChargableAccounts) {
        accountsJson.AppendValue(account);
    }
    if (AvailableBonus > 0) {
        meta["max_bonus"] = AvailableBonus;
    }
    if (AvailableYandexBonus > 0) {
        meta["max_yandex_bonus"] = AvailableYandexBonus;
    }
    if (CashbackPercent > 0) {
        meta["cashback_percent"] = CashbackPercent;
    }
    for (auto&& logicData : LogicsData) {
        meta[::ToString(logicData.first)] = logicData.second->SerializeToJson();
    }
    if (Comment) {
        meta["comment"] = Comment;
    }
    if (RealSessionId) {
        meta["real_session_id"] = RealSessionId;
    }
    if (IsPlusUser) {
        meta["is_plus_user"] = *IsPlusUser;
    }
    if (SessionFinishTime) {
        meta["session_finish_time"] = SessionFinishTime->Seconds();
    }
    meta["use_only_selected_paymethod"] = UseOnlySelectedPaymethod;
    if (OfferType) {
        meta["offer_type"] = OfferType;
    }
    if (OfferName) {
        meta["offer_name"] = OfferName;
    }
    NJson::TJsonValue& mobilePaymethodsJson = meta.InsertValue("mobile_paymethods", NJson::JSON_ARRAY);
    for (auto&& method : MobilePaymethods) {
        mobilePaymethodsJson.AppendValue(method.Serialize());
    }
    if (CashbacksInfo) {
        meta["cashback_info"] = CashbacksInfo->Serialize();
    }
    return meta;
}

NJson::TJsonValue TBillingTask::SerializeToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result["session_id"] = Id;
    result["bill"] = Bill;
    result["cashback"] = Cashback;
    result["user_id"] = UserId;
    result["billing_type"] = ToString(BillingType);
    result["state"] = State;
    result["deposit"] = Deposit;
    result["discretization"] = Discretization;
    result["last_status_update"] = LastUpdate.Seconds();
    result["task_status"] = ToString(TaskStatus);
    result["next_queue"] = ToString(NextQueue);
    result["queue"] = ToString(Queue);
    result["last_payment_id"] = LastPaymentId;
    result["event_id"] = EventId;
    result["meta"] = SerializeMetaToJson();
    result["exec_context"] = ExecContext.Serialize();
    return result;
}

NStorage::TTableRecord TBillingTask::SerializeToTableRecord() const {
    NStorage::TTableRecord record;
    record.Set("session_id", Id);
    record.Set("bill", Bill);
    record.Set("cashback", Cashback);
    record.Set("user_id", UserId);
    record.Set("billing_type", ToString(BillingType));
    record.Set("state", State);
    record.Set("deposit", Deposit);
    record.Set("discretization", Discretization);
    record.Set("last_status_update", LastUpdate.Seconds());
    record.Set("task_status", TaskStatus);
    record.Set("next_queue", NextQueue);
    record.Set("queue", Queue);
    record.Set("last_payment_id", LastPaymentId);
    if (EventId == 0) {
        // SQLLite compatibility
        record.Set("event_id", "nextval('billing_tasks_event_id_seq')");
    }
    NJson::TJsonValue meta = SerializeMetaToJson();
    record.Set("meta", meta.GetStringRobust());
    record.Set("exec_context", ExecContext.Serialize().GetStringRobust());
    return record;
}

bool TPaymentTask::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors) {
    NJson::TJsonValue jsonMeta;
    if (!NJson::ParseField(json, "meta", jsonMeta, /* required = */ false, errors)) {
        return false;
    }
    if (!(jsonMeta.IsString() && jsonMeta.GetString().empty())) {
        NJson::TJsonValue preparsedMeta;
        if (ReadJsonTree(jsonMeta.GetString(), &preparsedMeta)) {
            jsonMeta = preparsedMeta;
        }
        if (!DeserializeMeta(jsonMeta)) {
            return false;
        }
    }
    return
        NJson::ParseField(json, "id", Id, /* required = */ true, errors)
        && NJson::ParseField(json, "account_id", AccountId, /* required = */ false, errors)

        && NJson::ParseField(json, "billing_type", NJson::Stringify(BillingType), /* required = */ true, errors)
        && NJson::ParseField(json, "payment_type", NJson::Stringify(PaymentType), /* required = */ true, errors)
        && NJson::ParseField(json, "sum", Sum, /* required = */ true, errors)
        && NJson::ParseField(json, "cleared", Cleared, /* required = */ true, errors)
        && NJson::ParseField(json, "status", NJson::Stringify(Status), /* required = */ true, errors)

        && NJson::ParseField(json, "payment_id", PaymentId, /* required = */ false, errors)
        && NJson::ParseField(json, "order_id", OrderId, /* required = */ false, errors)
        && NJson::ParseField(json, "payment_error", PaymentError, /* required = */ false, errors)
        && NJson::ParseField(json, "payment_error_desc", PaymentErrorDescription, /* required = */ false, errors)
        && NJson::ParseField(json, "session_id", SessionId, /* required = */ false, errors)
        && NJson::ParseField(json, "pay_method", PayMethod, /* required = */ false, errors)
        && NJson::ParseField(json, "rrn", RRN, /* required = */ false, errors)
        && NJson::ParseField(json, "card_mask", CardMask, /* required = */ false, errors)
        && NJson::ParseField(json, "user_id", UserId, /* required = */ false, errors)

        && NJson::ParseField(json, "last_update_ts", LastUpdate, /* required = */ true, errors)
        && NJson::ParseField(json, "created_at_ts", CreatedAt, /* required = */ true, errors);
}
