#pragma once

#include "account.h"
#include "compiled.h"
#include "entities.h"

#include <drive/backend/database/history/manager.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

enum class EPaymentStatus {
    NotStarted = 0 /* "not_started" */,
    Ok = 1 /* "ok" */,
    NoFunds = 2 /* "no_funds" */,
    NoCards = 3 /* "no_cards" */,
    CreationError = 4 /* "creation_error" */,
    StartingError = 5 /* "starting_error" */,
    InternalError = 6 /* "internal_error" */,
    RefundDeposit = 7 /* "refund_deposit" */,
    IncorrectYAccount = 8 /* "incorrect_yaccount" */,
    DeletedUser = 9 /* "deleted_user" */,
};

class ILogicData {
public:
    using TPtr = TAtomicSharedPtr<ILogicData>;
    virtual ~ILogicData() {};
    virtual bool DeserializeFromJson(const NJson::TJsonValue& json) = 0;
    virtual NJson::TJsonValue SerializeToJson() const = 0;
    template<class T>
    const T& GetImpl() const {
        const T* res = VerifyDynamicCast<const T*>(this);
        return *res;
    }
};

using TLogicsData = TMap<NDrive::NBilling::EAccount, ILogicData::TPtr>;

class TBillingTask {
public:
    enum class EState {
        Active /* "active" */,
        Finishing /* "finishing" */,
        Finished /* "finished" */,
        Canceled /* "canceled" */,
    };
public:
    struct TExecContext {
    public:
        R_FIELD(TInstant, LastUpdate, TInstant::Zero());
        R_FIELD(TString, Paymethod);
        R_FIELD(TString, UserId);
        R_FIELD(TString, MobilePaymethod);

    public:
        TExecContext() = default;

        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "last_update", LastUpdate, false)
                && NJson::ParseField(json, "paymethod", Paymethod, false)
                && NJson::ParseField(json, "mobile_paymethod", MobilePaymethod, false)
                && NJson::ParseField(json, "user_id", UserId, false);
        }

        NJson::TJsonValue Serialize() const {
            NJson::TJsonValue result;
            NJson::InsertField(result, "last_update", LastUpdate.Seconds());
            NJson::InsertNonNull(result, "paymethod", Paymethod);
            NJson::InsertNonNull(result, "mobile_paymethod", MobilePaymethod);
            NJson::InsertNonNull(result, "user_id", UserId);
            return result;
        }
    };

    struct TMobilePaymethod {
    public:
        R_FIELD(TString, MobilePaymethodId);

    public:
        TMobilePaymethod() = default;

        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "mobile_paymethod_id", MobilePaymethodId, true);
        }

        NJson::TJsonValue Serialize() const {
            NJson::TJsonValue result;
            NJson::InsertField(result, "mobile_paymethod_id", MobilePaymethodId);
            return result;
        }
    };

    struct TCashbackInfo {
    public:
        R_OPTIONAL(ui32, Value);
        R_FIELD(NJson::TJsonValue, Payload);
        R_FIELD(EBillingType, BillingType, EBillingType::YCashback);

    public:
        TCashbackInfo() = default;
        TCashbackInfo(const TMaybe<ui32>& value, const NJson::TJsonValue& payload, EBillingType type = EBillingType::YCashback)
            : Value(value)
            , Payload(payload)
            , BillingType(type)
        {}
    };

    class TCashbacksInfo {
    public:
        R_FIELD(TVector<TCashbackInfo>, Parts);

    public:
        TCashbacksInfo() = default;

        bool Parse(const NJson::TJsonValue& json);
        NJson::TJsonValue Serialize() const;

        void CorrectCashback(const TMap<EBillingType, ui32>& alreadyPaid, ui32& cashbackSum, TMaybe<NJson::TJsonValue>& payload, EBillingType& type) const;
    };

public:
    R_FIELD(TString, Id);
    R_FIELD(EBillingType, BillingType, EBillingType::CarUsage);
    R_FIELD(TString, UserId);
    R_READONLY(TString, State, "active");
    R_FIELD(double, Bill, 0.0);
    R_FIELD(double, Cashback, 0.0);
    R_FIELD(ui32, Deposit, 0);
    R_FIELD(ui32, AvailableBonus, 0);
    R_FIELD(ui32, AvailableYandexBonus, 0);
    R_FIELD(ui32, Discretization, 0);
    R_FIELD(EBillingQueue, Queue, EBillingQueue::Active);
    R_FIELD(EBillingQueue, NextQueue, EBillingQueue::Active);
    R_FIELD(TExecContext, ExecContext, {});
    R_READONLY(EPaymentStatus, TaskStatus, EPaymentStatus::NotStarted);
    R_READONLY(TInstant, LastPaymentUpdate, TInstant::Zero());
    R_FIELD(ui64, LastPaymentId, 0);
    R_READONLY(ui64, EventId, 0);

    // From meta
    R_FIELD(TSet<TString>, ChargableAccounts, {});
    R_READONLY(TInstant, LastUpdate, TInstant::Zero());
    R_READONLY(TLogicsData, LogicsData, {});
    R_FIELD(TString, Comment);
    R_FIELD(TString, RealSessionId);
    R_FIELD(ui32, CashbackPercent, 0);
    R_OPTIONAL(bool, IsPlusUser);
    R_OPTIONAL(TInstant, SessionFinishTime);
    R_FIELD(bool, UseOnlySelectedPaymethod, false);
    R_FIELD(TString, OfferType);
    R_FIELD(TString, OfferName);
    R_FIELD(TVector<TMobilePaymethod>, MobilePaymethods, {});
    R_OPTIONAL(TCashbacksInfo, CashbacksInfo);

public:
    void AddCashbackInfo(const TCashbackInfo& info) {
        if (!CashbacksInfo) {
            CashbacksInfo.ConstructInPlace();
        }
        CashbacksInfo->MutableParts().push_back(info);
    }

    ui32 GetBillCorrected() const {
        return std::floor(Bill);
    }

    const TString& GetHistoryUserId() const {
        return Default<TString>();
    }

    TBillingTask& SetLastUpdate(TInstant ts) {
        LastUpdate = ts;
        ExecContext.SetLastUpdate(ts);
        return *this;
    }

    ILogicData::TPtr GetLogicData(NDrive::NBilling::EAccount logicType) const {
        auto it = LogicsData.find(logicType);
        if (it != LogicsData.end()) {
            return it->second;
        }
        return nullptr;
    }

    bool SetLogicData(NDrive::NBilling::EAccount logicType, ILogicData::TPtr data) {
        return LogicsData.insert({ logicType, data }).second;
    }

    class TBillingTaskDecoder : public TBaseDecoder {
        R_FIELD(i32, Bill, -1);
        R_FIELD(i32, TaskStatus, -1);
        R_FIELD(i32, Deposit, -1);
        R_FIELD(i32, BillingType, -1);
        R_FIELD(i32, Discretization, -1);
        R_FIELD(i32, LastUpdate, -1);
        R_FIELD(i32, LastPaymentUpdate, -1);
        R_FIELD(i32, LastPaymentId, -1);
        R_FIELD(i32, State, -1);
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, Queue, -1);
        R_FIELD(i32, NextQueue, -1);
        R_FIELD(i32, Meta, -1);
        R_FIELD(i32, ExecContext, -1);
        R_FIELD(i32, EventId, -1);
        R_FIELD(i32, Cashback, -1);

    public:
        TBillingTaskDecoder() = default;
        TBillingTaskDecoder(const TMap<TString, ui32>& decoderBase) {
            Bill = GetFieldDecodeIndex("bill", decoderBase);
            TaskStatus = GetFieldDecodeIndex("task_status", decoderBase);
            Deposit = GetFieldDecodeIndex("deposit", decoderBase);
            BillingType = GetFieldDecodeIndex("billing_type", decoderBase);
            Discretization = GetFieldDecodeIndex("discretization", decoderBase);
            LastUpdate = GetFieldDecodeIndex("last_status_update", decoderBase);
            LastPaymentUpdate = GetFieldDecodeIndex("last_payment_update", decoderBase);
            LastPaymentId = GetFieldDecodeIndex("last_payment_id", decoderBase);
            State = GetFieldDecodeIndex("state", decoderBase);
            Id = GetFieldDecodeIndex("session_id", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            Queue = GetFieldDecodeIndex("queue", decoderBase);
            NextQueue = GetFieldDecodeIndex("next_queue", decoderBase);

            Meta = GetFieldDecodeIndex("meta", decoderBase);
            ExecContext = GetFieldDecodeIndex("exec_context", decoderBase);
            EventId = GetFieldDecodeIndex("event_id", decoderBase);
            Cashback = GetFieldDecodeIndex("cashback", decoderBase);
        }
    };
    using TDecoder = TBillingTaskDecoder;

    bool DeserializeWithDecoder(const TBillingTaskDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    bool IsFinished() const {
        return State == "finished" || State == "canceled";
    }

    bool CheckDeposit() const {
        if (Deposit > 0 && TaskStatus != EPaymentStatus::Ok) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue SerializeMetaToJson() const;
    NJson::TJsonValue SerializeToJson() const;

    NStorage::TTableRecord SerializeToTableRecord() const;

    const TStringBuf GetHistoryObjectId(const TBillingTask& e) const {
        return e.Id;
    }

    bool IsPriorityTask() const {
        return LastUpdate == TInstant::Zero();
    }

    bool IsDebt() const {
        return TaskStatus == EPaymentStatus::NoCards || TaskStatus == EPaymentStatus::NoFunds;
    }

    bool IsCanceled() const {
        return State == "canceled";
    }

    explicit operator bool() const {
        return !Id.empty();
    }

    bool operator<(const TBillingTask& other) const {
        return Id < other.Id;
    }

    bool operator<(const TString& sessionId) const {
        return Id < sessionId;
    }
};

using TBillingHistoryEvent = TObjectEvent<TBillingTask>;
using TBillingTasks = TVector<TBillingTask>;
using TOptionalBillingTask = TMaybe<TBillingTask>;
using TOptionalBillingTasks = TMaybe<TBillingTasks>;

class TClearingTask {
    R_READONLY(TString, Id);
    R_READONLY(TString, SessionId);
    R_READONLY(TString, PaymentId);
    R_READONLY(TString, OrderId);
    R_READONLY(double, Sum, 0.0);
    R_READONLY(double, Refund, 0.0);
    R_READONLY(TInstant, CreatedAt, TInstant::Zero());
    R_READONLY(EBillingType, BillingType, EBillingType::CarUsage);

public:
    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/) {
        if (!record.TryGet("sum", Sum)) {
            return false;
        }
        if (!record.TryGet("refund", Refund)) {
            return false;
        }
        ui64 ts;
        if (!record.TryGet("created_at_ts", ts)) {
            return false;
        }

        if (!record.TryGet("billing_type", BillingType)) {
            return false;
        }

        CreatedAt = TInstant::Seconds(ts);

        Id = record.Get("id");
        SessionId = record.Get("session_id");
        PaymentId = record.Get("payment_id");
        OrderId = record.Get("order_id");
        return true;
    }

    void Serialize(NStorage::TTableRecord& record) const {
        record.Set("sum", Sum);
        record.Set("refund", Refund);
        record.Set("session_id", SessionId);

        if (CreatedAt) {
            record.Set("created_at_ts", ToString(CreatedAt.Seconds()));
        } else {
            record.Set("created_at_ts", ToString(TInstant::Now().Seconds()));
        }
    }
};

using TClearingTasks = TVector<TClearingTask>;
using TOptionalClearingTasks = TMaybe<TClearingTasks>;

class IPaymentMeta {
public:
    virtual ~IPaymentMeta() {};
    virtual bool Parse(const NJson::TJsonValue& json) = 0;
    virtual NJson::TJsonValue ToJson() const = 0;
    using TPtr = TAtomicSharedPtr<IPaymentMeta>;
    using TFactory = NObjectFactory::TObjectFactory<IPaymentMeta, NDrive::NBilling::EAccount>;
};

class TPaymentTask {
    R_READONLY(ui64, Id, 0);
    R_READONLY(ui64, AccountId, 0);
    R_READONLY(TString, PayMethod);
    R_READONLY(TString, PaymentId);
    R_READONLY(TString, SessionId);
    R_READONLY(TString, RRN);
    R_READONLY(TString, CardMask);
    R_READONLY(TString, PaymentError);
    R_READONLY(TString, PaymentErrorDescription);
    R_READONLY(TString, OrderId);
    R_READONLY(NDrive::NBilling::EAccount, PaymentType, NDrive::NBilling::EAccount::Trust);
    R_READONLY(EBillingType, BillingType, EBillingType::CarUsage);
    R_READONLY(NDrive::NTrustClient::EPaymentStatus, Status, NDrive::NTrustClient::EPaymentStatus::Unknown);
    R_READONLY(ui32, Sum, 0);
    R_READONLY(ui32, Cleared, 0);
    R_READONLY(TInstant, LastUpdate, TInstant::Zero());
    R_READONLY(TInstant, CreatedAt, TInstant::Zero());
    R_READONLY(IPaymentMeta::TPtr, Meta);
    R_READONLY(TString, UserId);

public:
    using TEventId = NDrive::TEventId;
    using TEntity = TPaymentTask;

public:
    static constexpr auto IncorrectEventId = NDrive::IncorrectEventId;

public:
    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/) {
        record.TryGet("id", Id);
        record.TryGet("account_id", AccountId);

        if (!record.TryGet("billing_type", BillingType)) {
            return false;
        }
        if (!record.TryGet("payment_type", PaymentType)) {
            return false;
        }
        if (!record.TryGet("sum", Sum)) {
            return false;
        }
        if (!record.TryGet("cleared", Cleared)) {
            return false;
        }
        if (!record.TryGet("status", Status)) {
            return false;
        }

        record.TryGet("payment_id", PaymentId);
        record.TryGet("order_id", OrderId);
        record.TryGet("payment_error", PaymentError);
        record.TryGet("payment_error_desc", PaymentErrorDescription);
        record.TryGet("session_id", SessionId);
        record.TryGet("pay_method", PayMethod);
        record.TryGet("rrn", RRN);
        record.TryGet("card_mask", CardMask);
        record.TryGet("user_id", UserId);

        ui64 lastUpdate = 0;
        if (!record.TryGet("last_update_ts", lastUpdate)) {
            return false;
        }
        LastUpdate = TInstant::Seconds(lastUpdate);

        ui64 created = 0;
        if (!record.TryGet("created_at_ts", created)) {
            return false;
        }
        CreatedAt = TInstant::Seconds(created);

        NJson::TJsonValue jsonMeta;
        if (!record.TryGet("meta", jsonMeta)) {
            return false;
        }
        return DeserializeMeta(jsonMeta);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);

    void ToJson(NJson::TJsonValue& json) const {
        NStorage::TTableRecord record = SerializeToTableRecord();
        json = record.SerializeToJson();
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord record;
        record.Set("session_id", SessionId);
        record.Set("order_id", OrderId);
        record.Set("id", Id);
        record.Set("account_id", AccountId);
        record.Set("sum", Sum);
        record.Set("cleared", Cleared);
        record.Set("billing_type", BillingType);
        record.Set("payment_type", PaymentType);
        record.Set("pay_method", PayMethod);
        record.Set("payment_id", PaymentId);
        record.Set("status", Status);
        record.Set("created_at_ts", CreatedAt.Seconds());
        record.Set("last_update_ts", LastUpdate.Seconds());
        record.Set("card_mask", CardMask);
        record.Set("rrn", RRN);
        record.Set("payment_error", PaymentError);
        record.Set("payment_error_desc", PaymentErrorDescription);
        if (!!Meta) {
            record.Set("meta", Meta->ToJson());
        } else {
            record.Set("meta", "");
        }
        if (UserId) {
            record.Set("user_id", UserId);
        }
        return record;
    }

    bool operator<(const TPaymentTask& other) const {
        return PaymentId < other.PaymentId;
    }

    static TPaymentTask CreateFromPayment(const NDrive::NTrustClient::TPayment& payment, TInstant createdAt = TInstant::Zero()) {
        TPaymentTask task;
        task.BillingType = EBillingType::CarUsage;
        task.PaymentType = NDrive::NBilling::EAccount::Trust;
        task.Sum = payment.Amount;
        task.Cleared = payment.Cleared;
        task.Status = payment.PaymentStatus;
        task.PaymentId = payment.PurchaseToken;
        task.OrderId = payment.OrderId;
        task.PaymentError = payment.PaymentError;
        task.PayMethod = payment.PaymethodId;
        task.CreatedAt = createdAt;
        return task;
    }

    class TPaymentTaskDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, AccountId, -1);
        R_FIELD(i32, PayMethod, -1);
        R_FIELD(i32, PaymentId, -1);
        R_FIELD(i32, SessionId, -1);
        R_FIELD(i32, RRN, -1);
        R_FIELD(i32, CardMask, -1);
        R_FIELD(i32, PaymentError, -1);
        R_FIELD(i32, PaymentErrorDescription, -1);
        R_FIELD(i32, OrderId, -1);
        R_FIELD(i32, PaymentType, -1);
        R_FIELD(i32, BillingType, -1);
        R_FIELD(i32, Status, -1);
        R_FIELD(i32, Sum, -1);
        R_FIELD(i32, Cleared, -1);
        R_FIELD(i32, LastUpdate, -1);
        R_FIELD(i32, CreatedAt, -1);
        R_FIELD(i32, Meta, -1);
        R_FIELD(i32, UserId, -1);

    public:
        TPaymentTaskDecoder() = default;

        TPaymentTaskDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            AccountId = GetFieldDecodeIndex("account_id", decoderBase);
            PayMethod = GetFieldDecodeIndex("pay_method", decoderBase);
            PaymentId = GetFieldDecodeIndex("payment_id", decoderBase);
            SessionId = GetFieldDecodeIndex("session_id", decoderBase);
            RRN = GetFieldDecodeIndex("rrn", decoderBase);
            CardMask = GetFieldDecodeIndex("card_mask", decoderBase);
            PaymentError = GetFieldDecodeIndex("payment_error", decoderBase);
            PaymentErrorDescription = GetFieldDecodeIndex("payment_error_desc", decoderBase);
            OrderId = GetFieldDecodeIndex("order_id", decoderBase);
            PaymentType = GetFieldDecodeIndex("payment_type", decoderBase);
            BillingType = GetFieldDecodeIndex("billing_type", decoderBase);
            Status = GetFieldDecodeIndex("status", decoderBase);
            Sum = GetFieldDecodeIndex("sum", decoderBase);
            Cleared = GetFieldDecodeIndex("cleared", decoderBase);
            LastUpdate = GetFieldDecodeIndex("last_update_ts", decoderBase);
            CreatedAt = GetFieldDecodeIndex("created_at_ts", decoderBase);
            Meta = GetFieldDecodeIndex("meta", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
        }
    };

    bool DeserializeWithDecoder(const TPaymentTaskDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE_DEF(decoder, values, AccountId, 0);
        READ_DECODER_VALUE(decoder, values, PayMethod);
        READ_DECODER_VALUE(decoder, values, PaymentId);
        READ_DECODER_VALUE(decoder, values, SessionId);
        READ_DECODER_VALUE(decoder, values, RRN);
        READ_DECODER_VALUE(decoder, values, CardMask);
        READ_DECODER_VALUE(decoder, values, PaymentError);
        READ_DECODER_VALUE(decoder, values, PaymentErrorDescription);
        READ_DECODER_VALUE(decoder, values, OrderId);
        READ_DECODER_VALUE(decoder, values, PaymentType);
        READ_DECODER_VALUE(decoder, values, BillingType);
        READ_DECODER_VALUE(decoder, values, Status);
        READ_DECODER_VALUE(decoder, values, Sum);
        READ_DECODER_VALUE(decoder, values, Cleared);
        READ_DECODER_VALUE_INSTANT(decoder, values, LastUpdate);
        READ_DECODER_VALUE_INSTANT(decoder, values, CreatedAt);
        NJson::TJsonValue jsonMeta;
        READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);
        READ_DECODER_VALUE(decoder, values, UserId);
        return DeserializeMeta(jsonMeta);
    }

    bool DeserializeMeta(const NJson::TJsonValue& jsonMeta) {
        Meta = IPaymentMeta::TFactory::Construct(PaymentType);
        return !Meta || Meta->Parse(jsonMeta);
    }

    using TDecoder = TPaymentTaskDecoder;

    ui64 GetHistoryEventId() const {
        return Id;
    }

    const TInstant& GetHistoryInstant() const {
        return LastUpdate;
    }

    const TString& GetHistoryUserId() const {
        return Default<TString>();
    }
};

using TOptionalPayments = TMaybe<TVector<TPaymentTask>>;

class TRefundTask {
    R_READONLY(ui32, Id, 0);
    R_READONLY(TString, PaymentId);
    R_READONLY(TString, RefundId);
    R_READONLY(TString, SessionId);
    R_READONLY(ui64, OrderId, Max<ui64>());
    R_READONLY(ui32, Sum, 0);
    R_READONLY(TString, Status, "draft");
    R_READONLY(TInstant, CreatedAt, TInstant::Zero());
    R_READONLY(TInstant, LastUpdate, TInstant::Zero());
    R_READONLY(EBillingType, BillingType, EBillingType::CarUsage);
    R_READONLY(TString, UserId);
    R_READONLY(bool, RealRefund, true);

public:
    using TEventId = NDrive::TEventId;

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, PaymentId, -1);
        R_FIELD(i32, SessionId, -1);
        R_FIELD(i32, OrderId, -1);
        R_FIELD(i32, RefundId, -1);
        R_FIELD(i32, BillingType, -1);
        R_FIELD(i32, Status, -1);
        R_FIELD(i32, Sum, -1);
        R_FIELD(i32, LastUpdate, -1);
        R_FIELD(i32, CreatedAt, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, RealRefund, -1);

    public:
        TDecoder() = default;

        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            PaymentId = GetFieldDecodeIndex("payment_id", decoderBase);
            SessionId = GetFieldDecodeIndex("session_id", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            OrderId = GetFieldDecodeIndex("order_id", decoderBase);
            RefundId = GetFieldDecodeIndex("refund_id", decoderBase);
            BillingType = GetFieldDecodeIndex("billing_type", decoderBase);
            Status = GetFieldDecodeIndex("status", decoderBase);
            Sum = GetFieldDecodeIndex("sum", decoderBase);
            LastUpdate = GetFieldDecodeIndex("last_update_ts", decoderBase);
            CreatedAt = GetFieldDecodeIndex("created_at_ts", decoderBase);
            RealRefund = GetFieldDecodeIndex("real_refund", decoderBase);
        }
    };

public:
    static constexpr auto IncorrectEventId = NDrive::IncorrectEventId;

public:
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, PaymentId);
        READ_DECODER_VALUE_DEF(decoder, values, SessionId, Default<TString>());
        READ_DECODER_VALUE_DEF(decoder, values, UserId, Default<TString>());
        READ_DECODER_VALUE_DEF(decoder, values, OrderId, Default<ui64>());
        READ_DECODER_VALUE_DEF(decoder, values, RefundId, Default<TString>());
        READ_DECODER_VALUE(decoder, values, BillingType);
        READ_DECODER_VALUE(decoder, values, Status);
        READ_DECODER_VALUE(decoder, values, Sum);
        READ_DECODER_VALUE_INSTANT(decoder, values, LastUpdate);
        READ_DECODER_VALUE_INSTANT(decoder, values, CreatedAt);
        if (decoder.GetRealRefund() > 0) {
            READ_DECODER_VALUE_DEF(decoder, values, RealRefund, true);
        }
        return true;
    }

    bool GetFinished() const {
        return Status == "started" || Status == "success";
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord record;
        record.Set("id", Id);
        record.Set("session_id", SessionId);
        record.Set("user_id", UserId);
        record.Set("order_id", OrderId);
        record.Set("refund_id", RefundId);
        record.Set("payment_id", PaymentId);
        record.Set("sum", Sum);
        record.Set("billing_type", BillingType);
        record.Set("status", Status);
        record.Set("created_at_ts", CreatedAt.Seconds());
        record.Set("last_update_ts", LastUpdate.Seconds());
        record.Set("real_refund", RealRefund);
        return record;
    }

    ui64 GetHistoryEventId() const {
        return Id;
    }

    const TInstant& GetHistoryInstant() const {
        return CreatedAt;
    }

    const TString& GetHistoryUserId() const {
        return Default<TString>();
    }
};
