#include "manager.h"

#include <drive/backend/billing/accounts/bonus.h>
#include <drive/backend/billing/trust/charge_logic.h>
#include <drive/backend/billing/trust/updater.h>
#include <drive/backend/billing/trust_cache.h>
#include <drive/backend/logging/evlog.h>

#include <drive/backend/abstract/notifier.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/uuid.h>

#include <util/generic/adaptor.h>
#include <util/system/env.h>

namespace {
    TNamedSignalSimple BillingSignalsLost("billing-lost-sum");
    TNamedSignalCustom BillingSignalsDebt("billing-debt-tasks-count", EAggregationType::LastValue, "ammm");
    TNamedSignalCustom BillingSignalsDebtSum("billing-debt-tasks-sum", EAggregationType::LastValue, "ammm");

    TNamedSignalEnum<EBillingQueue> SortByBillingQueue("sort-by-billing-queue", EAggregationType::Sum, "dmmm");

    TNamedSignalHistogram SignalBillPriority("billing-task-priority", NRTLineHistogramSignals::IntervalsRTLineReply);
}

using TDurationGuard = TTimeGuardImpl<false, TLOG_INFO, true>;

TFakeLogic::TFakeLogic(const TBillingManager& manager, const ISettings& settings)
    : IChargeLogic(NDrive::NBilling::EAccount::Fake, settings)
    , Manager(manager)
{
}

TExpectedCharge TFakeLogic::ProducePayments(NDrive::NBilling::IBillingAccount::TPtr /*account*/, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager,
                                            const TChargeInfo& /*pContext*/, TLazySession& session, const bool /*sync*/) const {
    const TBillingTask& bTask = snapshot.GetBillingTask();
    TChargeInfo charge(bTask.GetBillingType());
    if (bTask.GetState() == "canceled") {
        charge.Sum = bTask.GetBillCorrected() - snapshot.GetSnapshot().GetHeldSum();
        if (charge.Sum == 0) {
            return charge;
        }
        TPaymentTask fakePayment;
        if (!BuildPaymentTask(nullptr, paymentsManager, bTask, charge, fakePayment, session.Get())) {
            return MakeUnexpected(EProduceResult::TransactionError);
        }

        ui32 refundSum = charge.Sum;
        if (refundSum) {
            NDrive::NBilling::IBillingAccount::TPtr trustAccount = Manager.GetAccountsManager().GetTrustAccount(bTask.GetUserId());
            if (!trustAccount) {
                ERROR_LOG << "Inconsistent user accounts for " << bTask.GetUserId() << Endl;
                return MakeUnexpected(EProduceResult::Error);
            }

            TCompiledRefund refundField;
            refundField.SetBill(refundSum);
            refundField.SetSessionId(bTask.GetId());
            refundField.SetBillingType(bTask.GetBillingType());
            refundField.SetRealSessionId(bTask.GetRealSessionId());
            refundField.SetComment(bTask.GetComment());

            NDrive::NBilling::TFiscalItem details;
            details.SetUniqueName(trustAccount->GetUniqueName());
            details.SetName(trustAccount->GetName());
            details.SetAccountId(trustAccount->GetId());
            details.SetSum(refundSum);
            details.SetType(NDrive::NBilling::EAccount::Trust);
            details.SetTransactionId(NUtil::CreateUUID());
            refundField.SetDetails(NDrive::NBilling::TFiscalDetails().AddItem(std::move(details)));

            if (!Manager.GetCompiledRefunds().AddHistory(refundField, bTask.GetUserId(), EObjectHistoryAction::Add, session.Get())) {
                return MakeUnexpected(EProduceResult::TransactionError);
            }
        }
    }
    if (!session.Commit()) {
        return MakeUnexpected(EProduceResult::TransactionError);
    }
    return charge;
}

bool TFakeLogic::DoRefund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session, NStorage::TTableRecord& /*patchPaymentRecord*/) const {
    Y_UNUSED(account);
    Y_UNUSED(session);
    Y_UNUSED(payment);
    Y_UNUSED(sum);
    return true;
}

TVector<TString> TBillingManager::GetKnownAccounts() const {
    TVector<TString> accounts;
    auto registeredAccounts = AccountsManager.GetRegisteredAccounts();
    for (auto&& description : registeredAccounts) {
        accounts.emplace_back(description.GetName());
    }
    return accounts;
}


bool TBillingManager::OnCycleFinish(const TPaymentsData& paymentsData) const {
    const TBillingTask& billingTask = paymentsData.GetBillingTask();
    const TCachedPayments& payments = paymentsData.GetSnapshot();
    if (payments.WaitAnyPayments() || !billingTask.IsFinished()) {
        return false;
    }

    ui32 bill = billingTask.GetBillCorrected();
    const ui32 minTrustPayment = 100;
    if (payments.GetHeldSum() > bill) {
        bool forceRemove = AccountsManager.GetSettings().GetValueDef<ui32>("billing.force_remove_canceled", false);
        if (forceRemove && billingTask.IsCanceled()) {
            auto session = BuildSession(false);
            if (!RemoveBillingTask(billingTask.GetId(), session) || !session.Commit()) {
                ERROR_LOG << session.GetStringReport() << Endl;
            }
            return true;
        }

        if (billingTask.GetTaskStatus() == EPaymentStatus::RefundDeposit) {
            return true;
        }
        ui32 overhead = bill < minTrustPayment ? payments.GetHeldSum() : payments.GetHeldSum() - bill;
        auto issues = payments.CalculateRefunds(overhead, {}, AccountsManager.GetSettings().GetValue<ui64>("billing.card_pay_min_sum").GetOrElse(0));
        if (!issues) {
            ERROR_LOG << "Can't create refund for " << billingTask.GetId() << " " << issues.GetError() << Endl;
            return true;
        } else {
            auto session = BuildSession(false);
            for (auto&& issue : issues.GetRef()) {
                ui32 sum2refund = issue.GetSum();
                if (issue.GetPayment().GetStatus() == NDrive::NTrustClient::EPaymentStatus::Authorized && issue.GetPayment().GetPaymentType() == NDrive::NBilling::EAccount::Trust) {
                    sum2refund = Max<ui32>(issue.GetSum(), issue.GetPayment().GetSum());
                }
                if (!ProcessFiscalRefundIssue(issue.GetPayment(), billingTask.GetUserId(), sum2refund, false, session)) {
                    ERROR_LOG << "Can't create refund for " << billingTask.GetId() << ": " << session.GetStringReport() << Endl;
                    return true;
                }
            }
            if (PaymentsManager.UpdateTaskStatus(billingTask.GetId(), EPaymentStatus::RefundDeposit, session) != EDriveOpResult::Ok
                || !session.Commit()) {
                ERROR_LOG << session.GetStringReport() << Endl;
                return true;
            }
        }
    } else {
        ui32 restSum = bill - payments.GetHeldSum();
        if (restSum < minTrustPayment) {
            if (billingTask.GetCashback()) {
                auto paymentTimeline = payments.GetTimeline();
                TInstant lastPaymentTime(ModelingNow());
                for (auto&& payment : Reversed(paymentTimeline)) {
                    if (TBillingGlobals::HeldStatuses.contains(payment.GetStatus())) {
                        lastPaymentTime = payment.GetCreatedAt();
                        break;
                    }
                }

                TInstant sessionFinishTime = TInstant::Zero();
                if (billingTask.HasSessionFinishTime()) {
                    sessionFinishTime = billingTask.GetSessionFinishTimeRef();
                }
                if (sessionFinishTime + AccountsManager.GetSettings().GetValueDef<TDuration>("billing.cashback_period", TDuration::Max()) > lastPaymentTime) {
                    auto session = BuildSession(false);
                    auto events = GetCompiledBills().GetBillsFromDB(billingTask.GetId(), session);
                    if (!events) {
                        return true;
                    }
                    auto found = false;
                    for (auto&& bill : *events) {
                        if (bill.GetBillingType() == EBillingType::YCashback) {
                            found = true;
                            if (bill.GetCashback() > payments.GetCashbackSum()) {
                                return false;
                            }
                        }
                    }
                    // TODO: several bills for every type
                    if (!found) {
                        if (!BuildCompiledCashback(paymentsData, EBillingType::YCashback, session) || !session.Commit()) {
                            ERROR_LOG << session.GetStringReport() << Endl;
                        }
                        return true;
                    }
                }
            }
            TUnistatSignalsCache::SignalAdd("cashback", "lost", billingTask.GetCashback() - payments.GetCashbackSum());
            BillingSignalsLost.Signal(restSum);
            auto session = BuildSession(false);
            if (!RemoveBillingTask(billingTask.GetId(), session) || !session.Commit()) {
                ERROR_LOG << session.GetStringReport() << Endl;
            }
            return true;
        }
    }
    return false;
}

bool TBillingManager::SwitchQueue(const TCachedPayments& payments, const TBillingTask& billingTask, const TString& userId, const TMaybe<EBillingQueue> queue, const TMaybe<EBillingQueue> nextQueue) const {
    if (!queue && !nextQueue) {
        return false;
    }
    if (queue && (queue == billingTask.GetQueue() || payments.WaitAnyPayments())) {
        return false;
    }

    NStorage::ITableAccessor::TPtr table = Database->GetTable("billing_tasks");
    NStorage::TTableRecord condition;
    condition.Set("session_id", billingTask.GetId());

    NStorage::TTableRecord update;
    update.Set("event_id", "nextval('billing_tasks_event_id_seq')");

    if (queue) {
        condition.Set("last_payment_id", billingTask.GetLastPaymentId());
        update.Set("queue", ToString(queue));
    }
    if (nextQueue) {
        update.Set("next_queue", ToString(nextQueue));
    }

    auto session = BuildSession(false);
    auto transaction = session.GetTransaction();
    NStorage::TObjectRecordsSet<TBillingTask> affected;
    auto result = table->UpdateRow(condition, update, transaction, &affected);
    if (!result || !result->IsSucceed() || affected.size() != 1) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }

    if (!ActiveTasksManager.GetHistoryManager().AddHistory(affected, userId, EObjectHistoryAction::UpdateData, session)
        || !session.Commit())
    {
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }
    return true;
}

void TBillingManager::ProcessTask(const TClearingTask& task, const TDuration clearingInterval, NDrive::TEntitySession& session, TBillingSyncGuard& guard) const {
    auto logic = GetLogicByType(NDrive::NBilling::EAccount::Trust);
    if (!logic) {
        return;
    }
    const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();
    trustLogic->ProcessClearing(task, PaymentsManager, clearingInterval, session, guard);
}

bool TBillingManager::RemoveBillingTask(const TString& sessionId, NDrive::TEntitySession& session) const {
    auto tasksTable = Database->GetTable("billing_tasks");
    NStorage::TObjectRecordsSet<TBillingTask> removed;
    tasksTable->RemoveRow("session_id='" + sessionId + "'", session.GetTransaction(), &removed);
    if (removed.size() == 1) {
        const TBillingTask& task = removed.front();
        session.SetComment("Payment finished");
        if (!ActiveTasksManager.GetHistoryManager().AddHistory(task, task.GetUserId(), EObjectHistoryAction::Remove, session)) {
            return false;
        }
        return true;
    }
    session.SetErrorInfo("remove_billing", "no_task_founds", EDriveSessionResult::DataCorrupted);
    return false;
}

NDrive::NBilling::IBillingAccount::TPtr TBillingManager::GetLastUsedAccount(const TString& userId, NDrive::TEntitySession& session, EBillingType billingType) const {
    auto compiledBills = FiscalHistoryManager.GetUserFullBillsFromDB(userId, {}, {}, session, billingType, 1);
    if (!compiledBills || compiledBills->size() != 1) {
        return nullptr;
    }

    auto userAccounts = AccountsManager.GetUserAccounts(userId);
    for (auto&& item : compiledBills->begin()->second.GetDetails().GetItems()) {
        for (auto&& account : userAccounts) {
            if (account->IsSelectable() && account->IsActual(ModelingNow()) && account->GetUniqueName() == item.GetUniqueName() && account->GetType() != NDrive::NBilling::EAccount::YAccount) {
                return account;
            }
        }
    }
    return nullptr;
}

TString TBillingManager::GetDefaultAccountName(const TString& userId, NDrive::TEntitySession& session) const {
    auto lastUsedAccont = GetLastUsedAccount(userId, session);
    return lastUsedAccont ? lastUsedAccont->GetUniqueName() : "card";
}

TString TBillingManager::GetDefaultAccountName(const TString& userId) const {
    auto session = BuildSession(true);
    return GetDefaultAccountName(userId, session);
}

TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> TBillingManager::GetUserCards(const TString& userId, bool useCache) const {
    if (useCache && TrustCache) {
        auto cacheValue = TrustCache->GetValue(userId);
        cacheValue.Wait();
        try {
            return cacheValue.GetValue();
        } catch (const std::exception& e) {
            auto evlog = NDrive::GetThreadEventLogger();
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "GetPaymentMethodsFailed")
                    ("user_id", userId)
                    ("error", FormatExc(e))
                );
            }
            return {};
        }
    }

    auto logic = GetLogicByType(NDrive::NBilling::EAccount::Trust);
    if (!logic) {
        return Nothing();
    }
    const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();
    return trustLogic->GetUserCards(userId);
}

TMaybe<NDrive::NTrustClient::TPaymentMethod> TBillingManager::GetYandexAccount(const TString& userId, bool useCache) const {
    auto payMethods = GetUserCards(userId, useCache);
    return GetYandexAccount(payMethods.Get());
}

TMaybe<NDrive::NTrustClient::TPaymentMethod> TBillingManager::GetYandexAccount(const TVector<NDrive::NTrustClient::TPaymentMethod>* payMethods) {
    if (!payMethods) {
        return {};
    }

    for (auto&& method : *payMethods) {
        if (TBillingGlobals::SupportedCashbackMethods.contains(method.GetPaymentMethod()) && TBillingGlobals::SupportedCashbackCurrency.contains(method.GetCurrency())) {
            return method;
        }
    }
    return {};
}

TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> TBillingManager::GetUserPaymentCards(const TVector<NDrive::NTrustClient::TPaymentMethod>* payMethods, bool withExpired) {
    if (!payMethods) {
        return {};
    }

    TSet<NDrive::NTrustClient::TPaymentMethod> realCards;
    for (auto&& method : *payMethods) {
        if (TBillingGlobals::SupportedPayMethods.contains(method.GetPaymentMethod())) {
            if (withExpired || !method.GetIsExpired()) {
                realCards.emplace(method);
            }
        }
    }
    return MakeVector(realCards);
}

TMaybe<TString> TBillingManager::GetDefaultCreditCard(const TString& userId, const TInstant actuality) const {
    return GetDefaultCreditCard(AccountsManager.GetTrustAccount(userId, actuality));
}

TMaybe<TString> TBillingManager::GetDefaultCreditCard(NDrive::NBilling::IBillingAccount::TPtr account) const {
    if (account) {
        auto impl = account->GetAs<NDrive::NBilling::ITrustAccount>();
        if (impl) {
            return impl->GetDefaultCard();
        }
    }
    return {};
}

bool TBillingManager::SetDefaultCreditCard(const TString& cardId, const TString& userId, bool checkCard, NDrive::TEntitySession& session) const {
    auto trustAccount = AccountsManager.GetOrCreateTrustAccount(userId, userId, session);
    if (!trustAccount) {
        session.AddErrorMessage("SetDefaultCreditCard", "no trust account");
        return false;
    }

    NDrive::NBilling::ITrustAccount* impl = trustAccount->GetAs<NDrive::NBilling::ITrustAccount>();
    if (!impl) {
        session.AddErrorMessage("SetDefaultCreditCard", "incorrect trust account");
        return false;
    }
    if (impl->GetDefaultCard() == cardId) {
        return true;
    }

    if (cardId.empty() || !checkCard) {
        NDrive::NTrustClient::TPaymentMethod method;
        method.SetId(cardId);
        return impl->SetDefaultCard(method, userId, session);
    } else {
        auto trustLogic = GetLogicByType(NDrive::NBilling::EAccount::Trust);
        if (!trustLogic) {
            session.AddErrorMessage("SetDefaultCreditCard", "no trust logic");
            return false;
        }
        return trustLogic->GetAsSafe<TTrustLogic>()->SetDefaultCard(trustAccount, cardId, userId, session).Defined();
    }
    return true;
}

TString TBillingManager::GetPassportUid(const TString& userId) const {
    auto userData = UserDB->GetCachedObject(userId);
    if (!userData) {
        ERROR_LOG << "Unknown user: " << userId << Endl;
        return "";
    }
    return userData->GetUid();
}

bool TBillingManager::FillUserReport(const TString& userId, NJson::TJsonValue& report, const TDuration& maxAge) const {
    TInstant actuality = ModelingNow() - maxAge;
    report["bonus"] = AccountsManager.GetBonuses(userId, actuality);
    report["limited_bonus"] = NJson::ToJson(AccountsManager.GetLimitedBonuses(userId, actuality));
    ui32 debt = 0;
    auto session = BuildSession(true);
    auto payments = GetActiveUserPayments(userId, session);
    if (!payments) {
        return false;
    }
    NJson::TJsonValue statuses(NJson::JSON_ARRAY);
    ui32 forcedCount = 0;
    for (auto&& task : *payments) {
        if ((ui32)GetTaskPriority(task, actuality) > (ui32) ETaskPriority::Normal) {
            ++forcedCount;
        }
        debt += task.GetDebt();
        statuses.AppendValue(ToString(task.GetBillingTask().GetTaskStatus()));
    }

    report["debt"] = debt;
    report["tasks_count"] = payments->size();
    report["forced_tasks_count"] = forcedCount;
    report["statuses"] = statuses;
    auto yandexAccount = GetYandexAccount(userId, true);
    if (yandexAccount) {
        report["yandex_account"]["balance"] = yandexAccount->GetBalance();
    }
    return true;
}

TCorrectedSessionPayments TBillingManager::GetSessionReport(const TPaymentsData& payments) const {
    return GetSessionReport(payments.GetBillingTask(), payments.GetSnapshot());
}

TCorrectedSessionPayments TBillingManager::GetSessionReport(const TBillingTask& billingTask, const TCachedPayments& snapshot) const {
    TCorrectedSessionPayments report(billingTask.GetUserId(), billingTask.GetBillCorrected());
    report.Init(snapshot, AccountsManager.GetSettings().GetValue<ui64>("billing.card_pay_min_sum").GetOrElse(0));
    return report;
}

TMaybe<TVector<TPaymentsData>> TBillingManager::GetActiveUserPayments(const TString& userId, NDrive::TEntitySession& session) const {
    return GetActiveUsersPayments(NContainer::Scalar(userId), session);
}

TMaybe<TVector<TPaymentsData>> TBillingManager::GetActiveUsersPayments(TConstArrayRef<TString> userIds, NDrive::TEntitySession& session) const {
    auto fullTasksList = ActiveTasksManager.GetUsersTasks(userIds, session);
    if (!fullTasksList) {
        return {};
    }
    auto payments = PaymentsManager.GetSessionsPayments(*fullTasksList, session, true);
    if (!payments) {
        return {};
    }
    return MakeVector(NContainer::Values(*payments));
}

TMaybe<ui32> TBillingManager::GetDebt(const TString& userId, NDrive::TEntitySession& session, EPaymentStatus* pStatus, const TSet<EBillingType>& allowedBillingTypes, bool* inProc) const {
    return GetDebt(TSet<TString>{ userId }, session, pStatus, allowedBillingTypes, inProc);
}

TMaybe<ui32> TBillingManager::GetDebt(const TSet<TString>& userIds, NDrive::TEntitySession& session, EPaymentStatus* pStatus, const TSet<EBillingType>& allowedBillingTypes, bool* inProc) const {
    TVector<TPaymentsData> payments;
    for (auto&& userId : userIds) {
        auto debts = GetActiveUserPayments(userId, session);
        if (!debts) {
            return {};
        }
        std::move(debts->begin(), debts->end(), std::back_inserter(payments));
    }
    return GetDebt(payments, pStatus, allowedBillingTypes, inProc);
}

TMaybe<TMap<TString, TMaybe<ui32>>> TBillingManager::GetUsersDebts(const TSet<TString>& userIds, NDrive::TEntitySession& session, const TSet<EBillingType>& allowedBillingTypes) const {
    auto payments = GetActiveUsersPayments(MakeVector(userIds), session);
    if (!payments)
        return {};
    TMap<TString, TMaybe<ui32>> usersDebts;
    for (auto&& data : *payments) {
        if (allowedBillingTypes.empty() || allowedBillingTypes.contains(data.GetType())) {
            TMaybe<ui32>& debt = usersDebts[data.GetBillingTask().GetUserId()];
            debt = debt.GetOrElse(0) + data.GetDebt();
        }
    }
    return usersDebts;
}

ui32 TBillingManager::GetDebt(const TVector<TPaymentsData>& payments, EPaymentStatus* pStatus, const TSet<EBillingType>& allowedBillingTypes, bool* inProc) const {
    ui32 debt = 0;
    if (pStatus) {
        *pStatus = EPaymentStatus::NoCards;
    }
    for (auto&& data : payments) {
        if (allowedBillingTypes.empty() || allowedBillingTypes.contains(data.GetType())) {
            debt += data.GetDebt();
            if (pStatus && data.GetBillingTask().GetTaskStatus() == EPaymentStatus::NoFunds) {
                *pStatus = EPaymentStatus::NoFunds;
            }
            if (inProc && data.GetDebt() > 0 && (ui32)GetTaskPriority(data, TInstant::Zero()) > (ui32)TBillingLogic::ETaskPriority::Normal) {
                *inProc = true;
            }
        }
    }
    if (!debt && pStatus) {
        *pStatus = EPaymentStatus::Ok;
    }
    return debt;
}

TOptionalClearingTasks TBillingManager::GetClearingTasks(const TDuration clearingInterval, NDrive::TEntitySession& session) const {
    TStringStream condition;
    condition << "refund > 0 OR created_at_ts <= " << (ModelingNow() - clearingInterval).Seconds();
    return GetTasksImpl<TClearingTask>("clearing_tasks", condition.Str(), session);
}

TBillingManager::~TBillingManager() {
}

TBillingManager::TBillingManager(const TBillingConfig& config, const IHistoryContext& context, TUsersDB& userDB, const ISettings& settings, const NDrive::INotifiersStorage* notifiers, const IRTLineAPIStorage* rtlineApi)
    : TBillingLogic(config)
    , AccountsManager(context.GetDatabase(), settings, config.GetAccountsHistoryConfig())
    , Database(context.GetDatabase())
    , UserDB(&userDB)
    , ActiveTasksManager(context, config.GetUseDBJsonStatements())
    , FiscalHistoryManager(context)
    , FiscalRefundsHistoryManager(context)
    , PaymentsManager(context, ActiveTasksManager, config.GetUsePaymentsCache())
    , PromocodeAccountLinks(context)
{
    ChargeLogics.insert({ NDrive::NBilling::EAccount::Fake, new TFakeLogic(*this, settings) });
    OrderedLogic.push_back(ChargeLogics.at(NDrive::NBilling::EAccount::Fake));

    THolder<NDrive::ITrustUpdater> trustCacheUpdater;
    TAtomicSharedPtr<NDrive::INotifier> notifier = notifiers ? notifiers->GetNotifier(Config.GetNotifierName()) : nullptr;
    for (auto&& logicConfig : config.GetLogicConfigs()) {
        IChargeLogic::TPtr logic = logicConfig->ConstructLogic(settings, *UserDB, notifier);
        if (logic) {
            ChargeLogics.insert( { logic->GetType(),  logic });
            OrderedLogic.push_back(logic);

            if (logic->GetType() == config.GetTrustCacheStorageLogic()) {
                trustCacheUpdater = MakeHolder<TTrustUpdater>(logic);
            }
        } else {
            ERROR_LOG << "Error constructing logic " << Endl;
        }
    }
    if (auto trustConfig = config.GetTrustStorageConfig()) {
        if (rtlineApi) {
            const TRTYTrustStorageConfig* rtyConfig = dynamic_cast<const TRTYTrustStorageConfig*>(trustConfig);
            CHECK_WITH_LOG(rtyConfig);
            const TRTLineAPI* storage = rtlineApi->GetRTLineAPI(rtyConfig->GetAPIName());

            TrustCache = trustConfig->Construct(MakeHolder<TTrustRTYStorageOptions>(storage), std::move(trustCacheUpdater));
        } else {
            TrustCache = trustConfig->Construct({}, std::move(trustCacheUpdater));
        }
    }
}

bool TBillingManager::DoStart() try {
    auto threadCount = FromStringWithDefault<size_t>(GetEnv("BILLING_MANAGER_START_THREADS"), 2);
    auto threadPool = CreateThreadPool(threadCount);
    TVector<NThreading::TFuture<void>> futures;
    futures.push_back(NThreading::Async([this]() {
        TDurationGuard dg("StartComponent AccountsManager");
        Y_ENSURE(AccountsManager.Start());
    }, *threadPool));
    futures.push_back(NThreading::Async([this]() {
        TDurationGuard dg("StartComponent PaymentsManager");
        Y_ENSURE(PaymentsManager.Start());
    }, *threadPool));
    NThreading::WaitAll(futures).GetValueSync();
    return true;
} catch (...) {
    ERROR_LOG << "cannot start BillingManager: " << CurrentExceptionInfo().GetStringRobust() << Endl;
    return false;
}

bool TBillingManager::DoStop() {
    return PaymentsManager.Stop()
        && AccountsManager.Stop();
}

EProduceResult TBillingManager::ProduceTask(const TPaymentsData& paymentsData,
                                  TVector<NDrive::NBilling::IBillingAccount::TPtr> filteredAccounts,
                                  TLazySession& session,
                                  bool syncMode) const
{
    const TBillingTask& bTask = paymentsData.GetBillingTask();
    const TCachedPayments& snapshot = paymentsData.GetSnapshot();

    if (filteredAccounts.empty()) {
        TUnistatSignalsCache::SignalAdd("billing-produce-tasks", "no_accounts", 1);
        ERROR_LOG << "No accounts for task " << bTask.GetId() << Endl;
        return EProduceResult::Ok;
    }

    for (auto&& chargeLogic : OrderedLogic) {
        auto result = chargeLogic->SyncPayments(paymentsData, PaymentsManager, AccountsManager, GetTrustCache(), syncMode);
        if (result != EProduceResult::Ok) {
            TUnistatSignalsCache::SignalAdd("billing-produce-tasks", ::ToString(result), 1);
            if (result != EProduceResult::Wait) {
                ERROR_LOG << "Billing task processing error for " << bTask.GetId() << "/" << result << Endl;
            }
            return result;
        }
    }

    if (CheckCycleCondition(paymentsData)) {
        auto charge = CalcCharge(paymentsData);
        Y_ENSURE_BT(!snapshot.WaitAnyPayments());
        for (auto&& chargeLogic : OrderedLogic) {
            if (syncMode && !chargeLogic->IsSync()) {
                break;
            }
            if (chargeLogic->GetType() == NDrive::NBilling::EAccount::Fake || chargeLogic->GetType() == NDrive::NBilling::EAccount::YCashback) {
                auto result = chargeLogic->ProducePayments(nullptr, paymentsData, PaymentsManager, charge, session, syncMode);
                if (!result) {
                    TUnistatSignalsCache::SignalAdd("billing-produce-tasks", ::ToString(result.GetError()), 1);
                    return result.GetError();
                } else {
                    Y_ENSURE_BT(charge.Sum >= result.GetValue().Sum, charge.Sum << "/" << result.GetValue().Sum);
                    charge.Sum -= result.GetValue().Sum;
                    charge.PaymentTaskId = result.GetValue().PaymentTaskId;
                }
                continue;
            }

            for (auto&& account : filteredAccounts) {
                if (!account->IsActive()) {
                    continue;
                }

                if (account->GetType() == chargeLogic->GetType()) {
                    auto result = chargeLogic->ProducePayments(account, paymentsData, PaymentsManager, charge, session, syncMode);
                    if (!result) {
                        TUnistatSignalsCache::SignalAdd("billing-produce-tasks", ::ToString(result.GetError()), 1);
                        if (result.GetError() != EProduceResult::Wait) {
                            ERROR_LOG << "Billing task processing error for " << bTask.GetId() << "/" << result.GetError() << Endl;
                        }
                        return result.GetError();
                    } else {
                        Y_ENSURE_BT(charge.Sum >= result.GetValue().Sum, charge.Sum << "/" << result.GetValue().Sum);
                        charge.Sum -= result.GetValue().Sum;
                        charge.PaymentTaskId = result.GetValue().PaymentTaskId;
                    }
                }
            }
        }
        TUnistatSignalsCache::SignalAdd("billing-produce-tasks", "ok", 1);
    }
    return EProduceResult::Ok;
}

TBillingManager::TDeepDebts::TDeepDebts(const NDrive::NBilling::TAccountsManager& AccountsManager)
    : CountLimit(AccountsManager.GetSettings().GetValueDef<ui32>("billing.deep_debt.limit", 0))
    , PeriodCoefficient(AccountsManager.GetSettings().GetValueDef<ui32>("billing.deep_debt.days_period", 7))
    , DayPartCoefficient(AccountsManager.GetSettings().GetValueDef<ui32>("billing.deep_debt.day_parts", 24))
{
    NJson::TJsonValue timeRestrictionJson;
    TString timeRestrictionSetting;
    TTimeRestriction newTimeRestriction;
    if (AccountsManager.GetSettings().GetValue("billing.deep_debt.time_restriction", timeRestrictionSetting)) {
        if (NJson::ReadJsonFastTree(timeRestrictionSetting, &timeRestrictionJson)
            && newTimeRestriction.DeserializeFromJson(timeRestrictionJson)
            && newTimeRestriction.Compile()) {
            TimeRestriction = std::move(newTimeRestriction);
        } else {
            TimeRestriction.Clear();
        }
    }
}

bool TBillingManager::TDeepDebts::Check(const TPaymentsData& task, const TInstant actuality) {
    if (!PeriodCoefficient || !DayPartCoefficient) {
        return false;
    }
    const ui64 mainPeriod = (task.GetBillingTask().GetLastUpdate().Seconds() % PeriodCoefficient) * TDuration::Days(1).Seconds();
    const ui64 dayDispersion = (task.GetBillingTask().GetLastUpdate().Seconds() % DayPartCoefficient + 1) * TDuration::Days(1).Seconds() / DayPartCoefficient;
    const auto timeCheck = task.GetBillingTask().GetLastUpdate().Seconds() + mainPeriod + dayDispersion;
    bool result = timeCheck < actuality.Seconds();
    result &= TimeRestriction && TimeRestriction->IsActualNow(actuality);
    return result;
}

void TBillingManager::TDeepDebts::Update(TPaymentsData&& task) {
    const auto lastUpdate = task.GetBillingTask().GetLastUpdate();
    if (Payments.size() < CountLimit) {
        Payments.emplace(lastUpdate, std::move(task));
    } else if (!Payments.empty() && lastUpdate < Payments.begin()->first) {
        Payments.emplace(lastUpdate, std::move(task));
        Payments.erase(Payments.begin());
    }
}

bool TBillingManager::TDeepDebts::TLastUpdateCmp::operator()(const TInstant left, const TInstant& right) const {
    return left > right;
}

TVector<TPaymentsData> TBillingManager::TDeepDebts::GetPayments() {
    TVector<TPaymentsData> result;
    Transform(Payments.begin(), Payments.end(), std::back_inserter(result), [](auto& item) { return std::move(item.second); });
    Payments.clear();
    return result;
}

TVector<TPaymentsData> TBillingManager::GetPriorityTasks(ui32 maxTasks, const TInstant actuality, ui32 minPriority) const {
    TVector<std::pair<TPaymentsData, ui32>> candidates;
    TDeepDebts deepDebts(AccountsManager);
    auto tx = BuildSession(false);
    auto fullTasksList = ActiveTasksManager.GetTasks(tx);
    if (!fullTasksList) {
        return {};
    }
    TVector<TBillingTask> actualTasks;
    for (auto&& task : *fullTasksList) {
        if (task.GetQueue() != Config.GetActiveQueue()) {
            continue;
        }
        auto priority = GetTaskPriority(task, actuality);
        if (!priority || (ui32)*priority >= minPriority) {
            actualTasks.emplace_back(std::move(task));
        }
    }
    candidates.reserve(actualTasks.size());

    TMaybe<TMap<TString, TPaymentsData>> payments;
    if (!PaymentsManager.GetUseCache()) {
        payments = PaymentsManager.GetSessionsPayments(actualTasks, tx, false);
    } else {
        payments = PaymentsManager.GetSessionsPayments(actualTasks, actuality, false);
    }
    if (!payments) {
        return {};
    }

    for (auto&& [sessionId, task] : *payments) {
        if (auto priority = GetTaskPriority(task, actuality); (ui32)priority >= minPriority
            && (priority != ETaskPriority::DeepDebts || deepDebts.Check(task, actuality))) {
            if (priority == ETaskPriority::DeepDebts) {
                deepDebts.Update(std::move(task));
            } else {
                candidates.emplace_back(std::move(task), (ui32)priority);
            }
        }
    }

    auto deepDebtsCandidates = deepDebts.GetPayments();
    const auto deepDebtCount = deepDebtsCandidates.size();
    Transform(deepDebtsCandidates.begin(), deepDebtsCandidates.end(), std::back_inserter(candidates), [](auto& item) { return std::make_pair(std::move(item), (ui32)ETaskPriority::DeepDebts); });

    if (candidates.empty()) {
        return {};
    }

    const auto compare = [](const std::pair<TPaymentsData, ui32>& left, const std::pair<TPaymentsData, ui32>& right) -> bool {
        ui32 lPriority = left.second;
        ui32 rPriority = right.second;
        return rPriority < lPriority || (rPriority == lPriority && left.first.GetBillingTask().GetLastUpdate() < right.first.GetBillingTask().GetLastUpdate());
    };

    TInstant startSort = Now();
    Sort(candidates.begin(), candidates.end(), compare);
    SortByBillingQueue.Signal(Config.GetActiveQueue(), (Now() - startSort).MilliSeconds());
    TUnistatSignalsCache::SignalAdd("billing_cycle", "shadow", maxTasks < candidates.size() ? candidates.size() - maxTasks : 0);
    TUnistatSignalsCache::SignalAdd("billing_cycle", "deep_debts", deepDebtCount);
    TVector<TPaymentsData> result;
    Transform(candidates.begin(), candidates.begin() + Min<ui32>(candidates.size(), maxTasks), std::back_inserter(result), [](const auto& item) { return std::move(item.first); });
    return result;
}

void TBillingManager::WaitBillingCycle(ui32 tasksAtOnce, ui32 maxIterations) const {
    TTimeGuardImpl<false, TLOG_INFO> g("BillingCycle " + ::ToString((ui64)&PaymentsManager));
    TInstant cycleStart = TInstant::Now();
    for (ui32 iteration = 0; iteration < maxIterations; ++iteration) {
        TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("Billing single iteration: " + ::ToString(iteration));
        TInstant startGet = TInstant::Now();
        ui32 minPriority = AccountsManager.GetSettings().GetValueDef<ui32>("billing.min_task_priority", (ui32)ETaskPriority::RegularDebts);
        auto payments = GetPriorityTasks(tasksAtOnce, Now(), minPriority);
        TUnistatSignalsCache::SignalLastX("billing_cycle", "init_time", (TInstant::Now() - startGet).MilliSeconds());
        TUnistatSignalsCache::SignalLastX("billing_cycle_abs", "all", payments.size());
        {
            TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tpg("Billing cycle with " + ::ToString(payments.size()) + " tasks");
            if (AccountsManager.GetSettings().GetValueDef<bool>("billing.log_tasks", false)) {
                TVector<TString> bTaskIds;
                for (const auto& paymentsData : payments) {
                    bTaskIds.push_back(paymentsData.GetBillingTask().GetId());
                }
                NDrive::TEventLog::Log("TasksInCycle", JoinStrings(bTaskIds.begin(), bTaskIds.end(), ", "));
            }
            TMap<std::pair<TBillingTask::EState, EPaymentStatus>, ui32> statusCount;
            ui32 onFinishTask = 0;
            for (auto&& paymentsData : payments) {
                const TBillingTask& bTask = paymentsData.GetBillingTask();
                const TCachedPayments& snapshot = paymentsData.GetSnapshot();
                TBillingTask::EState billingState = TBillingTask::EState::Active;
                if (TryFromString(bTask.GetState(), billingState)) {
                    std::pair<TBillingTask::EState, EPaymentStatus> state(billingState, bTask.GetTaskStatus());
                    auto stateIt = statusCount.find(state);
                    if (stateIt == statusCount.end()) {
                        statusCount[state] = 1;
                    } else {
                        statusCount[state]++;
                    }
                }

                if (snapshot.Empty() && bTask.GetLastPaymentId() != 0) {
                    WARNING_LOG << "Skip inconsistent " << bTask.GetId() << Endl;
                    TUnistatSignalsCache::SignalAdd("billing-produce-tasks", "inconsistent", 1);
                    continue;
                }

                if (SwitchQueue(snapshot, bTask, bTask.GetUserId(), bTask.GetNextQueue())) {
                    TUnistatSignalsCache::SignalAdd("billing-produce-tasks", "on_skip", 1);
                    continue;
                }

                if (OnCycleFinish(paymentsData)) {
                    onFinishTask++;
                    continue;
                }

                auto userAccounts = AccountsManager.GetSortedUserAccounts(bTask.GetUserId(), cycleStart);
                TVector<NDrive::NBilling::IBillingAccount::TPtr> filteredAccounts;
                for (auto&& account : userAccounts) {
                    if (bTask.GetChargableAccounts().contains(account->GetUniqueName())) {
                        filteredAccounts.push_back(account);
                    }
                }
                TLazySession session(Database, new TBillingSessionContext(PaymentsManager));
                ProduceTask(paymentsData, filteredAccounts, session, false);
            }
            for (auto&& chargeLogic : OrderedLogic) {
                chargeLogic->WaitOperations();
            }
            for (auto&&[stateKey, stateName] : GetEnumNames<TBillingTask::EState>()) {
                for (auto&&[paymentKey, paymentName] : GetEnumNames<EPaymentStatus>()) {
                    auto it = statusCount.find({ stateKey, paymentKey });
                    TUnistatSignalsCache::SignalLastX("billing_cycle_abs_" + stateName, paymentName, (it != statusCount.end()) ? it->second : 0);
                }
            }
            TUnistatSignalsCache::SignalLastX("billing-produce-tasks", "on_finish", onFinishTask);
        }
    }
    TUnistatSignalsCache::SignalLastX("billing_cycle", "all_time", (TInstant::Now() - cycleStart).MilliSeconds());
}


ui32 TBillingManager::WaitClearingCycle(ui32 atOnce, ui32 atCycle, const TDuration clearingInterval, TMessagesCollector& errors) const {
    TVector<TClearingTask> tasks;
    {
        auto session = BuildSession(true);
        auto optionalTasks = GetClearingTasks(clearingInterval, session);
        if (!optionalTasks) {
            errors = session.GetMessages();
            return 0;
        }
        tasks = std::move(*optionalTasks);
        Sort(tasks.begin(), tasks.end(), [](const TClearingTask& lhs, const TClearingTask& rhs) {return lhs.GetRefund() > rhs.GetRefund();});
        if (atCycle < tasks.size()) {
            tasks.erase(tasks.begin() + atCycle, tasks.end());
        }
    }
    ui32 successTasks = 0;
    {
        ui32 steps = tasks.size() / atOnce + ((tasks.size() % atOnce == 0) ? 0 : 1);
        for (ui32 step = 0; step < steps; ++step) {
            {
                ui32 tasksCount = 0;
                auto session = BuildSession(false);
                auto result = EProduceResult::Ok;
                {
                    TBillingSyncGuard guard(result);
                    for (ui32 i = step * atOnce; i < Min<ui32>(tasks.size(), (step + 1) * atOnce); ++i) {
                        auto& task = tasks[i];
                        DEBUG_LOG << "Process clearing task " << task.GetPaymentId() << Endl;
                        ProcessTask(task, clearingInterval, session, guard);
                        tasksCount++;
                    }
                }
                Y_UNUSED(session.Commit());
                if (session.GetMessages().HasMessages()) {
                    errors.MergeMessages(session.GetMessages());
                } else {
                    successTasks += tasksCount;
                }
            }
            Sleep(TDuration::MilliSeconds(100));
        }
    }
    return successTasks;
}

ui32 TBillingManager::WaitRefundCycle() const {
    auto logic = GetLogicByType(NDrive::NBilling::EAccount::Trust);
    if (!logic) {
        return 0;
    }
    const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();

    TVector<TRefundTask> tasks;
    {
        auto session = BuildSession(true);
        tasks = PaymentsManager.GetRefundsDB().GetRefundsFromDB({ "draft", "wait_for_notification", "" }, session);
    }

    auto result = EProduceResult::Ok;
    for (auto&& task : tasks) {
        TBillingSyncGuard guard(result);
        INFO_LOG << "Process refund task " << task.GetPaymentId() << Endl;

        auto logger = [this](const TRefundTask& refundTask, NDrive::TEntitySession& session) -> bool {
            if (TrustCache) {
                TrustCache->UpdateValue(refundTask.GetUserId());
            }
            if (refundTask.GetRealRefund()) {
                TCompiledRefund refundField;
                refundField.SetBill(refundTask.GetSum());
                refundField.SetSessionId(refundTask.GetSessionId());
                refundField.SetBillingType(refundTask.GetBillingType() == EBillingType::Deposit ? EBillingType::CarUsage : refundTask.GetBillingType());

                auto trustAccount = AccountsManager.GetTrustAccount(refundTask.GetUserId());
                if (!trustAccount) {
                    ERROR_LOG << "Inconsistent user accounts for " << refundTask.GetUserId() << Endl;
                    return false;
                }

                NDrive::NBilling::TFiscalItem details;
                details.SetUniqueName(trustAccount->GetUniqueName());
                details.SetName(trustAccount->GetName());
                details.SetAccountId(trustAccount->GetId());
                details.SetSum(refundTask.GetSum());
                details.SetType(trustAccount->GetType());
                details.SetTransactionId(NUtil::CreateUUID());
                refundField.SetDetails(NDrive::NBilling::TFiscalDetails().AddItem(std::move(details)));
                return FiscalRefundsHistoryManager.AddHistory(refundField, refundTask.GetUserId(), EObjectHistoryAction::Add, session);
            }
            auto paymentsTable = session->GetDatabase().GetTable("drive_payments");

            NStorage::TTableRecord update;
            update.Set("status", NDrive::NTrustClient::EPaymentStatus::Refunded);
            update.Set("last_update_ts", ModelingNow().Seconds());
            update.Set("id", "nextval('drive_payments_id_seq')");

            NStorage::TTableRecord condition;
            condition.Set("payment_id", refundTask.GetPaymentId());
            TRecordsSet affected;
            auto status = paymentsTable->UpdateRow(condition.BuildCondition(*session.GetTransaction()), update.BuildSet(*session.GetTransaction()) + ", cleared = GREATEST(0, sum - " + ToString(refundTask.GetSum()) + ")", session.GetTransaction(), &affected);
            if (status && status->IsSucceed() && affected.GetRecords().size() == 1) {
                TPaymentTask payment;
                payment.DeserializeFromTableRecord(affected.GetRecords().front(), nullptr);
                return PaymentsManager.UpdatePaymentStatus(payment, EPaymentStatus::Ok, session) != EDriveOpResult::TransactionError;
            }
            return false;
        };
        trustLogic->ProcessRefund(task, guard, logger);
    }
    return tasks.size();
}

TMaybe<TBillingTask> TBillingManager::AddClosedBillingInfo(const TString& sessionId, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = Database->GetTable("billing_tasks");
    auto transaction = session.GetTransaction();
    NStorage::TObjectRecordsSet<TBillingTask> affected;
    NStorage::TTableRecord condition;
    condition.Set("state", "finishing");
    condition.Set("session_id", sessionId);

    NStorage::TTableRecord update;
    update.Set("state", "finished");
    update.Set("event_id", "nextval('billing_tasks_event_id_seq')");

    auto result = table->UpdateRow(condition, update, transaction, &affected);
    if (!result || !result->IsSucceed() || affected.size() != 1) {
        session.SetErrorInfo("finished_billing", "UpdateRow failed " + ::ToString(affected.size()), EDriveSessionResult::TransactionProblem);
        return {};
    }
    for (auto&& historyTask : affected) {
        if (!ActiveTasksManager.GetHistoryManager().AddHistory(historyTask, userId, EObjectHistoryAction::UpdateData, session)) {
            return {};
        }
    }
    return std::move(affected.front());
}

TExpected<TPaymentsData, TString> TBillingManager::AddClosedBillingInfo(const TString& sessionId, const TString& userId, TMaybe<ui32> totalPrice, TMaybe<ui32> cashbackPercent, const TBillingTask::TCashbacksInfo& cashbacksInfo) const {
    if (totalPrice.Defined()) {
        auto session = BuildSession(false);
        if (!SetBillingInfo({{sessionId, *totalPrice}}, session, nullptr, true) || !session.Commit()) {
            ERROR_LOG << "Cannot set billing info " << session.GetStringReport() << Endl;
            return MakeUnexpected<TString>("Cannot set billing info " + session.GetStringReport());
        }
    }

    if (!ProduceTaskSync(sessionId)) {
        return MakeUnexpected<TString>("Cannot produce task");
    }

    auto session = BuildSession(false);
    auto restored = ActiveTasksManager.GetTask(sessionId, session);
    if (!restored || !(*restored)) {
        session.AddErrorMessage("closed_billing_info", "Unknown session");
        return MakeUnexpected<TString>(session.GetStringReport());
    }

    TCachedPayments snapshot;
    if (!PaymentsManager.GetPayments(snapshot, sessionId, session)) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }

    ui32 cashback = 0;
    if (cashbackPercent) {
        TCorrectedSessionPayments report = GetSessionReport(*restored, snapshot);
        cashback += std::round(report.GetCashbackBaseSum() * *cashbackPercent / 10000.) * 100;
    }
    if (cashbacksInfo.GetParts()) {
        for (auto&& info : cashbacksInfo.GetParts()) {
            if (info.HasValue()) {
                cashback += info.GetValueRef();
            }
        }
    }
    if (cashback > 0) {
        if (!SetCashbackInfo(sessionId, cashback, cashbacksInfo, session, nullptr, true)) {
            return MakeUnexpected<TString>(session.GetStringReport());
        }
    }
    if (AccountsManager.GetSettings().GetValueDef("billing.cashback.limit.enabled", false)) {
        cashback = Min<ui32>(cashback, AccountsManager.GetSettings().GetValueDef("billing.cashback.limit.value", 100000));
    }

    auto finishingBillingTask = AddClosedBillingInfo(sessionId, userId, session);
    if (!finishingBillingTask || !session.Commit()) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }
    return TPaymentsData::BuildPaymentsData(*finishingBillingTask, std::move(snapshot), true);
}

TExpected<TPaymentsData, TString> TBillingManager::GetPaymentData(const TString& sessionId) const {
    auto session = BuildSession(false); // master transaction
    auto restored = ActiveTasksManager.GetTask(sessionId, session);
    if (!restored || !(*restored)) {
        return MakeUnexpected<TString>("cannot GetBillingTask: " + session.GetStringReport());
    }

    TCachedPayments snapshot;
    if (!PaymentsManager.GetPayments(snapshot, sessionId, session)) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }
    return TPaymentsData::BuildPaymentsData(std::move(*restored), std::move(snapshot));
}

bool TBillingManager::ProduceTaskSync(const TString& sessionId) const {
    auto startInstant = Now();
    EProduceResult paymentStatus = EProduceResult::Wait;
    while (paymentStatus == EProduceResult::Wait && Now() - startInstant < TDuration::Seconds(30)) {
        auto paymentData = GetPaymentData(sessionId);
        if (!paymentData) {
            return false;
        }
        if (paymentData->GetBillingTask().GetLastPaymentId() != paymentData->GetSnapshot().GetVersion()) {
            continue;
        }

        TVector<NDrive::NBilling::IBillingAccount::TPtr> filteredAccounts;
        auto userAccounts = AccountsManager.GetSortedUserAccounts(paymentData->GetBillingTask().GetUserId());
        for (auto&& account : userAccounts) {
            if (paymentData->GetBillingTask().GetChargableAccounts().contains(account->GetUniqueName())) {
                filteredAccounts.emplace_back(account);
            }
        }

        TLazySession wrapper(Database, new TBillingSessionContext(PaymentsManager));
        paymentData->SetOnFinalization(true);
        paymentStatus = ProduceTask(*paymentData, filteredAccounts, wrapper, true);
        if (paymentStatus == EProduceResult::Wait) {
            Sleep(TDuration::MilliSeconds(10));
        }
    }
    return paymentStatus == EProduceResult::Ok;
}

bool TBillingManager::BuildCompiledCashback(const TPaymentsData& paymentsData, EBillingType cashbackType, NDrive::TEntitySession& session) const {
    const auto& billingTask = paymentsData.GetBillingTask();
    TCompiledBill compiledBill;
    compiledBill.SetSessionId(billingTask.GetId());
    compiledBill.SetUserId(billingTask.GetUserId());

    compiledBill.SetBillingType(cashbackType);
    compiledBill.SetRealSessionId(billingTask.GetRealSessionId());
    compiledBill.SetComment(billingTask.GetComment());
    compiledBill.SetCashback(paymentsData.GetActualCashback());

    if (!FiscalHistoryManager.AddHistory(compiledBill, billingTask.GetUserId(), EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

TMaybe<bool> TBillingManager::BuildCompiledBill(const TBillingTask& billingTask, NDrive::TEntitySession& session, bool final, TMaybe<ui32> borderSum) const {
    auto lastBill = FiscalHistoryManager.GetFullBillFromDB(billingTask.GetId(), session);
    if (!lastBill) {
        return false;
    }
    return BuildCompiledBill(*lastBill, billingTask, session, final, borderSum);
}

TMaybe<bool> TBillingManager::BuildCompiledBill(const TCompiledBill& lastBill, const TBillingTask& billingTask, NDrive::TEntitySession& session, bool final, TMaybe<ui32> borderSum) const {
    auto finalSum = borderSum ? *borderSum : billingTask.GetBillCorrected();
    TCompiledBill compiledBill;
    compiledBill.SetSessionId(billingTask.GetId());
    compiledBill.SetUserId(billingTask.GetUserId());

    compiledBill.SetBillingType(billingTask.GetBillingType());
    compiledBill.SetRealSessionId(billingTask.GetRealSessionId());
    compiledBill.SetComment(billingTask.GetComment());

    TCachedPayments finalSnapshot;
    if (!PaymentsManager.GetPayments(finalSnapshot, billingTask.GetId(), session)) {
        return {};
    }

    TCorrectedSessionPayments report(billingTask.GetUserId(), finalSum);
    report.Init(finalSnapshot, AccountsManager.GetSettings().GetValue<ui64>("billing.card_pay_min_sum").GetOrElse(0));

    auto userAccounts = AccountsManager.GetSortedUserAccounts(billingTask.GetUserId());

    TMap<ui32, NDrive::NBilling::IBillingAccount::TPtr> accountsById;
    NDrive::NBilling::IBillingAccount::TPtr trustAccount;
    for (auto&& account : userAccounts) {
        accountsById[account->GetId()] = account;
        if (account->GetType() == NDrive::NBilling::EAccount::Trust) {
            trustAccount = account;
        }
    }

    if (!trustAccount) {
        session.SetErrorInfo("BuildCompiledBill", "no trust account for " + billingTask.GetUserId());
        return {};
    }

    NDrive::NBilling::TFiscalDetails details;
    auto lastAccountSum = lastBill.GetSumByAccounts();
    ui32 walletHeldSum = 0;
    ui32 currentBill = 0;
    ui32 cardHeldSum = 0;
    for (auto&& accData : report.SumByAccount) {
        NDrive::NBilling::TFiscalItem item;
        item.SetAccountId(accData.first);

        auto it = accountsById.find(accData.first);
        if (it != accountsById.end()) {
            if (it->second->GetType() == NDrive::NBilling::EAccount::Trust) {
                cardHeldSum += accData.second;
                continue;
            }
            item.SetType(it->second->GetType());
            item.SetName(it->second->GetDocName());
            item.SetUniqueName(it->second->GetUniqueName());
            item.SetTransactionId(NUtil::CreateUUID());
        } else {
            item.SetTransactionId(NUtil::CreateUUID());
            auto freeAccount = GetAccountsManager().GetAccountById(accData.first);
            if (freeAccount) {
                item.SetType(freeAccount->GetType());
                item.SetName(freeAccount->GetDocName());
                item.SetUniqueName(freeAccount->GetUniqueName());
            }
        }

        walletHeldSum += accData.second;

        auto itLastSum = lastAccountSum.find(accData.first);
        if (itLastSum != lastAccountSum.end()) {
            if (accData.second < itLastSum->second) {
                session.SetErrorInfo("BuildCompiledBill", "incorrect account sum " + ::ToString(itLastSum->second) + "/" + ::ToString(accData.second)  + " " + billingTask.GetId());
                return {};
            }
            auto newSum = accData.second - itLastSum->second;
            if (newSum == 0) {
                continue;
            }
            item.SetSum(newSum);
        } else {
            item.SetSum(accData.second);
        }
        currentBill += item.GetSum();
        details.AddItem(std::move(item));
    }

    if (finalSum < walletHeldSum) {
        session.SetErrorInfo("BuildCompiledBill", "incorrect wallet held sum " + ::ToString(walletHeldSum) + "/" + ::ToString(finalSum)  + " " + billingTask.GetId());
        return {};
    }

    ui32 cardSum = finalSum - walletHeldSum;
    if (!final) {
        cardSum = Min<ui32>(cardSum, cardHeldSum);
    }
    auto itLastSum = lastAccountSum.find(trustAccount->GetId());
    if (itLastSum != lastAccountSum.end()) {
        if (cardSum < itLastSum->second) {
            session.SetErrorInfo("BuildCompiledBill", "incorrect trust sum " + ::ToString(cardSum) + "/" + ::ToString(itLastSum->second)  + " " + billingTask.GetId());
            return {};
        }
        cardSum -= itLastSum->second;
    }
    currentBill += cardSum;

    if (final && (currentBill + lastBill.GetBill()) != finalSum) {
        session.SetErrorInfo("BuildCompiledBill", "incorrect bills sum " + ::ToString(currentBill + lastBill.GetBill()) + "/" + ::ToString(finalSum)  + " " + billingTask.GetId());
        return {};
    }

    if (final && details.GetItems().empty() || cardSum > 0) {
        NDrive::NBilling::TFiscalItem item;
        item.SetAccountId(trustAccount->GetId());
        item.SetSum(cardSum);
        item.SetType(trustAccount->GetType());
        item.SetName(trustAccount->GetName());
        item.SetUniqueName(trustAccount->GetUniqueName());
        item.SetTransactionId(billingTask.GetId());
        details.AddItem(std::move(item));
    }

    compiledBill.SetBill(currentBill);
    compiledBill.SetDetails(details);
    compiledBill.SetLastPaymentId(finalSnapshot.GetVersion());
    compiledBill.SetFinal(final);
    if (!compiledBill.Check()) {
        session.SetErrorInfo("BuildCompiledBill", "Inconsistent compiled bill constructed for " + billingTask.GetId());
        return {};
    }

    if (!final && details.GetItems().empty()) {
        return false;
    }

    if (!FiscalHistoryManager.AddHistory(compiledBill, billingTask.GetUserId(), EObjectHistoryAction::Add, session)) {
        return {};
    }
    return true;
}

bool TBillingManager::FinishingBillingTask(const TString& sessionId, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = Database->GetTable("billing_tasks");
    auto transaction = session.GetTransaction();
    NStorage::TObjectRecordsSet<TBillingTask> affected;
    {
        NStorage::TTableRecord condition;
        condition.Set("session_id", sessionId);
        condition.Set("state", "active");

        NStorage::TTableRecord update;
        update.Set("state", "finishing");
        update.Set("event_id", "nextval('billing_tasks_event_id_seq')");

        auto result = table->UpdateRow(condition, update, transaction, &affected);
        if (!result || !result->IsSucceed()) {
            session.SetErrorInfo("finishing_billing", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (affected.size() != 1) {
            session.SetErrorInfo("FinishingBillingTask", TStringBuilder() << "affected " << affected.size() << " rows");
            return false;
        }
    }

    for (auto&& historyTask : affected) {
        NStorage::TTableRecord condition;
        condition.Set("session_id", historyTask.GetId());
        condition.Set("state", "finishing");

        historyTask.SetSessionFinishTime(Now());

        NStorage::TTableRecord update;
        update.Set("meta", historyTask.SerializeMetaToJson().GetStringRobust());
        update.Set("event_id", "nextval('billing_tasks_event_id_seq')");

        NStorage::TObjectRecordsSet<TBillingTask> localAffected;
        auto result = table->UpdateRow(condition, update, transaction, &localAffected);
        if (!result || !result->IsSucceed()) {
            session.SetErrorInfo("finishing_billing", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (localAffected.size() != 1) {
            session.SetErrorInfo("FinishingBillingTask", TStringBuilder() << "affected " << affected.size() << " rows");
            return false;
        }
        if (!ActiveTasksManager.GetHistoryManager().AddHistory(localAffected.front(), historyTask.GetUserId(), EObjectHistoryAction::UpdateData, session)) {
            return false;
        }
    }
    return true;
}


bool TBillingManager::CreateBillingTask(const TBillingTask& task, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = Database->GetTable("billing_tasks");
    auto transaction = session.GetTransaction();
    NStorage::TTableRecord record = task.SerializeToTableRecord();

    if (table->AddRow(record, transaction)->GetAffectedRows() != 1) {
        session.SetErrorInfo("create_billing", "AddRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    if (!ActiveTasksManager.GetHistoryManager().AddHistory(task, userId, EObjectHistoryAction::Add, session)) {
        return false;
    }

    return true;
}

bool TBillingManager::UpdateBillingTask(const TBillingTask& task, const TString& userId, NDrive::TEntitySession& session) const {
    auto tasksTable = Database->GetTable("billing_tasks");
    NStorage::TTableRecord tCond;
    tCond.Set("session_id", task.GetId());
    NStorage::TTableRecord update = task.SerializeToTableRecord();
    update.Set("event_id", "nextval('billing_tasks_event_id_seq')");

    NStorage::TObjectRecordsSet<TBillingTask> affected;
    tasksTable->UpdateRow(tCond, update, session.GetTransaction(), &affected);
    if (affected.size() == 1) {
        const TBillingTask& historyTask = affected.front();
        if (!ActiveTasksManager.GetHistoryManager().AddHistory(historyTask, userId, EObjectHistoryAction::UpdateData, session)) {
            return false;
        }
        return true;
    }
    session.AddErrorMessage("update_task", "No task updated");
    return false;
}


bool TBillingManager::CreateBillingTask(const TString& userId, const ICommonOffer::TPtr offer, NDrive::TEntitySession& session, const EBillingType bType) const {
    auto eg = NDrive::BuildEventGuard("CreateBillingTask");
    Y_ENSURE_BT(offer);
    TBillingTask task;
    task.SetId(offer->GetOfferId()).SetRealSessionId(offer->GetOfferId()).SetUserId(userId).SetBill(0).SetDeposit(offer->GetDeposit()).SetBillingType(bType);
    task.SetDiscretization(offer->GetPaymentDiscretization());

    auto userAccounts = AccountsManager.GetUserAccounts(userId);
    TMap<TString, NDrive::NBilling::IBillingAccount::TPtr> accountByName;
    ui32 bonuses = 0;
    TSet<TString> accounts;
    for (auto&& account : userAccounts) {
        if (!account->IsActual(offer->GetTimestamp())) {
            continue;
        }
        accountByName[account->GetUniqueName()] = account;
        if (account->GetType() == NDrive::NBilling::EAccount::Bonus) {
            bonuses += account->GetBalance();
        }
        if (account->GetType() == NDrive::NBilling::EAccount::Trust) {
            accounts.insert(account->GetUniqueName());
        }
    }
    task.SetAvailableBonus(bonuses);
    if (offer->GetYandexAccountBalance()) {
        task.SetAvailableYandexBonus(offer->GetYandexAccountBalance());
    } else {
        auto yandexAccount = GetYandexAccount(userId, false);
        if (!yandexAccount) {
            yandexAccount = GetYandexAccount(userId, true);
        }
        if (yandexAccount && yandexAccount->GetBalance() > 0) {
            task.SetAvailableYandexBonus(yandexAccount->GetBalance());
        }
    }
    task.SetCashbackPercent(offer->GetCashbackPercent());
    if (offer->HasCashbackInfo()) {
        task.AddCashbackInfo({Nothing(), offer->GetCashbackInfoRef()});
    }
    task.SetIsPlusUser(offer->OptionalIsPlusUser());
    task.SetOfferType(offer->GetTypeName());
    task.SetOfferName(offer->GetBehaviourConstructorId());

    if (offer->GetChargableAccounts().empty() || accountByName.empty()) {
        session.SetCode(HTTP_BAD_REQUEST);
        session.SetErrorInfo("BillingManager::CreateBillingTask", "no chargable or active accounts in the offer " + offer->GetOfferId(), EDriveSessionResult::InternalError);
        return false;
    } else {
        for (auto&& accountName : offer->GetChargableAccounts()) {
            auto it = accountByName.find(accountName);
            if (it == accountByName.end()) {
                continue;
            }
            auto accountImpl = it->second;
            if (accountImpl->GetType() == NDrive::NBilling::EAccount::Trust || !accountImpl->IsSelectable()) {
                accounts.insert(accountName);
            }
            if (!offer->GetSelectedCharge() || accountName == offer->GetSelectedCharge()) {
                accounts.insert(accountName);
            }
        }
        task.SetChargableAccounts(accounts);
    }

    const auto& mobilePaymethodId = offer->GetMobilePaymethodId();
    if (mobilePaymethodId) {
        task.SetMobilePaymethods({TBillingTask::TMobilePaymethod().SetMobilePaymethodId(mobilePaymethodId)});
    }
    task.SetExecContext(TBillingTask::TExecContext().SetPaymethod(offer->GetSelectedCreditCard()).SetMobilePaymethod(mobilePaymethodId));
    task.SetLastUpdate(ModelingNow());

    ui32 prePercent = 0;
    AccountsManager.GetSettings().GetValue<ui32>("billing.prestable_rate", prePercent);
    ui64 userHash = FnvHash<ui64>(userId);
    if (userHash % 100 < prePercent) {
        task.SetQueue(Config.GetAlternativeQueue());
        task.SetNextQueue(Config.GetAlternativeQueue());
    } else {
        task.SetQueue(Config.GetActiveQueue());
        task.SetNextQueue(Config.GetActiveQueue());
    }
    return CreateBillingTask(task, userId, session);
}

bool TBillingManager::SetBillingInfo(const TMap<TString, double>& moneyExt, NDrive::TEntitySession& session, TMap<TString, double>* billDelta, bool withHistory) const {
    return SetBillingInfoImpl(moneyExt, "bill", session, billDelta, withHistory);
}

bool TBillingManager::SetDepositInfo(const TMap<TString, double>& moneyExt, NDrive::TEntitySession& session, TMap<TString, double>* billDelta, bool withHistory) const {
    return SetBillingInfoImpl(moneyExt, "deposit", session, billDelta, withHistory);
}

bool TBillingManager::SetCashbackInfo(const TString& sessionId, double cashback, const TBillingTask::TCashbacksInfo& infos, NDrive::TEntitySession& session, TMap<TString, double>* billDelta, bool withHistory) const {
    auto update = [&](TVector<TBillingTask>&& tasks, NDrive::TEntitySession& session) -> TMaybe<TVector<TBillingTask>> {
        TVector<TBillingTask> affected;
        for (auto&& task : tasks) {
            for (const auto& info : infos.GetParts()) {
                task.AddCashbackInfo(info);
            }
            NStorage::TTableRecord condition;
            condition.Set("session_id", task.GetId());

            NStorage::TTableRecord update;
            update.Set("meta", task.SerializeMetaToJson().GetStringRobust());

            auto tasks = ActiveTasksManager.Update(condition, update, session);
            if (!tasks) {
                return Nothing();
            }
            affected.insert(affected.end(), tasks->begin(), tasks->end());
        }
        return affected;
    };
    return SetBillingInfoImpl({{sessionId, cashback}}, "cashback", session, billDelta, withHistory, update);
}

bool TBillingManager::SetBillingInfoImpl(const TMap<TString, double>& moneyExt, const TString& field, NDrive::TEntitySession& session, TMap<TString, double>* billDelta, bool withHistory, std::function<TMaybe<TVector<TBillingTask>>(TVector<TBillingTask>&&, NDrive::TEntitySession& session)> additionalUpdate) const {
    TMap<TString, double> money;
    for (auto&& i : moneyExt) {
        if (i.second > 1e-5) {
            money.emplace(i.first, i.second);
        }
    }
    if (money.empty()) {
        return true;
    }
    TSet<TString> readyUsers;
    double sumDeltaMoney = 0;
    double sumBrokenAccums = 0;

    auto transaction = session.GetTransaction();
    {
        TStringStream ss;
        for (auto&& i : money) {
            if (!ss.Empty()) {
                ss << ",";
            }
            ss << "'" << i.first << "'";
        }
        TRecordsSet billsAtStart;
        NStorage::IQueryResult::TPtr result = transaction->Exec("SELECT " + field + ", session_id FROM billing_tasks WHERE session_id IN (" + ss.Str() + ")", &billsAtStart);
        if (!result || !result->IsSucceed()) {
            session.MergeErrorMessages(transaction->GetErrors(), "billing_tasks_select");
            return false;
        }
        for (auto&& i : billsAtStart.GetRecords()) {
            double bill;
            Y_ENSURE_BT(i.TryGet(field, bill));
            auto it = money.find(i.Get("session_id"));
            Y_ENSURE_BT(it != money.end());
            sumDeltaMoney += Max<double>(0.0, it->second - bill);
            if (billDelta) {
                (*billDelta)[it->first] = Max<double>(0.0, it->second - bill);
            }
        }
        Y_ASSERT(money.size() >= billsAtStart.GetRecords().size());
        sumBrokenAccums = Max<double>(0.0, money.size() - billsAtStart.GetRecords().size());
    }
    TVector<TBillingTask> affected;
    if (Config.GetBatchUpdates()) {
        TStringStream ss;
        for (auto&& i : money) {
            if (!ss.Empty()) {
                ss << ",";
            }
            ss << "('" << i.first << "', " << i.second << ")";
        }

        NStorage::TObjectRecordsSet<TBillingTask> affectedLocal;
        NStorage::IQueryResult::TPtr result = transaction->Exec(
            "UPDATE billing_tasks SET " + field + " = GREATEST(" + field + ", c.add), event_id = nextval('billing_tasks_event_id_seq') FROM (VALUES "
            + ss.Str() +
            ") as c (session_id, add) WHERE c.session_id=billing_tasks.session_id AND state!='finished' RETURNING billing_tasks.*", &affectedLocal);

        if (!result || !result->IsSucceed()) {
            return false;
        }
        affected = affectedLocal.DetachObjects();
    } else {
        auto accessor = transaction->GetDatabase().GetTable("billing_tasks");
        for (auto&& i : money) {
            const TString& sessionId = i.first;
            double add = i.second;
            TString setQuery = Sprintf("%s = GREATEST(%s, %f), event_id = nextval('billing_tasks_event_id_seq')", field.c_str(), field.c_str(), add);
            TString condition = Sprintf("session_id='%s' AND state!='finished'", sessionId.c_str());

            NStorage::TObjectRecordsSet<TBillingTask> affectedLocal;
            auto result = accessor->UpdateRow(condition, setQuery, transaction, &affectedLocal);
            if (!result || !result->IsSucceed()) {
                return false;
            }
            if (affectedLocal.size() != 1) {
                continue;
            }
            affected.emplace_back(std::move(affectedLocal.front()));
        }
    }
    TMaybe<TVector<TBillingTask>> updatedTasks = additionalUpdate(std::move(affected), session);
    if (!updatedTasks) {
        return false;
    }
    if (withHistory) {
        session.SetComment("Payment update");
        for (auto&& task : *updatedTasks) {
            if (!ActiveTasksManager.GetHistoryManager().AddHistory(task, task.GetUserId(), EObjectHistoryAction::UpdateData, session)) {
                return false;
            }
        }
    }

    TUnistatSignalsCache::SignalAdd(field, "delta", sumDeltaMoney);
    TUnistatSignalsCache::SignalAdd(field, "deleted-accumulators", sumBrokenAccums);

    {
        TStringStream ss = TString("add billing: \n");
        for (auto&& i : money) {
            ss << i.first << " : " << i.second << Endl;
        }
        INFO_LOG << ss.Str() << Endl;
    }
    return true;
}

bool TBillingManager::CheckCycleCondition(const TPaymentsData& paymentsData) const {
    const TBillingTask& billingTask = paymentsData.GetBillingTask();
    const TCachedPayments& payments = paymentsData.GetSnapshot();
    if (payments.Empty() && billingTask.GetLastPaymentId() != 0) {
        WARNING_LOG << "Skip inconsistent " << billingTask.GetId() << Endl;
        return false;
    }

    if (billingTask.IsCanceled()) {
        return true;
    }

    if (billingTask.GetBillingType() == EBillingType::CarUsagePrepayment) {
        return true;
    }

    auto charge = CalcCharge(paymentsData);
    if (charge.Sum > 0) {
        return true;
    }

    return billingTask.GetCashback() > payments.GetCashbackSum();
}

TChargeInfo TBillingManager::CalcCharge(const TPaymentsData& paymentsData) const {
    const TBillingTask& billingTask = paymentsData.GetBillingTask();
    ui32 heldSum = paymentsData.GetSnapshot().GetHeldSum();
    TChargeInfo result(billingTask.GetBillingType());
    result.PaymentTaskId = billingTask.GetLastPaymentId();
    if (billingTask.GetState() == "finishing" && !paymentsData.GetOnFinalization()) {
        return result;
    }

    if (!billingTask.IsFinished() && billingTask.GetDeposit() > heldSum && !paymentsData.GetOnFinalization()) {
        result.Sum = billingTask.GetDeposit() - heldSum;
        return result;
    }

    if (heldSum >= billingTask.GetBillCorrected()) {
        return result;
    }

    ui32 diff = billingTask.GetBillCorrected() - heldSum;
    if (billingTask.IsFinished() || paymentsData.GetOnFinalization()) {
        result.Sum = diff;
        return result;
    }
    ui32 minimalPayment = !billingTask.GetDiscretization() ? Config.GetMinimalPayment() : billingTask.GetDiscretization();
    // Should left 1 rub because it is impossible to get less sum in TRUST
    auto payTail = Max<ui32>(100, AccountsManager.GetSettings().GetValueDef<ui32>("billing.card_pay_min_sum", 0));
    if (diff < minimalPayment + payTail) {
        return result;
    }

    ui32 payPeriods = (diff - payTail) / minimalPayment;
    result.Sum = minimalPayment * payPeriods;
    return result;
}

TBillingLogic::ETaskPriority TBillingLogic::GetTaskPriority(const TPaymentsData& data, const TInstant& timestamp) const {
    return GetTaskPriority(data.GetSnapshot(), data.GetBillingTask(), timestamp);
}

bool TBillingLogic::NeedIgnoreByDiscretization(const TCachedPayments& payments, const TBillingTask& billingTask) const {
    return !billingTask.IsFinished()
        && (billingTask.GetBillCorrected() <= payments.GetHeldSum()
            || (billingTask.GetBillCorrected() > payments.GetHeldSum()
            && billingTask.GetBillCorrected() - payments.GetHeldSum() < billingTask.GetDiscretization()));
}

TMaybe<TBillingLogic::ETaskPriority> TBillingLogic::GetTaskPriority(const TBillingTask& billingTask, const TInstant& timestamp) const {
    if (!billingTask.IsFinished() && Max<ui64>(billingTask.GetBillCorrected(), billingTask.GetDeposit()) == 0) {
        return ETaskPriority::Ignore;
    }

    if (billingTask.GetTaskStatus() == EPaymentStatus::DeletedUser) {
        return ETaskPriority::Ignore;
    }

    if (billingTask.GetLastUpdate() == TInstant::Zero()) {
        return ETaskPriority::ForceUpdates;
    }

    if (billingTask.IsCanceled()) {
        return ETaskPriority::ForceUpdates;
    }

    if (billingTask.GetBillingType() == EBillingType::CarUsagePrepayment) {
        return ETaskPriority::ForceUpdates;
    }

    if (billingTask.GetLastUpdate() + Config.GetOperationTimeout() > timestamp) {
        return {};
    }

    if (billingTask.GetTaskStatus() == EPaymentStatus::InternalError &&
        billingTask.GetLastUpdate() + Config.GetErrorTaskCheckInterval() > timestamp) {
        return ETaskPriority::Ignore;
    }

    if (billingTask.IsFinished()) {
        if (billingTask.GetTaskStatus() == EPaymentStatus::RefundDeposit) {
            return ETaskPriority::Normal;
        }

        if (!billingTask.IsDebt()) {
            return ETaskPriority::ForceUpdates;
        }
    }

    if (billingTask.IsDebt() &&
        billingTask.GetLastUpdate() + Config.GetFailedTaskCheckInterval() > timestamp) {
        return ETaskPriority::Ignore;
    }

    if (!billingTask.IsDebt() && billingTask.GetBillingType() == EBillingType::Ticket) {
        return ETaskPriority::ManualTickets;
    }
    return {};
}

TBillingLogic::ETaskPriority TBillingLogic::GetTaskPriority(const TCachedPayments& payments, const TBillingTask& billingTask, const TInstant& timestamp) const {
    if (billingTask.GetLastUpdate() == TInstant::Zero()) {
        return ETaskPriority::ForceUpdates;
    }

    if (billingTask.IsCanceled()) {
        return ETaskPriority::ForceUpdates;
    }

    if (billingTask.GetBillingType() == EBillingType::CarUsagePrepayment) {
        return ETaskPriority::ForceUpdates;
    }

    if (!!payments.GetProcessingPayment()) {
        return ETaskPriority::ForceUpdates;
    }

    if (billingTask.GetTaskStatus() == EPaymentStatus::InternalError &&
        billingTask.GetLastUpdate() + Config.GetErrorTaskCheckInterval() > timestamp) {
        return ETaskPriority::Ignore;
    }

    if (billingTask.IsFinished()) {
        if (billingTask.GetTaskStatus() == EPaymentStatus::RefundDeposit) {
            return ETaskPriority::Normal;
        }

        if (!billingTask.IsDebt()) {
            return ETaskPriority::ForceUpdates;
        }
    }

    if (billingTask.IsDebt() &&
        billingTask.GetLastUpdate() + Config.GetFailedTaskCheckInterval() > timestamp) {
        return ETaskPriority::Ignore;
    }

    if (!billingTask.IsDebt() && billingTask.GetBillingType() == EBillingType::Ticket) {
        return ETaskPriority::ManualTickets;
    }

    if (billingTask.IsDebt() && payments.GetFirstPaymentTs() != TInstant::Max()
        && payments.GetFirstPaymentTs() + Config.GetMaxAutoCheckDebtDeep() < timestamp) {
        return ETaskPriority::DeepDebts;
    }

    if (payments.GetHeldSum() < billingTask.GetDeposit()) {
        return ETaskPriority::Deposit;
    }

    if (NeedIgnoreByDiscretization(payments, billingTask)) {
        return ETaskPriority::Ignore;
    }

    if (billingTask.IsDebt()) {
        return ETaskPriority::RegularDebts;
    }

    return ETaskPriority::Normal;
}

NJson::TJsonValue TBillingManager::GetPaymentMethodsReport(
    bool addBonuses,
    ELocalization locale,
    const TInstant timestamp,
    const TVector<NDrive::NBilling::IBillingAccount::TPtr>& userAccounts,
    const TMaybe<TSet<TString>>& filtredAccounts,
    const TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>>& userCards,
    const TMaybe<NDrive::NTrustClient::TPaymentMethod>& yandexAccount,
    const IServerBase& server,
    const TMaybe<TString>& mobilePay,
    const TString& selectedCreditCard,
    const TSet<TString>& selectedAccounts
) {
    NJson::TJsonValue paymentMethodsReport(NJson::JSON_ARRAY);
    if (addBonuses) {
        auto bonuses = NDrive::NBilling::TAccountsManager::GetBonuses(userAccounts);
        if (bonuses > 0) {
            NDrive::NBilling::IBillingAccount::TReportContext context;
            context.ExternalBalance = bonuses;
            auto report = NDrive::NBilling::GetBonusPaymethodUserReport(locale, server, context);
            paymentMethodsReport.AppendValue(std::move(report));
        }
    }

    for (auto&& account : userAccounts) {
        if (!account->IsActual(timestamp) || !account->IsSelectable()) {
            continue;
        }

        if (filtredAccounts && !filtredAccounts->contains(account->GetUniqueName())) {
            continue;
        }

        NDrive::NBilling::IBillingAccount::TReportContext context;
        if (account->GetType() == NDrive::NBilling::EAccount::Trust) {
            if (account->IsSelectable()) {
                auto trustAccount = account->GetAs<NDrive::NBilling::ITrustAccount>();
                if (!userCards || userCards->empty() || !trustAccount) {
                    NJson::TJsonValue accountReport = account->GetNewUserReport(locale, server, userAccounts, context);
                    if (selectedAccounts.contains(account->GetUniqueName())) {
                        accountReport["selected"] = true;
                    }
                    paymentMethodsReport.AppendValue(accountReport);
                } else {
                    TMaybe<TString> selectedCard;
                    if (selectedAccounts.contains(account->GetUniqueName())) {
                        for (const auto& method : *userCards) {
                            if (method.Check(trustAccount->GetDefaultCard())) {
                                selectedCard = method.GetId();
                                break;
                            }
                        }
                        if (!selectedCard && !userCards->empty()) {
                            selectedCard = userCards->front().GetId();
                        }
                    }

                    for (const auto& method : *userCards) {
                        if (selectedCreditCard && !method.Check(selectedCreditCard)) {
                            continue;
                        }
                        NJson::TJsonValue accountReport = GetPaymentMethodUserReport(method, locale, server);
                        if (selectedCreditCard && selectedAccounts.contains(account->GetUniqueName()) || selectedCard && method.GetId() == *selectedCard) {
                            accountReport["selected"] = true;
                        }

                        account->AddAccountToReport(accountReport);
                        paymentMethodsReport.AppendValue(accountReport);
                    }
                }
            }
            continue;
        } else if (account->GetType() == NDrive::NBilling::EAccount::YAccount) {
            if (!yandexAccount || yandexAccount->GetBalance() == 0) {
                continue;
            }
            context.ExternalBalance = yandexAccount->GetBalance();
        }
        if (account->GetType() == NDrive::NBilling::EAccount::MobilePayment) {
            if (!mobilePay) {
                continue;
            }
            context.System = mobilePay;
        }
        if (account->IsSelectable()) {
            NJson::TJsonValue accountReport = account->GetNewUserReport(locale, server, userAccounts, context);
            if (selectedAccounts.contains(account->GetUniqueName())) {
                accountReport["selected"] = true;
            }
            paymentMethodsReport.AppendValue(accountReport);
        }
    }
    return paymentMethodsReport;
}
