#pragma once

#include "accounts_manager.h"
#include "config.h"

#include <drive/backend/billing/client/client.h>
#include <drive/backend/billing/interfaces/logic.h>
#include <drive/backend/billing/interfaces/payments.h>
#include <drive/backend/billing/interfaces/signals.h>
#include <drive/backend/billing/interfaces/tasks.h>

#include <drive/backend/offers/abstract.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/users/user.h>

#include <library/cpp/messagebus/scheduler/scheduler.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/auto_actualization.h>

class TBillingLogic {
public:
    enum class ETaskPriority: ui32 {
        Deposit = 6,
        ForceUpdates = 5,
        ManualTickets = 4,
        Normal = 3,
        RegularDebts = 2,
        DeepDebts = 1,
        Ignore = 0,
    };

    class TDeepDebts {
    private:
        ui32 CountLimit;
        ui32 PeriodCoefficient;
        ui32 DayPartCoefficient;
        TMaybe<TTimeRestriction> TimeRestriction;

        struct TLastUpdateCmp {
            bool operator()(const TInstant left, const TInstant& right) const;
        };
        std::multimap<TInstant, TPaymentsData, TLastUpdateCmp> Payments;

    public:
        TDeepDebts() = delete;
        TDeepDebts(const NDrive::NBilling::TAccountsManager& AccountsManager);
        bool Check(const TPaymentsData& task, const TInstant actuality);
        void Update(TPaymentsData&& task);
        TVector<TPaymentsData> GetPayments();
    };

public:
    TBillingLogic(const TBillingConfig& config)
        : Config(config)
    {}

    virtual ~TBillingLogic() {}

    bool NeedIgnoreByDiscretization(const TCachedPayments& payments, const TBillingTask& billingTask) const;
    ETaskPriority GetTaskPriority(const TPaymentsData& payments, const TInstant& timestamp = TInstant::Zero()) const;
    TMaybe<ETaskPriority> GetTaskPriority(const TBillingTask& billingTask, const TInstant& timestamp = TInstant::Zero()) const;

private:
    ETaskPriority GetTaskPriority(const TCachedPayments& payments, const TBillingTask& billingTask, const TInstant& timestamp) const;
protected:
    const TBillingConfig& Config;
};

class TBillingManager;

class TFakeLogic : public IChargeLogic {
public:
    TFakeLogic(const TBillingManager& manager, const ISettings& settings);
    virtual TExpectedCharge ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager, const TChargeInfo& chargeInfo, TLazySession& session, const bool /*sync*/) const override;
    virtual bool DoRefund(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentTask& payment, ui32 sum, NDrive::TEntitySession& session, NStorage::TTableRecord& newPaymentRecord) const override;
    virtual TString GetPaymentPrefix() const override {
        return "f_";
    }
private:
    const TBillingManager& Manager;
};

struct TReplyData {
    NJson::TJsonValue Json;
    bool Success = false;
};

class TBillingManager : public TBillingLogic, public IStartStopProcess {
public:
    TBillingManager(const TBillingConfig& config, const IHistoryContext& context, TUsersDB& userDB, const ISettings& settings, const NDrive::INotifiersStorage* notifiers = nullptr, const IRTLineAPIStorage* rtlineApi = nullptr);
    ~TBillingManager();

    TVector<TString> GetKnownAccounts() const;

    void WaitBillingCycle(ui32 tasksAtOnce = 15, ui32 maxIteratins = 100) const;
    ui32 WaitClearingCycle(ui32 atOnce, ui32 atCycle, const TDuration clearingInterval, TMessagesCollector& errors) const;
    ui32 WaitRefundCycle() const;

    bool SetBillingInfo(const TMap<TString, double>& money, NDrive::TEntitySession& session, TMap<TString, double>* billDelta = nullptr, bool withHistory = false) const;
    bool SetDepositInfo(const TMap<TString, double>& money, NDrive::TEntitySession& session, TMap<TString, double>* billDelta = nullptr, bool withHistory = false) const;
    bool SetCashbackInfo(const TString& sessionId, double cashback, const TBillingTask::TCashbacksInfo& infos, NDrive::TEntitySession& session, TMap<TString, double>* billDelta = nullptr, bool withHistory = false) const;
    bool FinishingBillingTask(const TString& sessionId, NDrive::TEntitySession& session) const;
    TMaybe<TBillingTask> AddClosedBillingInfo(const TString& sessionId, const TString& userId, NDrive::TEntitySession& session) const;
    TExpected<TPaymentsData, TString> AddClosedBillingInfo(const TString& sessionId, const TString& userId, TMaybe<ui32> totalPrice = {}, TMaybe<ui32> cashbackPercent = {}, const TBillingTask::TCashbacksInfo& infos = {}) const;

    bool CreateBillingTask(const TBillingTask& task, const TString& userId, NDrive::TEntitySession& session) const;
    bool CreateBillingTask(const TString& userId, const ICommonOffer::TPtr offer, NDrive::TEntitySession& session, const EBillingType bType = EBillingType::CarUsage) const;
    bool UpdateBillingTask(const TBillingTask& task, const TString& userId, NDrive::TEntitySession& session) const;

    void ProcessTask(const TClearingTask& task, const TDuration clearingInterval, NDrive::TEntitySession& session, TBillingSyncGuard& guard) const;
    TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> GetUserCards(const TString& userId, bool useCache = false) const;
    static TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> GetUserPaymentCards(const TVector<NDrive::NTrustClient::TPaymentMethod>* payMethods, bool withExpired = true);
    TMaybe<NDrive::NTrustClient::TPaymentMethod> GetYandexAccount(const TString& userId, bool useCache = false) const;
    static TMaybe<NDrive::NTrustClient::TPaymentMethod> GetYandexAccount(const TVector<NDrive::NTrustClient::TPaymentMethod>* payMethods);
    TMaybe<TString> GetDefaultCreditCard(const TString& userId, const TInstant actuality = TInstant::Zero()) const;
    TMaybe<TString> GetDefaultCreditCard(NDrive::NBilling::IBillingAccount::TPtr account) const;
    bool SetDefaultCreditCard(const TString& cardId, const TString& userId, bool checkCard, NDrive::TEntitySession& session) const;

    // Refunds
    bool ProcessFiscalRefundIssue(const TPaymentTask& paymentTask, const TString& userId, ui32 sum, bool realRefund, NDrive::TEntitySession& session) const;

    // Clearing
    TOptionalClearingTasks GetClearingTasks(const TDuration clearingInterval, NDrive::TEntitySession& session) const;

    TMaybe<ui32> GetDebt(const TString& userId, NDrive::TEntitySession& session, EPaymentStatus* pStatus = nullptr, const TSet<EBillingType>& allowedBillingTypes = {}, bool* inProc = nullptr) const;
    TMaybe<ui32> GetDebt(const TSet<TString>& userIds, NDrive::TEntitySession& session, EPaymentStatus* pStatus = nullptr, const TSet<EBillingType>& allowedBillingTypes = {}, bool* inProc = nullptr) const;
    TMaybe<TMap<TString, TMaybe<ui32>>> GetUsersDebts(const TSet<TString>& userIds, NDrive::TEntitySession& session, const TSet<EBillingType>& allowedBillingTypes = {}) const;
    ui32 GetDebt(const TVector<TPaymentsData>& payments, EPaymentStatus* pStatus = nullptr, const TSet<EBillingType>& allowedBillingTypes = {}, bool* inProc = nullptr) const;

    bool FillUserReport(const TString& userId, NJson::TJsonValue& report, const TDuration& maxAge = TDuration::Max()) const;

    bool ProduceTaskSync(const TString& sessionId) const;
    TExpected<TPaymentsData, TString> GetPaymentData(const TString& sessionId) const;

    TString GetPassportUid(const TString& userId) const;
    NDrive::TEntitySession BuildSession(bool readonly = false) const {
        return PaymentsManager.BuildSession(readonly);
    }

private:
    bool SetBillingInfoImpl(const TMap<TString, double>& money, const TString& field, NDrive::TEntitySession& session, TMap<TString, double>* billDelta = nullptr, bool withHistory = false, std::function<TMaybe<TVector<TBillingTask>>(TVector<TBillingTask>&&, NDrive::TEntitySession& session)> additionalUpdate = [](TVector<TBillingTask>&& tasks, NDrive::TEntitySession&) -> TMaybe<TVector<TBillingTask>> { return tasks; }) const;

    template <class TTask>
    TMaybe<TVector<TTask>> GetTasksImpl(const TString& tableName, const TString& condition, NDrive::TEntitySession& session) const {
        TVector<TTask> result;
        NStorage::ITableAccessor::TPtr table = Database->GetTable(tableName);
        TRecordsSet records;
        auto queryResult = table->GetRows(condition, records, session.GetTransaction());
        if (!queryResult || !queryResult->IsSucceed()) {
            session.SetErrorInfo("BillingManager::GetTasksImpl", "cannot execute query", EDriveSessionResult::TransactionProblem);
            return {};
        }

        for (auto&& i : records) {
            TTask task;
            if (task.DeserializeFromTableRecord(i, nullptr)) {
                result.emplace_back(task);
            } else {
                WARNING_LOG << "Cannot parse task from " << tableName << Endl;
            }
        }
        return result;
    }

    bool CheckCycleCondition(const TPaymentsData& paymentsData) const;
    TChargeInfo CalcCharge(const TPaymentsData& paymentsData) const;

public:
    EProduceResult ProduceTask(const TPaymentsData& paymentsData,
                     TVector<NDrive::NBilling::IBillingAccount::TPtr> filteredAccounts,
                     TLazySession& session,
                     bool syncMode) const;

    virtual bool DoStart() override;
    virtual bool DoStop() override;

    const TPaymentsManager& GetPaymentsManager() const {
        return PaymentsManager;
    }

    const TBillingConfig& GetConfig() const {
        return Config;
    }

    const IChargeLogic::TPtr GetLogicByType(NDrive::NBilling::EAccount accountType) const {
        auto it = ChargeLogics.find(accountType);
        if (it == ChargeLogics.end()) {
            return nullptr;
        }
        return it->second;
    }

    const NDrive::NBilling::TAccountsManager& GetAccountsManager() const {
        return AccountsManager;
    }

    const TBillingHistoryManager& GetHistoryManager() const {
        return ActiveTasksManager.GetHistoryManager();
    }

    const NDrive::NBilling::TActiveTasksManager& GetActiveTasksManager() const {
        return ActiveTasksManager;
    }

    const TFiscalHistoryManager& GetCompiledBills() const {
        return FiscalHistoryManager;
    }

    const TFiscalRefundsHistoryManager& GetCompiledRefunds() const {
        return FiscalRefundsHistoryManager;
    }

    const TPromocodeAccountLinksHistoryManager& GetPromocodeAccountLinks() const {
        return PromocodeAccountLinks;
    }

    TCorrectedSessionPayments GetSessionReport(const TPaymentsData& payments) const;
    TCorrectedSessionPayments GetSessionReport(const TBillingTask& billingTask, const TCachedPayments& snapshot) const;
    TMaybe<TVector<TPaymentsData>> GetActiveUserPayments(const TString& userId, NDrive::TEntitySession& session) const;
    TMaybe<TVector<TPaymentsData>> GetActiveUsersPayments(TConstArrayRef<TString> userIds, NDrive::TEntitySession& session) const;
    NDrive::NBilling::IBillingAccount::TPtr GetLastUsedAccount(const TString& userId, NDrive::TEntitySession& session, EBillingType billingType = EBillingType::CarUsage) const;
    TString GetDefaultAccountName(const TString& userId) const;
    TString GetDefaultAccountName(const TString& userId, NDrive::TEntitySession& session) const;
    TVector<TPaymentsData> GetPriorityTasks(ui32 maxTasks, const TInstant actuality, ui32 minPriority = 0) const;
    TMaybe<bool> BuildCompiledBill(const TBillingTask& billingTask, NDrive::TEntitySession& session, bool final = true, TMaybe<ui32> borderSum = {}) const;
    TMaybe<bool> BuildCompiledBill(const TCompiledBill& lastBill, const TBillingTask& billingTask, NDrive::TEntitySession& session, bool final = true, TMaybe<ui32> borderSum = {}) const;
    bool BuildCompiledCashback(const TPaymentsData& paymentsData, EBillingType cashbackType, NDrive::TEntitySession& session) const;

    NDrive::ITrustStorage* GetTrustCache() const {
        return TrustCache.Get();
    }

    bool SwitchQueue(const TCachedPayments& payments, const TBillingTask& billingTask, const TString& userId, const TMaybe<EBillingQueue> queue, const TMaybe<EBillingQueue> nextQueue = {}) const;

    static NJson::TJsonValue GetPaymentMethodsReport(bool addBonuses, ELocalization locale, const TInstant timestamp, const TVector<NDrive::NBilling::IBillingAccount::TPtr>& userAccounts, const TMaybe<TSet<TString>>& filtredAccounts, const TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>>& userCards, const TMaybe<NDrive::NTrustClient::TPaymentMethod>& yandexAccount, const IServerBase& server, const TMaybe<TString>& mobilePay = {}, const TString& creditCard = "", const TSet<TString>& selectedAccount = {});

private:
    bool RemoveBillingTask(const TString& sessionId, NDrive::TEntitySession& session) const;
    bool OnCycleFinish(const TPaymentsData& paymentsData) const;

private:
    NDrive::NBilling::TAccountsManager AccountsManager;
    TMap<NDrive::NBilling::EAccount, IChargeLogic::TPtr> ChargeLogics;
    TVector<IChargeLogic::TPtr> OrderedLogic;

    TDatabasePtr Database;
    TUsersDB* UserDB;
    mutable TFakeThreadPool FakeHandler;

    NDrive::NBilling::TActiveTasksManager ActiveTasksManager;
    TFiscalHistoryManager FiscalHistoryManager;
    TFiscalRefundsHistoryManager FiscalRefundsHistoryManager;
    TPaymentsManager PaymentsManager;
    TPromocodeAccountLinksHistoryManager PromocodeAccountLinks;

    THolder<NDrive::ITrustStorage> TrustCache;
};
