#include "charge_logic.h"

#include <drive/backend/billing/accounts/trust.h>

ILogicConfig::TFactory::TRegistrator<TMobilePaymentLogicConfig> TMobilePaymentLogicConfig::Registrator(NDrive::NBilling::EAccount::MobilePayment);

IChargeLogic::TPtr TMobilePaymentLogicConfig::ConstructLogic(const ISettings& settings, TUsersDB& users, TAtomicSharedPtr<NDrive::INotifier> notifier) const {
    return new TMobilePaymentLogic(*this, settings, users, notifier);
}

TExpectedCharge TMobilePaymentLogic::ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager,
                                                     const TChargeInfo& chargeInfo, TLazySession& session, const bool /*sync*/) const {

    TChargeInfo charge = CorrectCharge(account, snapshot, chargeInfo);
    if (charge.Sum == 0) {
        return charge;
    }
    TString userId = snapshot.GetBillingTask().GetUserId();
    auto gUser = TBase::UsersDB.FetchInfo(userId, session.Get());
    auto userData = gUser.GetResultPtr(userId);
    if (!userData) {
        return MakeUnexpected(EProduceResult::Error);
    }

    TString passportUid = GetPassportUid(userData->GetUserId(), userData);
    if (!passportUid) {
        return MakeUnexpected(EProduceResult::Error);
    }

    auto billingTask = snapshot.GetBillingTask();

    TPaymentOpContext pContext(*userData, charge, billingTask, passportUid);

    auto product = TBase::TrustClient.GetTrustProduct(charge.Type);
    if (!product.Defined()) {
        // TODO: tell about error somehow
        return MakeUnexpected(EProduceResult::Error);
    }

    TString paymethodId;
    if (billingTask.GetExecContext().GetMobilePaymethod()) {
        paymethodId = billingTask.GetExecContext().GetMobilePaymethod();
    } else if (billingTask.GetMobilePaymethods()) {
        paymethodId = billingTask.GetMobilePaymethods().back().GetMobilePaymethodId();
    } else {
        return TChargeInfo(chargeInfo.Type);
    }

    bool found = false;
    for (const auto& method : billingTask.GetMobilePaymethods()) {
        if (method.GetMobilePaymethodId() == paymethodId) {
            found = true;
            break;
        }
    }

    if (!found) {
        NStorage::TTableRecord condition, update;
        condition.Set("session_id", billingTask.GetId());
        TBillingTask::TMobilePaymethod mobilePaymethod;
        mobilePaymethod.SetMobilePaymethodId(paymethodId);
        billingTask.MutableMobilePaymethods().push_back(mobilePaymethod);
        update.Set("meta", billingTask.SerializeMetaToJson().GetStringRobust());

        if (!paymentsManager.GetActiveTasksManager().UpdateWithHistory(condition, update, billingTask.GetUserId(), session.Get()) || !session.Commit()) {
            ERROR_LOG << session.Get().GetStringReport() << Endl;
            session.Clear();
        }
    }

    NDrive::NTrustClient::TPayment payment;
    payment.Amount = charge.Sum;
    payment.ProductId = product->GetTrustProduct();
    payment.Currency = "RUB";
    payment.FiscalTitle = product->GetFiscalTitle();
    payment.FiscalNDS = product->GetFiscalNDS();
    payment.Email = pContext.GetUserData().GetEmail();
    payment.PaymethodId = paymethodId;
    pContext.SetPayment(payment);

    // no need to receive payment method from trust
    TBillingClient::TOperation operation(ETrustOperatinType::PaymentCreate, new TCreatePaymentCallback<TLogicCallback>(const_cast<TMobilePaymentLogic&>(*this), pContext, account, paymentsManager));
    operation.SetPayment(payment).SetPassportId(passportUid).SetBillingType(snapshot.GetBillingTask().GetBillingType());
    this->AddOperation(std::move(operation));
    return MakeUnexpected(EProduceResult::Wait);
}

TChargeInfo TMobilePaymentLogic::CorrectCharge(const NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const {
    if (snapshot.GetBillingTask().GetExecContext().GetMobilePaymethod()) {
        return chargeInfo;
    }
    if (snapshot.GetBillingTask().GetExecContext().GetPaymethod()) {
        return TChargeInfo(chargeInfo.Type);
    }

    if (!snapshot.GetBillingTask().GetMobilePaymethods()) {
        return TChargeInfo(chargeInfo.Type);
    }
    auto mobilePaymethod = snapshot.GetBillingTask().GetMobilePaymethods().back().GetMobilePaymethodId();

    const auto& fatalErrors = MakeSet(StringSplitter(Settings.GetValueDef<TString>("billing.mobile_payment_fatal_error", "")).Split(',').SkipEmpty().ToList<TString>());
    ui32 count = 0;
    ui32 failedCount = 0;
    for (auto && task : snapshot.GetSnapshot().GetPayments()) {
        if (task.GetPayMethod() != mobilePaymethod) {
            continue;
        }
        if (TBillingGlobals::FailedStatuses.contains(task.GetStatus())) {
            failedCount++;
            if (fatalErrors.contains(task.GetPaymentError())) {
                return TChargeInfo(chargeInfo.Type);
            }
        }
        count++;
    }
    auto errors = snapshot.GetSnapshot().GetPayMethodError(mobilePaymethod);
    if (errors) {
        failedCount = errors->Count;
        count += errors->Count;
    }

    const auto& limit = Settings.GetValue<ui32>("billing.mobile_payment_tasks_limit").GetOrElse(1000);
    const auto& failedLimit = Settings.GetValue<ui32>("billing.mobile_payment_tasks_failed_limit").GetOrElse(1000);
    if (count >= limit || failedCount >= failedLimit) {
        return TChargeInfo(chargeInfo.Type);
    }
    {
        auto filter = [&fatalErrors, &mobilePaymethod](const TCachedPayments::TErrorInfo& errorInfo) {
            return errorInfo.PaymentMethod == mobilePaymethod && fatalErrors.contains(errorInfo.PaymentError);
        };
        if (!snapshot.GetSnapshot().GetFailedPayMethodsQueue(filter).empty()) {
            return TChargeInfo(chargeInfo.Type);
        }
    }
    return TBase::CorrectCharge(account, snapshot, chargeInfo);
}
