#include "redis_cache.h"

#include <rtline/library/storage/redis/abstract.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/instant_model.h>

#include <library/cpp/json/json_reader.h>

#include <util/generic/yexception.h>
#include <util/string/builder.h>

namespace NDriveRedis {
namespace NCache {

TTrustRedisCacheConfig::TFactory::TRegistrator<TTrustRedisCacheConfig> TTrustRedisCacheConfig::Registrator("Redis");

void TTrustRedisCacheConfig::Init(const TYandexConfig::Section* section) {
    DBName = section->GetDirectives().Value("DBName", DBName);
    NDrive::ITrustStorageConfig::Init(section);
}

void TTrustRedisCacheConfig::ToString(IOutputStream &os) const {
    os << "DBName: " << DBName << Endl;
    NDrive::ITrustStorageConfig::ToString(os);
}

TString TTrustRedisCacheConfig::GetExternalDBName() const {
    return GetDBName();
}

THolder<NDrive::ITrustStorage> TTrustRedisCacheConfig::Construct(THolder<NDrive::ITrustStorageOptions>&& ops, THolder<NDrive::ITrustUpdater>&& updater) const {
    return MakeHolder<TTrustRedisCache>(ops->As<TTrustRedisStorageOptions>()->Storage, *this, std::move(updater));
}

TTrustRedisCache::TTrustRedisCache(TAtomicSharedPtr<NKVAbstract::TRedisVersionedStorage> cache, const TTrustRedisCacheConfig& storageConfig, THolder<NDrive::ITrustUpdater>&& updater)
    : NDrive::ITrustStorage(storageConfig, std::move(updater))
    , Cache(cache)
    , CacheId(storageConfig.GetCacheId())
{
}

bool TTrustRedisCache::GetValue(const TString& key, TString& result) const {
    if (!Checked(Cache)) {
        return false;
    }

    bool succeed = Cache->GetValue(key, result);

    if (!succeed) {
        TUnistatSignalsCache::SignalAdd("get-payment-methods-" + CacheId, "failed", 1);
        return false;
    } else if (result.Empty()) {
        TUnistatSignalsCache::SignalAdd("get-payment-methods-" + CacheId, "cache_miss", 1);
        return false;
    }

    return true;
}

bool TTrustRedisCache::SetTTLValue(const TString& value, const TTimedMethods& payments) {
    if (!Checked(Cache)) {
        return false;
    }

    if (ModelingNow() >= payments.Deadline) {
        return true;
    }

    TDuration duration = payments.Deadline - ModelingNow();
    NJson::TJsonValue json = payments.ToJson();

    return Cache->SetTimedValue(value, Base64Encode(json.GetStringRobust()), duration);
}

NThreading::TFuture<NDrive::ITrustStorage::TTimedMethods> TTrustRedisCache::GetTimedPayments(const TString& userId) const {
    TString json_payment_methods;
    auto startTime = Now();
    bool success = GetValue(userId, json_payment_methods);
    auto duration = Now() - startTime;

    if (!success) {
        SignalGetResult(CacheId, /* success = */ false, duration);
        return NThreading::MakeErrorFuture<TTimedMethods>(std::make_exception_ptr(yexception() << "Couldn't get payment methods"));
    }

    NJson::TJsonValue json = NJson::ReadJsonFastTree(Base64Decode(json_payment_methods));

    TTimedMethods payment_methods;
    success = payment_methods.FromJson(json);

    if (!success) {
        SignalGetResult(CacheId, /* success = */ false, duration);
        return NThreading::MakeErrorFuture<TTimedMethods>(std::make_exception_ptr(yexception() << "Could not parse json object"));
    }

    SignalGetResult(CacheId, /* success = */ true, duration);
    return NThreading::MakeFuture<TTimedMethods>(payment_methods);
}

void TTrustRedisCache::DoUpdateValue(const TString& userId, const TTimedMethods& payments) {
    auto startTime = Now();
    bool success = SetTTLValue(userId, payments);
    SignalUpdateResult(CacheId, success, Now() - startTime);
}

}  // namespace NDriveRedis::NCache
}  // namespace NDriveRedis
