#include "redis_proxy_cache.h"

#include <drive/library/cpp/threading/future.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/instant_model.h>

#include <library/cpp/json/json_reader.h>

#include <util/generic/yexception.h>
#include <util/string/builder.h>

namespace NDrive::NRedis::NCache {
    TTrustCacheConfig::TFactory::TRegistrator<TTrustCacheConfig> TTrustCacheConfig::Registrator("redis_service");

    void TTrustCacheConfig::Init(const TYandexConfig::Section* section) {
        NDrive::ITrustStorageConfig::Init(section);
        ClientConfig.Init(section);
    }

    void TTrustCacheConfig::ToString(IOutputStream &os) const {
        NDrive::ITrustStorageConfig::ToString(os);
        ClientConfig.ToString(os);
    }

    THolder<NDrive::ITrustStorage> TTrustCacheConfig::Construct(THolder<NDrive::ITrustStorageOptions>&& /* ops */, THolder<NDrive::ITrustUpdater>&& updater) const {
        return MakeHolder<TTrustCache>(*this, std::move(updater));
    }

    const TConfig& TTrustCacheConfig::GetClientConfig() const {
        return ClientConfig;
    }

    TTrustCache::TTrustCache(const TTrustCacheConfig& storageConfig, THolder<NDrive::ITrustUpdater>&& updater)
        : NDrive::ITrustStorage(storageConfig, std::move(updater))
        , CacheId(storageConfig.GetCacheId())
        , Cache(storageConfig.GetClientConfig())
    {
    }

    NThreading::TFuture<NDrive::ITrustStorage::TTimedMethods> TTrustCache::GetTimedPayments(const TString& userId) const {
        TInstant startTime = Now();
        return Cache.Get(userId).Apply([userId](const auto& result) {
            NJson::TJsonValue json = NJson::ReadJsonFastTree(Base64Decode(result.GetValue()));
            TTimedMethods paymentMethods;
            if (!paymentMethods.FromJson(json)) {
                ythrow yexception() << "Couldn't get payment methods for " << userId;
            }
            return NThreading::MakeFuture<TTimedMethods>(paymentMethods);
        }).Apply([startTime, cacheId = CacheId](const auto& r) {
            SignalGetResult(cacheId, r.HasValue(), Now() - startTime);
            return r.GetValue();
        });
    }

    void TTrustCache::DoUpdateValue(const TString& userId, const TTimedMethods& payments) {
        if (ModelingNow() >= payments.Deadline) { // don't upload expired cache
            return;
        }
        TInstant startTime = Now();
        TDuration ttl = payments.Deadline - ModelingNow();
        Cache.Set(userId, Base64Encode(payments.ToJson().GetStringRobust()), ttl)
            .Subscribe([startTime, cacheId = CacheId](const auto& result)
        {
            SignalUpdateResult(cacheId, result.HasValue(), Now() - startTime);
            if (!result.HasValue()) {
                ERROR_LOG << "TTrustRedisCache::DoUpdateValue: Could not cache payments: " << NThreading::GetExceptionMessage(result) << Endl;
            }
        });
    }

}
