#include "manager.h"

#include <drive/backend/billing/trust/charge_logic.h>

#include <rtline/util/types/uuid.h>

#include <util/generic/adaptor.h>


namespace {
    TNamedSignalSimple BillingSignalsRefundCleared("billing-refunds-cleared-sum");
    TNamedSignalSimple BillingSignalsRefundAuthorized("billing-refunds-authorized-sum");
}

bool TBillingManager::ProcessFiscalRefundIssue(const TPaymentTask& payment, const TString& userId, ui32 refundSum, bool realRefund, NDrive::TEntitySession& session) const {
    const TString& paymentId = payment.GetPaymentId();
    auto transaction = session.GetTransaction();

    if (payment.GetPaymentType() != NDrive::NBilling::EAccount::Trust &&
        payment.GetPaymentType() != NDrive::NBilling::EAccount::YAccount &&
        payment.GetPaymentType() != NDrive::NBilling::EAccount::YCashback &&
        payment.GetPaymentType() != NDrive::NBilling::EAccount::MobilePayment) {
        if (payment.GetCleared() < refundSum) {
            session.AddErrorMessage("fiscal_refund", "Incorrect sum " + ToString(refundSum) + " for " + paymentId);
            return false;
        }

        auto userAccounts = AccountsManager.GetUserAccounts(userId, session);
        NDrive::NBilling::IBillingAccount::TPtr paymentAccount;
        for (auto&& account : *userAccounts) {
            if (account->GetId() == payment.GetAccountId()) {
                paymentAccount = account;
            }
            // For old bonus payments
            if (account->GetUniqueName() == "bonus" && !paymentAccount && paymentId.StartsWith("b_")) {
                paymentAccount = account;
            }
        }
        if (!paymentAccount && payment.GetPaymentType() == NDrive::NBilling::EAccount::Wallet) {
            paymentAccount = AccountsManager.GetAccountById(payment.GetAccountId(), Now());
        }

        if (!paymentAccount) {
            session.AddErrorMessage("fiscal_refund", "Can't find wallet for " + paymentId);
            return false;
        }

        auto logic = GetLogicByType(paymentAccount->GetType());
        if (!logic) {
            session.AddErrorMessage("fiscal_refund", "No logic for " + paymentAccount->GetUniqueName());
            return false;
        }

        if (!logic->Refund(paymentAccount, PaymentsManager, payment, refundSum, session)) {
            return false;
        }

        if (!realRefund) {
            return true;
        }

        TCompiledRefund refundField;
        refundField.SetBill(refundSum);
        refundField.SetSessionId(payment.GetSessionId());
        refundField.SetBillingType(payment.GetBillingType() == EBillingType::Deposit ? EBillingType::CarUsage : payment.GetBillingType());

        NDrive::NBilling::TFiscalItem details;
        details.SetUniqueName(paymentAccount->GetUniqueName());
        details.SetName(paymentAccount->GetDocName());
        details.SetAccountId(paymentAccount->GetId());
        details.SetSum(refundSum);
        details.SetType(paymentAccount->GetType());
        details.SetTransactionId(NUtil::CreateUUID());
        refundField.SetDetails(NDrive::NBilling::TFiscalDetails().AddItem(std::move(details)));
        return FiscalRefundsHistoryManager.AddHistory(refundField, userId, EObjectHistoryAction::Add, session);
    }

    if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared || payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Refunded) {
        auto refundsTable = Database->GetTable("drive_refunds");
        ui64 now = ModelingNow().Seconds();

        NStorage::TTableRecord record;
        record.Set("sum", refundSum);
        record.Set("payment_id", paymentId);
        record.Set("status", "draft");
        record.Set("created_at_ts", now);
        record.Set("last_update_ts", now);
        record.Set("session_id", payment.GetSessionId());
        record.Set("order_id", payment.GetOrderId());
        record.Set("billing_type", payment.GetBillingType());
        record.Set("user_id", userId);
        record.Set("real_refund", realRefund);
        record.Set("payment_type", payment.GetPaymentType());

        if (refundsTable->AddRow(record, transaction)->GetAffectedRows() != 1) {
            session.SetErrorInfo("fiscal_refund", "AddRow failed", EDriveSessionResult::IncorrectRequest);
            return false;
        }

        BillingSignalsRefundCleared.Signal(refundSum);
        return true;
    } else if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Authorized) {
        TClearingTask clearingTask;
        TRecordsSet records;
        auto clearingsTable = Database->GetTable("clearing_tasks");
        clearingsTable->GetRows(NStorage::TRecordBuilder("payment_id", paymentId), records, transaction);

        if (records.GetRecords().size() != 1) {
            session.AddErrorMessage("fiscal_refund", "No clearing task for " + paymentId);
            return false;
        }

        if (!clearingTask.DeserializeFromTableRecord(records.GetRecords().front(), nullptr)) {
            session.AddErrorMessage("fiscal_refund", "Can't parse clearing task for " + paymentId);
            return false;
        }

        if (clearingTask.GetSum() - clearingTask.GetRefund() < refundSum) {
            session.AddErrorMessage("fiscal_refund", "Sum is too large for " + paymentId);
            return false;
        }

        NStorage::TRecordBuilder condition;
        condition("payment_id", paymentId)("refund", clearingTask.GetRefund());
        auto result = clearingsTable->UpdateRow(condition, NStorage::TRecordBuilder("refund", clearingTask.GetRefund() + refundSum), transaction);
        if (result->GetAffectedRows() != 1) {
            session.SetErrorInfo("fiscal_refund", "Can't save modification for " + paymentId, EDriveSessionResult::TransactionProblem);
            return false;
        }

        BillingSignalsRefundAuthorized.Signal(refundSum);
        if (!realRefund) {
            return true;
        }
        TCompiledRefund refundField;
        refundField.SetBill(refundSum);
        refundField.SetSessionId(payment.GetSessionId());
        refundField.SetBillingType(payment.GetBillingType() == EBillingType::Deposit ? EBillingType::CarUsage : payment.GetBillingType());

        auto trustAccount = AccountsManager.GetTrustAccount(userId);
        if (!trustAccount) {
            ERROR_LOG << "Inconsistent user accounts for " << userId << Endl;
            return false;
        }
        NDrive::NBilling::TFiscalItem details;
        details.SetUniqueName(trustAccount->GetUniqueName());
        details.SetName(trustAccount->GetName());
        details.SetAccountId(trustAccount->GetId());
        details.SetSum(refundSum);
        details.SetType(trustAccount->GetType());
        details.SetTransactionId(NUtil::CreateUUID());
        refundField.SetDetails(NDrive::NBilling::TFiscalDetails().AddItem(std::move(details)));
        return FiscalRefundsHistoryManager.AddHistory(refundField, userId, EObjectHistoryAction::Add, session);
    } else {
        session.AddErrorMessage("fiscal_refund", "Incorrect payment status for refund " + ToString(payment.GetStatus()));
        return false;
    }
    return true;
}
