#include "callback.h"

#include "charge_logic.h"

#include <rtline/library/unistat/signals.h>


TMaybe<NDrive::NTrustClient::TPaymentMethod> ParsePaymentMethod(const TCachedPayments& payments, const TSet<TString>& noRetryStatuses, const NJson::TJsonValue& card) {
    NDrive::NTrustClient::TPaymentMethod method;
    if (!method.FromJson(card)) {
        return {};
    }
    if (method.GetIsExpired()) {
        return {};
    }
    auto error = payments.GetPayMethodError(method.GetId());
    if (error && error->Date > method.GetBinding() && noRetryStatuses.contains(error->PaymentError)) {
        return {};
    }
    if (!TBillingGlobals::SupportedPayMethods.contains(method.GetPaymentMethod())) {
        return {};
    }
    return method;
}

void TGetPaymethodsCallback::OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& trustClient) {
    if (isOk) {
        const TBillingTask& billingTask = Context.GetBillingTask();
        TVector<NDrive::NTrustClient::TPaymentMethod> methods;
        TString defaultCard;
        TMap<TString, TString> samePersonPassIds;
        if (json.IsArray()) {
            bool useExtraUsers = false;
            if (auto logic = TrustLogic.GetAsSafe<TTrustLogic>()) {
                useExtraUsers = logic->UseExtraUsers();
            }
            for (auto&& item : json.GetArray()) {
                auto method = ParsePaymentMethod(Payments, TrustLogic.GetNoRetryStatuses(), item);
                TString pass;
                if (!method || method->GetUid().empty() || (!useExtraUsers && method->GetUid() != Context.GetPassportUid())) {
                    continue;
                }
                samePersonPassIds.emplace(method->GetId(), method->GetUid());
                methods.push_back(*method);
                if (method->GetId() == billingTask.GetExecContext().GetPaymethod()) {
                    defaultCard = method->GetId();
                }
            }
        } else {
            for (auto&& payment : json["bound_payment_methods"].GetArray()) {
                auto method = ParsePaymentMethod(Payments, TrustLogic.GetNoRetryStatuses(), payment);
                if (!method) {
                    continue;
                }
                methods.push_back(*method);
                if (method->GetId() == billingTask.GetExecContext().GetPaymethod()) {
                    defaultCard = method->GetId();
                }
            }
        }
        const TSet<TString>& failedMethods = Payments.GetFailedPayMethods();
        if (billingTask.GetUseOnlySelectedPaymethod() && (!defaultCard || failedMethods.contains(defaultCard))) {
            auto session = Owner.BuildSession(false);
            if (Owner.GetActiveTasksManager().CancelTask(billingTask, billingTask.GetUserId(), session, 0)) {
                bool committed = session.Commit();
                Y_UNUSED(committed); // TODO: error handling
            }
            return;
        }

        auto minFailedMethodAge = TDuration::Max();
        if (auto logic = TrustLogic.GetAsSafe<TTrustLogic>()) {
            minFailedMethodAge = logic->GetMimFailedMethodsAge();
        }
        auto filter = [&minFailedMethodAge](const TCachedPayments::TErrorInfo& errorInfo) {
            return errorInfo.Date <= ModelingNow() - minFailedMethodAge;
        };
        TVector<TString> failedMethodsQueue = Payments.GetFailedPayMethodsQueue(filter);

        auto impl = Account->GetAsSafe<NDrive::NBilling::ITrustAccount>();
        TVector<TString> cardsList = NDrive::NBilling::BuildCardList(methods, impl ? impl->GetDefaultCard() : "", failedMethods, failedMethodsQueue);
        if (cardsList.empty()) {
            auto session = Owner.BuildSession(false);
            if (billingTask.GetUseOnlySelectedPaymethod()) {
                if (Owner.GetActiveTasksManager().CancelTask(billingTask, billingTask.GetUserId(), session, 0)) {
                    bool committed = session.Commit();
                    Y_UNUSED(committed); // TODO: error handling
                }
            } else if (Owner.UpdateTaskStatus(billingTask.GetId(), EPaymentStatus::NoCards, session) == EDriveOpResult::Ok) {
                bool committed = session.Commit();
                Y_UNUSED(committed); // TODO: error handling
            }
            return;
        }

        auto charge = Context.GetCharge();
        if (charge.Sum == 0) {
            return;
        }

        auto product = trustClient.GetTrustProduct(charge.Type);
        if (!product.Defined()) {
            Notify("Incorrect terminal " + ToString(charge.Type));
            return;
        }

        const TString targetEmail = Context.GetUserData().GetEmail();
        const TString card = defaultCard ? defaultCard : cardsList.front();
        auto passportId = samePersonPassIds[card];
        if (!passportId) {
            passportId = Context.GetPassportUid();
        } else if (passportId != Context.GetPassportUid()) {
            auto userDataPtr = UsersData.FindPtr(passportId);
            if (!userDataPtr) {
                Notify("Fail to get user data " + passportId);
                return;
            }
            Context.SetUserData(std::move(*userDataPtr));
            Context.SetPassportUid(passportId);
        }
        NDrive::NTrustClient::TPayment payment = BuildPayment(Context, charge, product.GetRef(), card);
        if (Context.GetUserData().GetEmail() != targetEmail) {
            payment.Email = targetEmail;
        }
        Context.SetPayment(payment);
        TBillingClient::TOperation operation(ETrustOperatinType::PaymentCreate, new TCreatePaymentCallback<TLogicCallback>(TrustLogic, Context, Account, Owner));
        operation.SetPayment(payment).SetPassportId(passportId).SetBillingType(charge.Type);
        TrustLogic.AddOperation(std::move(operation));
    } else {
        Notify(json.GetStringRobust());
    }
}

TGetPaymethodsCallback::TGetPaymethodsCallback(ITrustLogic& trustLogic, const TPaymentOpContext& context, NDrive::NBilling::IBillingAccount::TPtr account, const TCachedPayments& payments, const TPaymentsManager& manager)
    : TLogicCallback(trustLogic)
    , Payments(payments)
    , Context(context)
    , Account(account)
    , Owner(manager)
{
}

void TGetPaymethodsCallback::SetUsersData(const TDriveUsers& usersData) {
    Transform(usersData.begin(), usersData.end(), std::inserter(UsersData, UsersData.begin()), [](const auto& userData) -> std::pair<TString, TDriveUserData> {
        return {userData.GetUid(), userData};
    });
}

NDrive::NTrustClient::TPayment TGetPaymethodsCallback::BuildPayment(const TPaymentOpContext& context, const TChargeInfo& charge, const TVirtualTerminal& product, const TString& card) {
    NDrive::NTrustClient::TPayment payment;
    payment.Amount = charge.Sum;
    payment.ProductId = product.GetTrustProduct();
    payment.Currency = "RUB";
    payment.FiscalTitle = product.GetFiscalTitle();
    payment.FiscalNDS = product.GetFiscalNDS();
    payment.Email = context.GetUserData().GetEmail();
    payment.PaymethodId = card;
    auto customTerminal = NDrive::NTrustClient::TPayment::TTerminalRouteData(product.GetTerminal());
    if (customTerminal) {
        payment.TerminalRouteData = customTerminal;
    }
    return payment;
}

TSet<NDrive::NTrustClient::EPaymentStatus> TClearingCallback::ClearedStatuses = {
    NDrive::NTrustClient::EPaymentStatus::Cleared,
    NDrive::NTrustClient::EPaymentStatus::Canceled,
    NDrive::NTrustClient::EPaymentStatus::Refunded
};

void TClearingCallback::OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& trustClient) {
    auto product = trustClient.GetTrustProduct(Task.GetBillingType());
    auto ctx = Session.GetContextAs<TBillingSessionContext>();
    Y_ENSURE_BT(ctx);
    if (!product.Defined()) {
        TGuard<TMutex> g(ctx->GetMutex());
        Session.AddErrorMessage(Task.GetId(), "Incorrect terminal " + ToString(Task.GetBillingType()));
        return;
    }
    TString terminal = product->GetTrustService();

    THolder<TClearingCallback> callback = MakeHolder<TClearingCallback>(TrustLogic, Guard, EClearingOpType::PaymentInfo, Task, Owner, Session, ClearingInterval);

    if (isOk) {
        switch (OpType) {
            case EClearingOpType::PaymentInfo: {
                NDrive::NTrustClient::TPayment result;
                result.FromJson(json);
                if (Task.GetRefund() > 0) {
                    if (Task.GetRefund() - result.Refunded > 1e-4) {
                        if (fabs(Task.GetRefund() - Task.GetSum()) < 1e-4) {
                            callback->SetOperationType(EClearingOpType::Cancel);
                            trustClient.CancelPayment(terminal, Task.GetPaymentId(), callback.Release());
                        } else {
                            callback->SetOperationType(EClearingOpType::Resize);
                            trustClient.ResizePayment(terminal, Task.GetPaymentId(), Task.GetOrderId(), Task.GetSum() - Task.GetRefund(), callback.Release());
                        }
                    } else if (result.Refunded - Task.GetRefund() > 1e-4) {
                        TGuard<TMutex> g(ctx->GetMutex());
                        Session.AddErrorMessage(Task.GetId(), "real refund more then need " + ::ToString(result.Refunded) + "/" + ::ToString(Task.GetRefund()));
                    } else if (ClearedStatuses.contains(result.PaymentStatus)) {
                        Y_UNUSED(RemoveClearingTask(Task.GetPaymentId(), result.Cleared, result.PaymentStatus));
                    } else if (fabs(Task.GetRefund() - Task.GetSum()) < 1e-4) {
                        callback->SetOperationType(EClearingOpType::Cancel);
                        trustClient.CancelPayment(terminal, Task.GetPaymentId(), callback.Release());
                    } else {
                        callback->SetOperationType(EClearingOpType::Clear);
                        trustClient.ClearPayment(terminal, Task.GetPaymentId(), callback.Release());
                    }
                    return;
                }
                if (ClearedStatuses.contains(result.PaymentStatus)) {
                    Y_UNUSED(RemoveClearingTask(Task.GetPaymentId(), result.Cleared, result.PaymentStatus));
                    return;
                }
                if (ModelingNow() - Task.GetCreatedAt() > ClearingInterval) {
                    callback->SetOperationType(EClearingOpType::Clear);
                    trustClient.ClearPayment(terminal, Task.GetPaymentId(), callback.Release());
                }
                break;
            }
            case EClearingOpType::Resize:
                callback->SetOperationType(EClearingOpType::Clear);
                trustClient.ClearPayment(terminal, Task.GetPaymentId(), callback.Release());
                break;
            case EClearingOpType::Cancel:
                Y_UNUSED(RemoveClearingTask(Task.GetPaymentId(), 0, NDrive::NTrustClient::EPaymentStatus::Canceled));
                break;
            case EClearingOpType::Clear:
                Y_UNUSED(RemoveClearingTask(Task.GetPaymentId(), Task.GetSum() - Task.GetRefund(), NDrive::NTrustClient::EPaymentStatus::Cleared));
                break;
            default:
                TGuard<TMutex> g(ctx->GetMutex());
                Session.AddErrorMessage(Task.GetId(), "undefined operation type " + ::ToString(OpType));
                break;
        }
    } else {
        TGuard<TMutex> g(ctx->GetMutex());
        Session.AddErrorMessage(Task.GetId(), json.GetStringRobust());
    }
}

bool TClearingCallback::RemoveClearingTask(const TString& paymentId, ui32 clearedSum, NDrive::NTrustClient::EPaymentStatus status) {
    auto tasksTable = Session->GetDatabase().GetTable("clearing_tasks");

    auto ctx = Session.GetContextAs<TBillingSessionContext>();
    Y_ENSURE_BT(ctx);
    TGuard<TMutex> g(ctx->GetMutex());

    NStorage::TTableRecord tCond;
    tCond.Set("payment_id", paymentId);
    auto result = tasksTable->RemoveRow(tCond, Session.GetTransaction());

    if (status == NDrive::NTrustClient::EPaymentStatus::Unknown) {
        return result->IsSucceed();
    }

    auto paymentsTable = Session->GetDatabase().GetTable("drive_payments");

    NStorage::TTableRecord update;
    update.Set("status", status);
    update.Set("last_update_ts", ModelingNow().Seconds());
    update.Set("id", "nextval('drive_payments_id_seq')");
    update.Set("cleared", clearedSum);

    NStorage::TTableRecord condition;
    condition.Set("payment_id", paymentId);
    TRecordsSet affected;
    paymentsTable->UpdateRow(condition, update, Session.GetTransaction(), &affected);
    if (affected.GetRecords().size() == 1) {
        TPaymentTask payment;
        payment.DeserializeFromTableRecord(affected.GetRecords().front(), nullptr);
        return Owner.UpdatePaymentStatus(payment, TMaybe<EPaymentStatus>(), Session) != EDriveOpResult::TransactionError;
    }
    return false;
}

void TCreateRefundCallback::OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& trustClient) {
    auto table = Database->GetTable("drive_refunds");
    NStorage::TTableRecord condition;
    condition.Set("id", RefundTask.GetId());

    NStorage::TTableRecord update;
    update.Set("last_update_ts", TInstant::Now().Seconds());
    update.Set("id", "nextval('drive_refunds_id_seq')");

    if (isOk) {
        TString trustRefundId = json["trust_refund_id"].GetString();
        auto transaction = Database->CreateTransaction();
        update.Set("refund_id", trustRefundId);
        NStorage::TObjectRecordsSet<TRefundTask> affected;
        auto result = table->UpdateRow(condition, update, transaction, &affected);
        if (!result || !result->IsSucceed() || affected.size() != 1 || !transaction->Commit()) {
            Notify(RefundTask.GetPaymentId() + "/" + transaction->GetErrors().GetStringReport());
        } else {
            auto product = trustClient.GetTrustProduct(affected.front().GetBillingType());
            if (!product.Defined()) {
                Notify(RefundTask.GetPaymentId() + "/incorrect billing type " + ::ToString(affected.front().GetBillingType()));
                return;
            }
            trustClient.StartRefund(product->GetTrustService(), trustRefundId, new TSyncRefundCallback(TrustLogic, Guard, affected.front(), Database, std::move(Logger)));
        }
    } else {
        TString statusCode = json["status_code"].GetString();
        TString status = json["status"].GetString();
        if (status == "error") {
            update.Set("status", statusCode);
            auto transaction = Database->CreateTransaction();
            auto result = table->UpdateRow(condition, update, transaction);
            if (!transaction->Commit()) {
                Notify(RefundTask.GetPaymentId() + "/" + transaction->GetErrors().GetStringReport());
            }
        }
        Notify(RefundTask.GetPaymentId() + "/" + json.GetStringRobust());
    }
}

void TSyncRefundCallback::OnResult(const NJson::TJsonValue& json, bool /*isOk*/, ui16 /*code*/, const TBillingClient& /*trustClient*/) {
    auto table = Database->GetTable("drive_refunds");
    NStorage::TTableRecord condition;
    condition.Set("refund_id", RefundTask.GetRefundId());

    NStorage::TTableRecord update;
    update.Set("last_update_ts", TInstant::Now().Seconds());
    update.Set("id", "nextval('drive_refunds_id_seq')");

    TString status;
    if (!json["status"].GetString(&status)) {
        Notify(RefundTask.GetRefundId() + "/" + json.GetStringRobust());
        return;
    }

    update.Set("status", status);

    NDrive::TEntitySession session(Database->CreateTransaction(false));
    if (status == "success") {
        if (!Logger(RefundTask, session)) {
            Notify(RefundTask.GetRefundId() + "/" + session.GetStringReport());
            return;
        }
    }

    auto result = table->UpdateRow(condition, update, session.GetTransaction());
    if (!result->IsSucceed() || !session.Commit()) {
        Notify(RefundTask.GetRefundId() + "/" + session.GetStringReport());
        return;
    }

    if (status != "wait_for_notification" && status != "success") {
        Notify(RefundTask.GetRefundId() + "/" + json.GetStringRobust());
    }
}

TVector<TString> NDrive::NBilling::BuildCardList(const TVector<NDrive::NTrustClient::TPaymentMethod>& trustMethods, const TString& defaultCard, const TSet<TString>& failedMethods, const TVector<TString>& failedMethodsQueue) {
    TVector<TString> cardsList;
    TSet<TString> availableMethods;
    for (auto&& method : trustMethods) {
        availableMethods.emplace(method.GetId());
    }

    TSet<TString> candidates;
    std::set_difference(availableMethods.begin(), availableMethods.end(),
                        failedMethods.begin(), failedMethods.end(),
                        std::inserter(candidates, candidates.begin()));

    auto check = [&defaultCard, &availableMethods](const TString& item) { return availableMethods.contains(item) && item != defaultCard; };

    if (availableMethods.contains(defaultCard)) {
        if (failedMethods.contains(defaultCard)) {
            cardsList.insert(cardsList.end(), candidates.begin(), candidates.end());
            if (cardsList.empty() && !failedMethodsQueue.empty()) {
                if (std::find(failedMethodsQueue.begin(), failedMethodsQueue.end(), defaultCard) != failedMethodsQueue.end()) {
                    cardsList.push_back(defaultCard);
                    std::copy_if(failedMethodsQueue.begin(), failedMethodsQueue.end(), std::back_inserter(cardsList), check);
                } else {
                    std::copy_if(failedMethodsQueue.begin(), failedMethodsQueue.end(), std::back_inserter(cardsList), check);
                    cardsList.push_back(defaultCard);
                }
            } else {
                cardsList.push_back(defaultCard);
            }
        } else {
            candidates.erase(defaultCard);
            cardsList.push_back(defaultCard);
            cardsList.insert(cardsList.end(), candidates.begin(), candidates.end());
        }
    } else {
        if (candidates.empty()) {
            std::copy_if(failedMethodsQueue.begin(), failedMethodsQueue.end(), std::back_inserter(cardsList), check);
            if (cardsList.empty()) {
                cardsList.insert(cardsList.end(), availableMethods.begin(), availableMethods.end());
            }
        } else {
            cardsList.insert(cardsList.end(), candidates.begin(), candidates.end());
        }
    }
    return cardsList;
}
