#pragma once

#include <drive/backend/billing/client/callback.h>

namespace NDrive::NBilling {
    TVector<TString> BuildCardList(const TVector<NDrive::NTrustClient::TPaymentMethod>& trustMethods, const TString& defaultCard, const TSet<TString>& failedMethods, const TVector<TString>& failedMethodsQueue = {});
}

class TTrustLogic;

class TGetPaymethodsCallback: public TLogicCallback {
public:
    TGetPaymethodsCallback(ITrustLogic& trustLogic,
                           const TPaymentOpContext& context,
                           NDrive::NBilling::IBillingAccount::TPtr account,
                           const TCachedPayments& payments,
                           const TPaymentsManager& manager);
    void SetUsersData(const TDriveUsers& usersData);
    static NDrive::NTrustClient::TPayment BuildPayment(const TPaymentOpContext& context, const TChargeInfo& charge, const TVirtualTerminal& product, const TString& card);
protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& client) override;
    virtual TString GetOperation() const override {
        return "get_paymethods";
    }

private:
    TCachedPayments Payments;
    TPaymentOpContext Context;
    NDrive::NBilling::IBillingAccount::TPtr Account;
    const TPaymentsManager& Owner;
    TMap<TString, TDriveUserData> UsersData;
};

enum EClearingOpType {
    Resize /* "resize" */,
    Cancel /* "cancel" */,
    Clear /* "clear" */,
    PaymentInfo /* "payment_info" */,
};


class TClearingCallback: public TGuardedCallback {
public:
    TClearingCallback(const ITrustLogic& trustLogic, TBillingSyncGuard& guard, EClearingOpType opType, const TClearingTask& task, const TPaymentsManager& manager, NDrive::TEntitySession& session, const TDuration clearingInterval)
        : TGuardedCallback(trustLogic, guard)
        , Task(task)
        , OpType(opType)
        , Owner(manager)
        , Session(session)
        , ClearingInterval(clearingInterval)
    {
    }

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& client) override;
    virtual TString GetOperation() const override {
        return ToString(OpType);
    }
    bool RemoveClearingTask(const TString& paymentId, ui32 clearedSum, NDrive::NTrustClient::EPaymentStatus status);

    void SetOperationType(const EClearingOpType opType) {
        OpType = opType;
    }

private:
    TClearingTask Task;
    EClearingOpType OpType;
    const TPaymentsManager& Owner;
    NDrive::TEntitySession& Session;
    TDuration ClearingInterval;

    static TSet<NDrive::NTrustClient::EPaymentStatus> ClearedStatuses;
};


class TCreateRefundCallback: public TGuardedCallback {
public:
    using TLogger = std::function<bool(const TRefundTask&, NDrive::TEntitySession&)>;

    TCreateRefundCallback(const ITrustLogic& trustLogic, TBillingSyncGuard& guard, const TRefundTask& refundTask, TDatabasePtr db, TLogger&& logger)
        : TGuardedCallback(trustLogic, guard)
        , RefundTask(refundTask)
        , Database(db)
        , Logger(std::move(logger))
    {
    }

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& client) override;
    virtual TString GetOperation() const override {
        return "create_refund";
    }

private:
    TRefundTask RefundTask;
    TDatabasePtr Database;
    TLogger Logger;
};

class TSyncRefundCallback: public TGuardedCallback {
public:
    using TLogger = TCreateRefundCallback::TLogger;

    TSyncRefundCallback(const ITrustLogic& trustLogic, TBillingSyncGuard& guard, const TRefundTask& refundTask, TDatabasePtr db, TLogger&& logger)
        : TGuardedCallback(trustLogic, guard)
        , RefundTask(refundTask)
        , Database(db)
        , Logger(std::move(logger))
    {}

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& client) override;
    virtual TString GetOperation() const override {
        return "sync_refund";
    }

private:
    TRefundTask RefundTask;
    TDatabasePtr Database;
    TLogger Logger;
};

