#include "charge_logic.h"

#include "account_old.h"

#include <drive/backend/billing/accounts/trust.h>

ILogicConfig::TFactory::TRegistrator<TTrustLogicConfig> TTrustLogicConfig::Registrator(NDrive::NBilling::EAccount::Trust);

IChargeLogic::TPtr TTrustLogicConfig::ConstructLogic(const ISettings& settings, TUsersDB& users, TAtomicSharedPtr<NDrive::INotifier> notifier) const {
    return new TTrustLogic(*this, settings, users, notifier);
}

TTrustLogic::TTrustLogic(const ITrustLogicConfig& trustConfig, const ISettings& settings, TUsersDB& usersDB, TAtomicSharedPtr<NDrive::INotifier> notifier)
    : TBase(trustConfig, settings, usersDB, notifier)
{
}

TTrustLogic::~TTrustLogic() {
}

EProduceResult TTrustLogic::SyncPayments(const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager, const NDrive::NBilling::TAccountsManager& /*accountsManager*/, NDrive::ITrustStorage* /*cache*/, const bool sync) const {
    return SyncPaymentsImpl(snapshot, paymentsManager, ETrustOperatinType::PaymentStart, ETrustOperatinType::PaymentInfo, nullptr, sync);
}

TExpectedCharge TTrustLogic::ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager,
                                             const TChargeInfo& chargeInfo, TLazySession& session, const bool /*sync*/) const {

    TChargeInfo charge = CorrectCharge(account, snapshot, chargeInfo);
    if (charge.Sum == 0) {
        return chargeInfo;
    }
    TString userId = snapshot.GetBillingTask().GetUserId();
    if (snapshot.GetBillingTask().GetExecContext().GetUserId()) {
        userId = snapshot.GetBillingTask().GetExecContext().GetUserId();
    }
    auto gUser = TBase::UsersDB.FetchInfo(userId, session.Get());
    auto userData = gUser.GetResultPtr(userId);
    if (!userData) {
        return MakeUnexpected(EProduceResult::Error);
    }

    TString passportUid = GetPassportUid(userData->GetUserId(), userData);
    if (!passportUid) {
        if (paymentsManager.UpdateTaskStatus(snapshot.GetBillingTask().GetId(), EPaymentStatus::DeletedUser, session.Get()) != EDriveOpResult::Ok || !session.Commit()) {
            return MakeUnexpected(EProduceResult::TransactionError);
        }
        return MakeUnexpected(EProduceResult::Error);
    }
    TDriveUsers sortedPersons = { *userData };
    {
        auto useSamePersonsAge = Settings.GetValue<TDuration>("billing.use_duplicates_age");
        if (useSamePersonsAge && snapshot.GetBillingTask().GetSessionFinishTimeDef(TInstant::Zero()) < ModelingNow() - *useSamePersonsAge) {
            auto allPersons = TBase::UsersDB.GetSamePersons(userId, session.Get());
            if (!allPersons) {
                return MakeUnexpected(EProduceResult::TransactionError);
            }
            CopyIf(allPersons->begin(), allPersons->end(), std::back_inserter(sortedPersons), [this](const auto& person) { return !GetPassportUid(person.GetUserId(), &person).empty(); });
            Sort(sortedPersons, TUsersDB::CompareUsersByFreshness);
        }
    }
    TPaymentOpContext pContext(sortedPersons.front(), charge, snapshot.GetBillingTask(), passportUid);
    auto callback = MakeHolder<TGetPaymethodsCallback>(const_cast<TTrustLogic&>(*this), pContext, account, snapshot.GetSnapshot(), paymentsManager);
    callback->SetUsersData(sortedPersons);
    TBillingClient::TOperation operation(ETrustOperatinType::Paymethods, callback.Release());
    operation.SetPassportId(passportUid).SetBillingType(snapshot.GetBillingTask().GetBillingType());
    if (sortedPersons.size() > 1) {
        TVector<TString> sortedSamePersons;
        Transform(sortedPersons.begin(), sortedPersons.end(), std::back_inserter(sortedSamePersons), [this](const auto& person) { return GetPassportUid(person.GetUserId(), &person); });
        operation.SetDuplicatePassIds(sortedSamePersons);
    }
    this->AddOperation(std::move(operation));
    return MakeUnexpected(EProduceResult::Wait);
}

TChargeInfo TTrustLogic::CorrectCharge(const NDrive::NBilling::IBillingAccount::TPtr /*account*/, const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const {
    if (snapshot.GetBillingTask().IsFinished()) {
        if (chargeInfo.Sum < 100) {
            return TChargeInfo(chargeInfo.Type);
        }
    }
    auto charge = chargeInfo;
    charge.Sum = SplitCharge(snapshot, chargeInfo.Sum);
    return charge;
}

void TTrustLogic::ProcessClearing(const TClearingTask& task, const TPaymentsManager& paymentsManager, const TDuration clearingInterval, NDrive::TEntitySession& session, TBillingSyncGuard& guard) const {
    auto ctx = session.GetContextAs<TBillingSessionContext>();
    Y_ENSURE_BT(ctx);
    if (task.GetSum() < task.GetRefund()) {
        TGuard<TMutex> g(ctx->GetMutex());
        session.AddErrorMessage(task.GetId(), "incorrect refund sum " + ::ToString(task.GetSum()) + "/" + ::ToString(task.GetRefund()));
        return;
    }

    auto product = TBase::TrustClient.GetTrustProduct(task.GetBillingType());
    if (!product.Defined()) {
        TGuard<TMutex> g(ctx->GetMutex());
        session.AddErrorMessage(task.GetId(), "incorrect terminal " + ::ToString(task.GetBillingType()));
        return;
    }
    if (task.GetRefund() > 0 || ModelingNow() - task.GetCreatedAt() > clearingInterval) {
        TBase::TrustClient.GetPayment(product->GetTrustService(), task.GetPaymentId(),
                                 new TClearingCallback(*this, guard, EClearingOpType::PaymentInfo, task, paymentsManager, session, clearingInterval));
    }
}

void TTrustLogic::ProcessRefund(const TRefundTask& task, TBillingSyncGuard& guard, TCreateRefundCallback::TLogger&& logger) const {
    auto product = TrustClient.GetTrustProduct(task.GetBillingType());
    if (!product.Defined()) {
        return;
    }
    if (!task.GetRefundId()) {
        NDrive::NTrustClient::TRefund refund;
        refund.OrderId = task.GetOrderId();
        refund.PaymentId = task.GetPaymentId();
        refund.Amount = task.GetSum();
        TrustClient.CreateRefund(product->GetTrustService(), refund, new TCreateRefundCallback(*this, guard, task, TBase::UsersDB.GetDatabase(), std::forward<TCreateRefundCallback::TLogger>(logger)));
    } else if (task.GetStatus() == "draft") {
        TrustClient.StartRefund(product->GetTrustService(),
                              task.GetRefundId(),
                              new TSyncRefundCallback(*this, guard, task, TBase::UsersDB.GetDatabase(), std::forward<TCreateRefundCallback::TLogger>(logger)));
    } else {
        TrustClient.GetRefund(product->GetTrustService(),
                              task.GetRefundId(),
                              new TSyncRefundCallback(*this, guard, task, TBase::UsersDB.GetDatabase(), std::forward<TCreateRefundCallback::TLogger>(logger)));
    }
}

TDuration TTrustLogic::GetMimFailedMethodsAge() const {
    return Settings.GetValueDef<TDuration>("billing.min_failed_method_age", TDuration::Max());
}

bool TTrustLogic::UseExtraUsers() const {
    return Settings.GetValueDef<bool>("billing.use_extra_users", false);
}

ui32 TTrustLogic::SplitCharge(const TPaymentsData& snapshot, const ui32 sum) const {
    auto paymentTs = snapshot.GetBillingTask().HasSessionFinishTime() ? snapshot.GetBillingTask().GetSessionFinishTimeRef() : snapshot.GetSnapshot().GetFirstPaymentTs();
    auto billingType = snapshot.GetBillingTask().GetBillingType();
    if (paymentTs == TInstant::Max() || !snapshot.GetBillingTask().IsFinished()) {
        return sum;
    }
    const TDuration splitAge = Settings.GetValueDef<TDuration>("billing.split.age", TDuration::Zero());
    if (!splitAge || paymentTs + splitAge > ModelingNow()) {
        return sum;
    }
    {
        TSet<TString> splittable;
        StringSplitter(Settings.GetValueDef<TString>("billing.split.last_payment_status", ToString(NDrive::NTrustClient::EPaymentStatus::NotAuthorized))).SplitBySet(",").SkipEmpty().Collect(&splittable);
        if (!splittable.contains(ToString(snapshot.GetSnapshot().GetLastStatus()))) {
            return sum;
        }
    }
    {
        TSet<TString> splittable;
        StringSplitter(Settings.GetValueDef<TString>("billing.split.billing_types", "")).SplitBySet(",").SkipEmpty().Collect(&splittable);
        if (!splittable.contains(ToString(billingType))) {
            return sum;
        }
    }
    const TString partsKey = "billing.split.parts";
    const ui32 defParts = Settings.GetValueDef<ui32>(partsKey, 1);
    const ui32 parts = Settings.GetValueDef<ui32>(partsKey + "." + ToString(billingType), defParts);
    if (parts <= 1) {
        return sum;
    }
    const TString minSumKey = "billing.split.min_sum";
    const ui32 defMinSum = Settings.GetValueDef<ui32>(minSumKey, 0);
    const ui32 minSum = Settings.GetValueDef<ui32>(minSumKey + "." + ToString(billingType), defMinSum);
    return Min(sum, Max(sum / parts, minSum));
}

bool TTrustLogic::GetTrustInfo(EBillingType billingType, const TString& paymentId, NJson::TJsonValue& reply) const {
    auto product = TBase::TrustClient.GetTrustProduct(billingType);
    if (!product.Defined()) {
        return false;
    }
    bool success = false;
    auto guard = TBase::TrustClient.GetPayment(product->GetTrustService(), paymentId, new TReplyGetter(reply, success), true, true);
    guard.Wait();
    return success;
}

TMaybe<NDrive::NTrustClient::TPaymentMethod> TTrustLogic::SetDefaultCard(const NDrive::NBilling::IBillingAccount::TPtr account, const TString& payMethod, const TString& historyUser, NDrive::TEntitySession& session) const {
    auto payMethods = GetUserCards(account->GetUserId());
    if (!payMethods.Defined()) {
        session.SetErrorInfo("SetDefaultCard", "cannot fetch cards from trust");
        return TMaybe<NDrive::NTrustClient::TPaymentMethod>();
    }

    auto trustAccount = account->GetAs<NDrive::NBilling::ITrustAccount>();
    if (!trustAccount) {
        session.SetErrorInfo("SetDefaultCard", "incorrect account type");
        return TMaybe<NDrive::NTrustClient::TPaymentMethod>();
    }

    for (auto&& method : *payMethods) {
        if (!TBillingGlobals::SupportedPayMethods.contains(method.GetPaymentMethod())) {
            continue;
        }

        if (method.Check(payMethod)) {
            if (trustAccount->SetDefaultCard(method, historyUser, session)) {
                return method;
            } else {
                return TMaybe<NDrive::NTrustClient::TPaymentMethod>();
            }
        }
    }
    session.SetErrorInfo("SetDefaultCard", "cannot find selected paymethod");
    return TMaybe<NDrive::NTrustClient::TPaymentMethod>();
}
