#include "trust_cache.h"

#include <drive/backend/logging/events.h>
#include <drive/backend/saas/api.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/api/action.h>
#include <rtline/api/search_client/query.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/protos/proto_helper.h>

TRTYTrustStorageConfig::TFactory::TRegistrator<TRTYTrustStorageConfig> TRTYTrustStorageConfig::Registrator("rtyserver");

THolder<NDrive::ITrustStorage> TRTYTrustStorageConfig::Construct(THolder<NDrive::ITrustStorageOptions>&& ops, THolder<NDrive::ITrustUpdater>&& updater) const {
    TTrustRTYStorageOptions* storageOptions = ops->As<TTrustRTYStorageOptions>();
    return MakeHolder<TRTYTrustStorage>(Checked(storageOptions)->RtlineApi, *this, std::move(updater));
}

NRTLine::TAction CreateSaasRequest(const TString& userId, const NDrive::ITrustStorage::TTimedMethods& timedMethods) {
    NRTLine::TAction result;
    NRTLine::TDocument& d = result.AddDocument();
    d.SetUrl(userId + "/payment_methods");
    d.SetMimeType("text/html");
    d.SetDeadline(timedMethods.Deadline + TDuration::Minutes(30));
    d.AddProperty("data", Base64Encode(timedMethods.ToJson().GetStringRobust()));
    d.AddSpecialKey("user_id", userId);
    return result;
}

void TRTYTrustStorage::DoUpdateValue(const TString& userId, const TTimedMethods& payments) {
    TInstant startTime = Now();
    auto future = Checked(PaymentStorage)->GetIndexingClient()->Send(CreateSaasRequest(userId, payments), NRTLine::TSendParams().SetRealtime());
    future.Subscribe([userId, startTime](const auto& f) {
        if (f.HasValue() && f.GetValue().IsSucceeded()) {
            TUnistatSignalsCache::SignalAdd("payment_methods", "store_success", 1); // deprecated
            SignalUpdateResult("rtyserver", /* success = */ true, Now() - startTime);
        } else {
            TUnistatSignalsCache::SignalAdd("payment_methods", "store_fail", 1); // deprecated
            SignalUpdateResult("rtyserver", /* success = */ false, Now() - startTime);
            NJson::TJsonValue data;
            data["user_id"] = userId;
            if (f.HasValue()) {
                data["code"] = f.GetValue().GetHttpCode();
                data["message"] = f.GetValue().GetMessage();
            } else {
                data["exception"] = NThreading::GetExceptionInfo(f);
            }
            NDrive::TEventLog::Log("StorePaymentMethodsFailure", data);
        }
    });
}

NThreading::TFuture<NDrive::ITrustStorage::TTimedMethods> TRTYTrustStorage::GetTimedPayments(const TString& userId) const {
    NRTLine::TQuery query;
    query.SetText(userId + "/payment_methods");
    query.AddExtraParam("meta_search", "first_found");
    TInstant startTime = Now();
    return Checked(PaymentStorage)->GetSearchClient().SendAsyncQueryF(query).Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        auto reply = r.GetValue();

        if (reply.GetCode() != 200 || reply.GetReport().GroupingSize() == 0) {
            ythrow yexception() << "payments not found";
        }

        if (reply.GetReport().GroupingSize() != 1 || reply.GetReport().GetGrouping(0).GroupSize() != 1 || reply.GetReport().GetGrouping(0).GetGroup(0).DocumentSize() != 1) {
            ythrow yexception() << "payments incorrect report " << reply.GetReport().DebugString();
        }
        TTimedMethods timedMethods;
        TReadSearchProtoHelper helper(reply.GetReport().GetGrouping(0).GetGroup(0).GetDocument(0));
        TString value;
        NJson::TJsonValue dataJson;
        if (!helper.GetProperty("data", value) || !NJson::ReadJsonFastTree(Base64Decode(value), &dataJson) || !timedMethods.FromJson(dataJson)) {
            ythrow yexception() << "payments incorrect report " << reply.GetReport().DebugString();
        }
        return timedMethods;
    }).Apply([startTime](const auto& r) {
        SignalGetResult("rtyserver", r.HasValue(), Now() - startTime);
        return r.GetValue();
    });
}
