#include <drive/backend/billing/ut/library/tests_helper.h>

#include <drive/backend/billing/accounts/limited.h>
#include <drive/backend/billing/bonus/charge_logic.h>
#include <drive/backend/billing/trust/account_old.h>
#include <drive/backend/billing/wallet/charge_logic.h>

Y_UNIT_TEST_SUITE(AccountsSuite) {
    Y_UNIT_TEST(ParallelUpdates) {
        TBillingTestEnvironment env("SQLite");
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();
        auto baDescription = accountsManager.GetDescriptionByName("bonus");
        UNIT_ASSERT(baDescription.Defined());
        {
            auto session = env.BuildSession();
            NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TBonusAccountRecord());
            accountRecord->SetTypeId(env.GetAccountDescriptionId("bonus"));
            accountRecord->SetActive(true);
            ui32 accountId = 0;
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &accountId));
            UNIT_ASSERT(accountsManager.LinkAccount(DefaultUserId, accountId, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session1 = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session1);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr acc = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session1);
            NDrive::NBilling::TBonusAccount* bonusAccount = dynamic_cast<NDrive::NBilling::TBonusAccount*>(acc.Get());
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);

            auto session2 = env.BuildSession();
            UNIT_ASSERT(bonusAccount->Add(100, session1) == EDriveOpResult::Ok);
            UNIT_ASSERT(bonusAccount->Add(50, session2) != EDriveOpResult::Ok);
            UNIT_ASSERT(session1.Commit());
        }
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 100);
            UNIT_ASSERT(session.Commit());
        }
    }

    Y_UNIT_TEST(RefreshPolicy) {
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Month, 1, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2019-03-01T00:00:00Z");
        }
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Month, 5, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2019-07-01T00:00:00Z");
        }
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Month, 12, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2020-02-01T00:00:00Z");
        }
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Day, 3, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2019-02-12T00:00:00Z");
        }
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Day, 21, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2019-03-02T00:00:00Z");
        }
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Week, 1, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2019-02-10T00:00:00Z");
        }
        {
            auto current = TInstant::ParseIso8601("2019-02-09T18:31:42");
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::Week, 2, current);
            UNIT_ASSERT_VALUES_EQUAL(next.ToStringUpToSeconds(), "2019-02-17T00:00:00Z");
        }
        {
            auto next = NDrive::NBilling::TRefreshSchedule::GetNextInstant(NDrive::NBilling::TRefreshSchedule::EPolicy::None, 1, TInstant::Seconds(1546012709));
            UNIT_ASSERT_VALUES_EQUAL(next, TInstant::Max());
        }
    }

    Y_UNIT_TEST(BaseOperations) {
        TBillingTestEnvironment env;
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();

        {
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, Now());
            UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 1);
        }

        auto baDescription = accountsManager.GetDescriptionByName("bonus");
        UNIT_ASSERT(baDescription.Defined());

        TInstant start = TInstant::Seconds(TInstant::Now().Seconds() + 1);
        TInstantGuard ig;

        {
            ui32 accountId = env.LinkBonusAccount(DefaultUserId);
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, Now());
            UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(accounts.front()->GetId(), accountId);
        }

        {
            auto session = env.BuildSession();
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);

            UNIT_ASSERT(bonusAccount->Add(100, session) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 100);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 1);
            UNIT_ASSERT(bonusAccount->Remove(50, session) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        TInstant useBonusesDeadline = start + TDuration::Hours(1);
        TString useBonusesSource = "test";
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 50);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 2);

            UNIT_ASSERT(bonusAccount->Add(100, session, useBonusesDeadline, useBonusesSource) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 150);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 3);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 0);

            ig.Set(useBonusesDeadline + TDuration::Minutes(1));
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Skip);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 150);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 3);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 0);

            UNIT_ASSERT(session.Commit());
        }

        ig.Set(useBonusesDeadline + TDuration::Minutes(20));
        ui32 trustAccountId = 0;
        {
            auto session = env.BuildSession();
            NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TTrustAccountRecord());
            accountRecord->SetTypeId(env.GetAccountDescriptionId("card"));
            accountRecord->SetActive(true);
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &trustAccountId));
            UNIT_ASSERT(accountsManager.LinkAccount(DefaultUserId, trustAccountId, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NJson::TJsonValue json;
        UNIT_ASSERT(NJson::ReadJsonFastTree(ReplyCard3350, &json));
        NDrive::NTrustClient::TPaymentMethod method;
        method.FromJson(json);

        ig.Set(useBonusesDeadline + TDuration::Minutes(30));
        {
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, ModelingNow());
            UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 2);
            NDrive::NBilling::TTrustAccount* trustAccount = nullptr;
            for (auto& account : accounts) {
                if (account->GetName() == "card") {
                    trustAccount = dynamic_cast<NDrive::NBilling::TTrustAccount*>(account.Get());
                }
            }
            UNIT_ASSERT(!!trustAccount);
            UNIT_ASSERT_VALUES_EQUAL(trustAccount->GetDefaultCard(), "");

            auto session = env.BuildSession();
            UNIT_ASSERT(trustAccount->SetDefaultCard(method, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        ig.Set(useBonusesDeadline + TDuration::Minutes(40));
        {
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, ModelingNow());
            UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 2);
            NDrive::NBilling::TTrustAccount* trustAccount = nullptr;
            for (auto& account : accounts) {
                if (account->GetName() == "card") {
                    trustAccount = dynamic_cast<NDrive::NBilling::TTrustAccount*>(account.Get());
                }
            }
            UNIT_ASSERT(!!trustAccount);
            UNIT_ASSERT_VALUES_EQUAL(trustAccount->GetDefaultCard(), method.GetId());
        }
    }

    Y_UNIT_TEST(BaseOperationsLimitedBonuses) {
        TBillingTestEnvironment env;
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();

        auto baDescription = accountsManager.GetDescriptionByName("limited_bonuses");
        UNIT_ASSERT(baDescription.Defined());

        TInstant start = TInstant::Seconds(TInstant::Now().Seconds() + 1);
        TInstantGuard ig;

        TInstant useBonusesDeadline = start + TDuration::Hours(1);
        TString useBonusesSource = "test";
        {
            auto session = env.BuildSession();
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 0);

            UNIT_ASSERT(bonusAccount->Add(100, session, useBonusesDeadline, useBonusesSource) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 100);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().begin()->GetBalance(), 100);

            UNIT_ASSERT(bonusAccount->Remove(50, session) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 50);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 2);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().begin()->GetBalance(), 50);

            UNIT_ASSERT(session.Commit());
        }
        ig.Set(useBonusesDeadline + TDuration::Minutes(1));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = env.BuildSession();
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 50);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 2);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().begin()->GetBalance(), 50);

            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Ok);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts && accounts->size() == 2);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetName(), "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetVersion(), 3);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetLimitedBalance().size(), 0);

            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(!!bonusAccount);

            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Skip);
            UNIT_ASSERT(session.Commit());
        }
    }

    Y_UNIT_TEST(AccountsHierarchy) {
        TBillingTestEnvironment env;
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();

        ui32 accountId = 0;
        ui32 parentId = 0;
        {
            auto session = env.BuildSession();
            NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TLimitedAccountRecord());
            accountRecord->SetTypeId(env.GetAccountDescriptionId("simple_limited"));
            accountRecord->SetActive(true);
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &accountId));
            UNIT_ASSERT(accountsManager.LinkAccount(DefaultUserId, accountId, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TLimitedAccountRecord());
            accountRecord->SetTypeId(env.GetAccountDescriptionId("parent_limited"));
            accountRecord->SetActive(true);
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &parentId));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT_VALUES_UNEQUAL(parentId, 0);
        {
            auto session = env.BuildSession();
            auto userAccounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(userAccounts && userAccounts->size() == 2);
            for (auto&& acc : *userAccounts) {
                if (acc->GetId() == accountId) {
                    UNIT_ASSERT_VALUES_EQUAL(acc->GetParent(), nullptr);
                }
            }
        }
        {
            auto session = env.BuildSession();
            NJson::TJsonValue descJson;
            descJson["type"] = ::ToString(NDrive::NBilling::EAccount::Wallet);
            descJson["hard_limit"] = WalletHardLimit;
            descJson["soft_limit"] = WalletSoftLimit;
            descJson["name"] = "simple_limited";
            descJson["meta"]["hr_name"] = "simple_limited";
            descJson["meta"]["selectable"] = true;
            descJson["meta"]["parent_id"] = parentId;
            NDrive::NBilling::TAccountDescriptionRecord description;
            UNIT_ASSERT(description.FromJson(descJson, nullptr));
            UNIT_ASSERT(accountsManager.UpsertAccountDescription(description, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            auto userAccounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(userAccounts && userAccounts->size() == 2);
            for (auto&& acc : *userAccounts) {
                if (acc->GetId() == accountId) {
                    UNIT_ASSERT_VALUES_UNEQUAL(acc->GetParent(), nullptr);
                    UNIT_ASSERT_VALUES_EQUAL(acc->GetParent()->GetId(), parentId);
                }
            }
        }
    }

    Y_UNIT_TEST(LimitedAccount) {
        TBillingTestEnvironment env;
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();

        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts);
            UNIT_ASSERT_VALUES_EQUAL(accounts->size(), 1);
        }

        ui32 accountId = 0;
        {
            auto session = env.BuildSession();
            NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TLimitedAccountRecord());
            accountRecord->SetTypeId(env.GetAccountDescriptionId("simple_limited"));
            accountRecord->SetActive(true);
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &accountId));
            UNIT_ASSERT(accountsManager.LinkAccount(DefaultUserId, accountId, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts);
            UNIT_ASSERT_VALUES_EQUAL(accounts->size(), 2);
            NDrive::NBilling::IBillingAccount::TPtr wallet;
            for (auto&& account : *accounts) {
                if (account->GetType() == NDrive::NBilling::EAccount::Wallet) {
                    wallet = account;
                }
            }
            UNIT_ASSERT(!!wallet);

            UNIT_ASSERT(wallet->Remove(10000, session) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }

        {
            auto session = env.BuildSession();
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts);
            UNIT_ASSERT_VALUES_EQUAL(accounts->size(), 2);
            NDrive::NBilling::IBillingAccount::TPtr wallet;
            for (auto&& account : *accounts) {
                if (account->GetType() == NDrive::NBilling::EAccount::Wallet) {
                    wallet = account;
                }
            }
            UNIT_ASSERT(!!wallet);
            UNIT_ASSERT_VALUES_EQUAL(wallet->GetBalance(), 40000);
            TWalletLogic chargeLogic(env.SettingsDB);
            TBillingTask billingTask;
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);

            {
                TChargeInfo charge(EBillingType::CarUsage);
                charge.Sum = 60000;
                TChargeInfo corrected = chargeLogic.CorrectCharge(wallet, snapshot, charge);
                UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 60000);
            }
            {
                TChargeInfo charge(EBillingType::CarUsage);
                charge.Sum = 160000;
                TChargeInfo corrected = chargeLogic.CorrectCharge(wallet, snapshot, charge);
                UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 90000);
            }
        }
    }

    Y_UNIT_TEST(ZeroFetch) {
        TDatabasePtr database(TDriveAPIConfigGenerator().CreateDatabase());
        TFakeHistoryContext context(database);
        TSettingsDB settings(context, TSettingsConfig());
        {
            NDrive::NBilling::TAccountsManager accountsManager(database, settings, THistoryConfig());
            UNIT_ASSERT(accountsManager.Start());
            TMap<TString, ui64> accountDescriptionIds;
            TBillingTestEnvironment::PrepareAccounts(database, accountsManager, &accountDescriptionIds);
            NDrive::TEntitySession session(database->CreateTransaction());
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, session);
            UNIT_ASSERT(accounts);
            for (auto&& account : *accounts) {
                UNIT_ASSERT_VALUES_EQUAL(account->GetId(), 0);
            }
            ui32 trustAccountId;
            ui32 accountId;
            {
                NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TBonusAccountRecord());
                accountRecord->SetTypeId(accountDescriptionIds["bonus"]);
                accountRecord->SetActive(true);
                UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &accountId));
                UNIT_ASSERT(accountsManager.LinkAccount(DefaultUserId, accountId, DefaultUserId, session));
            }
            {
                NDrive::NBilling::TAccountRecord::TPtr accountRecord(new NDrive::NBilling::TTrustAccountRecord());
                accountRecord->SetTypeId(accountDescriptionIds["card"]);
                accountRecord->SetActive(true);
                UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, DefaultUserId, session, &trustAccountId));
                UNIT_ASSERT(accountsManager.LinkAccount(DefaultUserId, trustAccountId, DefaultUserId, session));
            }
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(accountsManager.Stop());
        }

        NDrive::NBilling::TAccountsManager accountsManager(database, settings, THistoryConfig());
        UNIT_ASSERT(accountsManager.Start());
        auto accounts = accountsManager.GetUserAccounts(DefaultUserId, TInstant::Zero());
        for (auto&& account : accounts) {
            UNIT_ASSERT(account->GetId() > 0);
        }
        UNIT_ASSERT(accountsManager.Stop());
    }

    Y_UNIT_TEST(BonusAccountLogic) {
        TBillingTestEnvironment env;
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();
        auto baDescription = accountsManager.GetDescriptionByName("bonus");
        UNIT_ASSERT(baDescription.Defined());
        {
            auto session = env.BuildSession();
            auto bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount->GetId() > 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
            UNIT_ASSERT(bonusAccount->Add(10000, session) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        NDrive::NBilling::IBillingAccount::TPtr bonusAccount;
        {
            auto session = env.BuildSession();
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT(session.Commit());
        }
        TBonusLogic chargeLogic(NDrive::NBilling::EAccount::Bonus, env.SettingsDB);
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(5000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 5000);
        }
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10000);
        }
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(10000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 16400;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10000);
        }
        TInstant useBonusesDeadline = TInstant::Seconds(Now().Seconds()) + TDuration::Hours(1);
        TString useBonusesSource = "bonuses";
        {
            auto session = env.BuildSession();
            auto bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount->GetId() > 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 10000);
            UNIT_ASSERT(bonusAccount->Add(100, session, useBonusesDeadline, useBonusesSource) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT(session.Commit());
        }
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10100);
        }
        TInstantGuard ig(useBonusesDeadline + TDuration::Minutes(1));
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10100);
        }
        {
            auto session = env.BuildSession();
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Skip);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT(session.Commit());
        }
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10100);
        }
    }

    Y_UNIT_TEST(LimitedBonusAccountLogic) {
        TBillingTestEnvironment env;
        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();
        auto baDescription = accountsManager.GetDescriptionByName("limited_bonuses");
        UNIT_ASSERT(baDescription.Defined());

        {
            auto session = env.BuildSession();
            auto bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount->GetId() > 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
            UNIT_ASSERT(bonusAccount->Add(10000, session) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        NDrive::NBilling::IBillingAccount::TPtr bonusAccount;
        {
            auto session = env.BuildSession();
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT(session.Commit());
        }
        TBonusLogic chargeLogic(NDrive::NBilling::EAccount::Bonus, env.SettingsDB);
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(10000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 16400;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10000);
        }
        TInstant useBonusesDeadline = TInstant::Seconds(Now().Seconds()) + TDuration::Hours(1);
        TString useBonusesSource = "bonuses";
        {
            auto session = env.BuildSession();
            auto bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount->GetId() > 0);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 10000);
            UNIT_ASSERT(bonusAccount->Add(100, session, useBonusesDeadline, useBonusesSource) == EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession(false);
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT(session.Commit());
        }
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10100);
        }
        TInstantGuard ig(useBonusesDeadline + TDuration::Minutes(1));
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10100);
        }
        {
            auto session = env.BuildSession();
            auto bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Ok);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = env.BuildSession();
            bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(session.Commit());
        }
        {
            TBillingTask billingTask;
            billingTask.SetAvailableBonus(20000);
            auto payments = TPaymentsData::BuildPaymentsData(billingTask, TCachedPayments());
            UNIT_ASSERT(payments);
            auto snapshot = std::move(*payments);
            TChargeInfo charge(EBillingType::CarUsage);
            charge.Sum = 20000;
            TChargeInfo corrected = chargeLogic.CorrectCharge(bonusAccount, snapshot, charge);
            UNIT_ASSERT_VALUES_EQUAL(corrected.Sum, 10000);
        }
    }

    Y_UNIT_TEST(GetOrCreateAccount) {
        TBillingTestEnvironment env;
        const NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();

        {
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, Now());
            UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 1);
            for (auto&& account : accounts) {
                if (account->GetType() == NDrive::NBilling::EAccount::Bonus) {
                    UNIT_ASSERT(!account->GetAs<NDrive::NBilling::TBonusAccount>());
                }
                if (account->GetType() == NDrive::NBilling::EAccount::Trust) {
                    UNIT_ASSERT(!account->GetAs<NDrive::NBilling::TTrustAccount>());
                }
            }
        }
        {
            auto session = env.BuildSession();
            auto baDescription = accountsManager.GetDescriptionByName("bonus");
            UNIT_ASSERT(baDescription.Defined());
            auto bonusAccount = accountsManager.GetOrCreateAccount(DefaultUserId, baDescription.GetRef(), DefaultUserId, session);
            UNIT_ASSERT(bonusAccount);
            UNIT_ASSERT(bonusAccount->GetId() > 0);
            UNIT_ASSERT(session.Commit());
        }

        {
            auto accounts = accountsManager.GetUserAccounts(DefaultUserId, Now());
            UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 2);
            for (auto&& account : accounts) {
                if (account->GetType() == NDrive::NBilling::EAccount::Bonus) {
                    UNIT_ASSERT(!!account->GetAs<NDrive::NBilling::TBonusAccount>());
                }
                if (account->GetType() == NDrive::NBilling::EAccount::Trust) {
                    UNIT_ASSERT(!account->GetAs<NDrive::NBilling::TTrustAccount>());
                }
            }
        }
        {
            auto session = env.BuildSession();
            auto trustAccount = accountsManager.GetOrCreateTrustAccount(DefaultUserId, DefaultUserId, session);
            UNIT_ASSERT(trustAccount);
            UNIT_ASSERT(trustAccount->GetId() > 0);
            UNIT_ASSERT(session.Commit());
        }

        auto accounts = accountsManager.GetUserAccounts(DefaultUserId, Now());
        UNIT_ASSERT_VALUES_EQUAL(accounts.size(), 2);
        for (auto&& account : accounts) {
            if (account->GetType() == NDrive::NBilling::EAccount::Bonus) {
                UNIT_ASSERT(!!account->GetAs<NDrive::NBilling::TBonusAccount>());
            }
            if (account->GetType() == NDrive::NBilling::EAccount::Trust) {
                UNIT_ASSERT(!!account->GetAs<NDrive::NBilling::TTrustAccount>());
            }
        }
    }
}
